package org.nekosoft.pdffer.explorer;

import org.nekosoft.pdffer.PdfferProducerBean;
import org.nekosoft.pdffer.registry.PdfferRegistryBean;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Profile;
import org.springframework.core.io.Resource;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.*;

import static org.nekosoft.pdffer.PdfferCoreConfiguration.PROFILE_EXPLORER;
import static org.springframework.http.MediaType.APPLICATION_PDF_VALUE;
import static org.springframework.http.MediaType.TEXT_HTML_VALUE;

/**
 * The main controller for PDFfer Explorer. It displays the home page as well as the HTML form to generate a PDF. It also
 * exposes the endpoint that handles that form.
 */
@Controller
@RequestMapping("explorer")
@Profile(PROFILE_EXPLORER)
public class PdfferExplorerController extends AbstractPdfferExplorerController {

    // The template resource for the /home page
    @Value("classpath:org/nekosoft/pdffer/explorer/home.html")
    private Resource homeHtmlTemplate;

    // The template resource for the /download page
    @Value("classpath:org/nekosoft/pdffer/explorer/download.html")
    private Resource downloadHtmlTemplate;

    public PdfferExplorerController(PdfferProducerBean pdffer, PdfferRegistryBean registry) {
        super(pdffer, registry);
    }

    /**
     * Controller method mapped to GET /
     * @return the HTML content for the home page
     */
    @GetMapping
    @ResponseBody
    public String homePage() {
        return htmlTemplateAsString(homeHtmlTemplate, "");
    }


    /**
     * Controller method mapped to GET /download
     * @return the HTML content for the download form
     */
    @GetMapping(value = "download", produces = TEXT_HTML_VALUE)
    @ResponseBody
    public String downloadForm() {
        return htmlTemplateAsString(downloadHtmlTemplate, "");
    }

    /**
     * Controller method mapped to POST /download
     * @param templatePath the "pdfTemplate" HTML form field with the name of the template to apply
     * @param payload      the payload as a JSON string
     * @return the response entity with the bytes of the PDF file that was generated by the framework
     */
    @PostMapping(value = "download", produces = APPLICATION_PDF_VALUE)
    @ResponseBody
    public ResponseEntity<byte[]> processDownloadForm(@RequestParam("pdfTemplatePath") String templatePath, @RequestParam("pdfPayload") String payload) {
        try {
            byte[] body = pdffer.generatePdfDocumentByPathFromJsonString(templatePath, payload);
            return new ResponseEntity<>(body, HttpStatus.OK);
        } catch (IllegalArgumentException e) {
            return new ResponseEntity<>(HttpStatus.BAD_REQUEST);
        }
    }

}
