package org.nekosoft.pdffer.explorer;

import org.nekosoft.pdffer.PdfferProducerBean;
import org.nekosoft.pdffer.registry.PdfferRegistryBean;
import org.springframework.core.io.Resource;
import org.springframework.util.FileCopyUtils;
import org.springframework.web.util.HtmlUtils;

import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.util.List;

import static java.nio.charset.StandardCharsets.UTF_8;

/**
 * A base controller for PDFfer Explorer that holds a reference to a registry and a producer.
 * It also provides the {@link #htmlTemplateAsString(Resource, String)} method
 */
public class AbstractPdfferExplorerController {
    protected final PdfferProducerBean pdffer;
    protected final PdfferRegistryBean registry;

    public AbstractPdfferExplorerController(PdfferProducerBean pdffer, PdfferRegistryBean registry) {
        this.pdffer = pdffer;
        this.registry = registry;
    }

    /**
     * Takes a Resource object that represents an HTML template and returns a String with the HTML
     * content and any placeholder properly resolved to the corresponding values.
     * <p>Current placeholders are {{results}}, used in the email form to confirm that and email
     * was successfully sent or otherwise, and {{templates}}, that provides a list of template
     * paths to display on the page</p>
     * <p>In the future, a proper templating system should be used and this method would be deprecated</p>
     *
     * @param resource  the HTML resource to be rendered
     * @param result    an optional result to display on the pages that support it (e.g. the email form)
     * @return the HTML with all placeholder values resolved
     */
    protected String htmlTemplateAsString(Resource resource, String result) {
        try (Reader reader = new InputStreamReader(resource.getInputStream(), UTF_8)) {
            String htmlTemplate = FileCopyUtils.copyToString(reader);
            StringBuilder pdfTemplateHtml = new StringBuilder();
            List<String> templatePaths = registry.listTemplatePaths();
            for (String path : templatePaths) {
                pdfTemplateHtml.append("<option value=\"" + HtmlUtils.htmlEscape(path) + "\">");
                pdfTemplateHtml.append(HtmlUtils.htmlEscape(path));
                pdfTemplateHtml.append("</option>\n");
            }
            return htmlTemplate
                    .replace("{{result}}", result)
                    .replace("{{templates}}", pdfTemplateHtml);
        } catch (IOException e) {
            throw new IllegalArgumentException(e);
        }
    }

}
