package org.nekosoft.pdffer;

import org.nekosoft.pdffer.exception.InvalidPayloadClassException;
import org.nekosoft.pdffer.exception.InvalidPayloadException;
import org.nekosoft.pdffer.exception.InvalidTemplateClassException;
import org.nekosoft.pdffer.exception.PdfferProducerException;
import org.nekosoft.pdffer.registry.PdfferRegistryBean;
import org.nekosoft.pdffer.template.AbstractJacksonPdfTemplate;
import org.nekosoft.pdffer.template.PdfTemplate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.lang.reflect.Method;
import java.util.Map;

import static org.nekosoft.pdffer.template.PdfTemplate.splitTemplatePath;
import static org.nekosoft.pdffer.template.PdfTemplateComponent.ROOT_REGISTRY;

/**
 * The main Spring bean in PDFfer Core. This uses PDF templates - that must be provided by separate libraries - in
 * order to generate PDFs on the fly when needed. It needs an instance of {@link PdfferRegistryBean}, that will be
 * used when looking for templates.
 */
@Component
public class PdfferProducerBean {

    private static Method plainPayloadSetter;
    private static Method jsonPayloadSetter;
    private static Method mapPayloadSetter;

    private static final Logger logger = LoggerFactory.getLogger(PdfferProducerBean.class);

    static {
        try {
            plainPayloadSetter = PdfTemplate.class.getMethod("setPayload", Object.class);
            jsonPayloadSetter = AbstractJacksonPdfTemplate.class.getMethod("setPayloadJson", String.class);
            mapPayloadSetter = AbstractJacksonPdfTemplate.class.getMethod("setPayloadMap", Map.class);
        } catch (NoSuchMethodException e) {
            logger.error("FATAL CONDITION IN PRODUCER CREATION", e);
            throw new PdfferProducerException(e);
        }
    }

    private final PdfferRegistryBean registry;

    /**
     * Creates a new PdfferProducerBean with the given registry. This is not usually invoked manually. Spring will
     * create an instance of this class and inject the registry.
     *
     * @param registry the registry
     */
    public PdfferProducerBean(PdfferRegistryBean registry) {
        this.registry = registry;
    }

    /**
     * Generates a new PDF with the template at the given template path and with the given
     * payload. The path is split into group and name using {@code PdfTemplate::splitTemplatePath(String)}.
     *
     * @param templatePath the path (group + separator + name ) of the template to be used
     * @param data         the payload needed by the template for PDF generation
     * @return the bytes of the generated PDF document
     */
    public byte[] generatePdfDocumentByPath(String templatePath, Object data) {
        String[] split = splitTemplatePath(templatePath);
        return generatePdfDocument(split[0], split[1], data);
    }

    /**
     * Generates a new PDF with the given template (that must be present in the root registry) and the given
     * payload.
     *
     * @param templateName the template name (from the root registry)
     * @param data         the payload needed by the template for PDF generation
     * @return the bytes of the generated PDF document
     */
    public byte[] generatePdfDocument(String templateName, Object data) {
        return generatePdfDocument(ROOT_REGISTRY, templateName, data);
    }

    /**
     * Generates a new PDF with a template by the given name and group and with the given
     * payload.
     *
     * @param <T>          the type parameter
     * @param group        the group where the template resides
     * @param templateName the template name
     * @param data         the payload needed by the template for PDF generation
     * @return the bytes of the generated PDF document
     */
    public <T> byte[] generatePdfDocument(String group, String templateName, T data) {
        return innerGeneratePdfDocument(group, templateName, PdfTemplate.class, plainPayloadSetter, data);
    }

    /**
     * Generates a new PDF with a template at the given path and with the given
     * payload.
     *
     * @param templatePath the path (group + separator + name ) of the template to be used
     * @param data         a string containing a JSON representation of the payload for the PDF
     * @return the bytes of the generated PDF document
     */
    public byte[] generatePdfDocumentByPathFromJsonString(String templatePath, String data) {
        String[] split = splitTemplatePath(templatePath);
        return generatePdfDocumentFromJsonString(split[0], split[1], data);
    }

    /**
     * Generates a new PDF with the given template (that must be present in the root registry) and the given
     * payload.
     *
     * @param templateName the template name (from the root registry)
     * @param data         a string containing a JSON representation of the payload for the PDF
     * @return the bytes of the generated PDF document
     */
    public byte[] generatePdfDocumentFromJsonString(String templateName, String data) {
        return generatePdfDocumentFromJsonString(ROOT_REGISTRY, templateName, data);
    }

    /**
     * Generates a new PDF with a template by the given name and group and with the given
     * payload.
     *
     * @param group        the group where the template resides
     * @param templateName the template name
     * @param data         a string containing a JSON representation of the payload for the PDF
     * @return the bytes of the generated PDF document
     */
    public byte[] generatePdfDocumentFromJsonString(String group, String templateName, String data) {
        return innerGeneratePdfDocument(group, templateName, AbstractJacksonPdfTemplate.class, jsonPayloadSetter, data);
    }

    /**
     * Generates a new PDF with the template at the given template path and with the given
     * payload. The path is split into group and name using {@code PdfTemplate::splitTemplatePath(String)}.
     *
     * @param templatePath the path (group + separator + name ) of the template to be used
     * @param data         a {@code java.util.Map} representation of the payload for the PDF
     * @return the bytes of the generated PDF document
     */
    public byte[] generatePdfDocumentByPathFromJsonMap(String templatePath, Map<String,Object> data) {
        String[] split = splitTemplatePath(templatePath);
        return generatePdfDocumentFromJsonMap(split[0], split[1], data);
    }

    /**
     * Generates a new PDF with the given template (that must be present in the root registry) and the given
     * payload.
     *
     * @param templateName the template name (from the root registry)
     * @param data         a {@code java.util.Map} representation of the payload for the PDF
     * @return the bytes of the generated PDF document
     */
    public byte[] generatePdfDocumentFromJsonMap(String templateName, Map<String,Object> data) {
        return generatePdfDocumentFromJsonMap(ROOT_REGISTRY, templateName, data);
    }

    /**
     * Generates a new PDF with a template by the given name and group and with the given
     * payload.
     *
     * @param group        the group where the template resides
     * @param templateName the template name
     * @param data         a {@code java.util.Map} representation of the payload for the PDF
     * @return the bytes of the generated PDF document
     */
    public byte[] generatePdfDocumentFromJsonMap(String group, String templateName, Map<String,Object> data) {
        return innerGeneratePdfDocument(group, templateName, AbstractJacksonPdfTemplate.class, mapPayloadSetter, data);
    }

    private PdfTemplate<?> findTemplate(String group, String template) {
        return registry.findTemplate(group, template);
    }

    private byte[] innerGeneratePdfDocument(String group, String templateName, Class<? extends PdfTemplate> expectedTemplateClass, Method payloadSetter, Object payload) {
        // Get the template instance from the registry
        PdfTemplate<?> template = findTemplate(group, templateName);
        // For JSON we expect subclasses of AbstractJacksonPdfTemplate, for anything else a subclass of PdfTemplate
        Class<? extends PdfTemplate> actualTemplateClass = template.getClass();
        if (!expectedTemplateClass.isAssignableFrom(actualTemplateClass)) {
            logger.error("Class {} is not a valid template for {}", actualTemplateClass, expectedTemplateClass);
            throw new InvalidTemplateClassException(actualTemplateClass, expectedTemplateClass);
        }
        // Is it's a JSON payload, then we check the payload
        Class<?> actualPayloadClass = payload.getClass();
        if (!AbstractJacksonPdfTemplate.class.isAssignableFrom(expectedTemplateClass)) {
            Class<?> expectedPayloadClass = template.getPayloadClass();
            if (!expectedPayloadClass.isAssignableFrom(actualPayloadClass)) {
                logger.error("Payload class {} is not a valid payload for {}", actualPayloadClass, expectedPayloadClass);
                throw new InvalidPayloadClassException(actualPayloadClass, expectedPayloadClass);
            }
        }
        try {
            payloadSetter.invoke(template, payload);
        } catch (ReflectiveOperationException e) {
            logger.error("Was not able to set payload {} for template {}", actualPayloadClass, template);
            throw new PdfferProducerException(e);
        }
        if (!template.validate()) {
            throw new InvalidPayloadException(payload);
        }
        template.generate();
        return template.getPdfContent();
    }

}
