package org.nekosoft.pdffer.props;

import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.boot.context.properties.ConstructorBinding;
import org.springframework.validation.annotation.Validated;

import javax.validation.Valid;
import javax.validation.constraints.NotNull;

/**
 * A Property POJO that reflects the configuration of the mailer bean.
 */
@ConfigurationProperties(prefix = "pdffer.mailer")
@ConstructorBinding
@Validated
@ConditionalOnProperty(name = "pdffer.mailer.enabled", havingValue = "true", matchIfMissing = false)
public class PdfferMailerProps {
    // field declarations
    private final boolean enabled;
    private final EmailAddressInfo sendFrom;
    private final EmailAddressInfo replyTo;
    private final SmtpServerInfo smtp;

    /**
     * Instantiates a new Pdffer mailer props.
     *
     * @param sendFrom the default email address that will be used as the sender for the emails sent by PDFfer
     * @param replyTo  the default email address that will be used as the reply-to header for the emails sent by PDFfer
     * @param smtp     the settings of the SMTP server that will be used to send email messages from PDFfer
     */
    public PdfferMailerProps(boolean enabled, EmailAddressInfo sendFrom, EmailAddressInfo replyTo, SmtpServerInfo smtp) {
        // set field values
        this.enabled = enabled;
        this.sendFrom = sendFrom;
        this.replyTo = replyTo;
        this.smtp = smtp;
    }

    // getters only

    /**
     * Indicates whether the mailer beans should be present.
     *
     * @return {@code true} if the mailer beans should be present
     */
    public boolean isEnabled() {
        return enabled;
    }

    /**
     * Gets the default email address that will be used as the sender for the emails sent by PDFfer.
     *
     * @return the default email address that will be used as the sender
     */
    @Valid
    public EmailAddressInfo getSendFrom() {
        return sendFrom;
    }

    /**
     * Gets the default email address that will be used as the reply-to header for the emails sent by PDFfer.
     *
     * @return the default email address that will be used as the reply-to header
     */
    @Valid
    public EmailAddressInfo getReplyTo() {
        return replyTo;
    }

    /**
     * Gets the settings of the SMTP server that will be used to send email messages from PDFfer.
     *
     * @return the settings of the SMTP server
     */
    @NotNull(message = "You must provide SMTP server information in order to use the PDFfer Mailer")
    @Valid
    public SmtpServerInfo getSmtp() {
        return smtp;
    }

}
