package org.nekosoft.pdffer.mail;

import org.nekosoft.pdffer.props.PdfferMailerProps;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.core.io.ByteArrayResource;
import org.springframework.http.MediaType;
import org.springframework.mail.javamail.JavaMailSender;
import org.springframework.mail.javamail.MimeMessageHelper;
import org.springframework.stereotype.Component;

import javax.mail.MessagingException;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeMessage;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * The PDFfer bean that deals with email. It has methods that will take PDF content as an array of bytes and
 * will send it over to the given email address or addresses.
 */
@Component
@ConditionalOnClass(name = "org.springframework.mail.javamail.JavaMailSender")
@ConditionalOnProperty(name = "pdffer.skip.mailer.beans", havingValue = "false", matchIfMissing = true)
public class PdfferMailerBean {

    private final JavaMailSender mailSender;

    private PdfferMailerProps props;

    /**
     * Creates a new PdfferMailerBean instace with the given {@code JavaMailSender} and {@link PdfferMailerProps}.
     * This is not usually invoked manually. Spring will create an instance of this class and inject the
     * {@code JavaMailSender} and {@link PdfferMailerProps} beans.
     *
     * @param mailSender a {@code JavaMailSender} instance
     * @param props      a {@link PdfferMailerProps} instance
     */
    public PdfferMailerBean(JavaMailSender mailSender, PdfferMailerProps props) {
        this.mailSender = mailSender;
        this.props = props;
    }

    /**
     * Send an email message with a PDF attachment.
     *
     * @param to         the email addresses to send the email to as a {@code java.util.List} of
     *                   {@code javax.mail.internet.InternetAddress}
     * @param subject    the subject of the email
     * @param text       the text of the main message of the email
     * @param attachment the attachment PDF as an array of bytes
     * @param filename   the filename of the attachment
     * @param sendFrom   the send-from email address (can be null)
     * @param replyTo    the reply-to email address (can be null)
     * @throws IllegalArgumentException the {@code IllegalArgumentException} raised if any issues were encountered
     */
    public void sendMessageWithPdfAttachment(List<InternetAddress> to, String subject, String text, byte[] attachment, String filename, String sendFrom, String replyTo) throws IllegalArgumentException {
        MimeMessage message = mailSender.createMimeMessage();

        try {
            MimeMessageHelper helper = new MimeMessageHelper(message, true);
            if (sendFrom != null) {
                helper.setFrom(sendFrom);
            } else {
                helper.setFrom(String.format("\"%s\" <%s>", props.getSendFrom().getName(), props.getSendFrom().getAddress()));
            }
            if (replyTo != null) {
                helper.setReplyTo(replyTo);
            } else {
                helper.setReplyTo(String.format("\"%s\" <%s>", props.getReplyTo().getName(), props.getReplyTo().getAddress()));
            }
            helper.setTo(to.toArray(new InternetAddress[] {}));
            helper.setSubject(subject);
            helper.setText(text);

            helper.addAttachment(filename, new ByteArrayResource(attachment), MediaType.APPLICATION_PDF_VALUE);

            mailSender.send(message);
        } catch (MessagingException e) {
            throw new IllegalArgumentException(e);
        }
    }

    /**
     * Send an email message with a PDF attachment.
     *
     * @param toString   the email addresses to send the email to as a comma-separated string
     * @param subject    the subject of the email
     * @param text       the text of the main message of the email
     * @param attachment the attachment PDF as an array of bytes
     * @param filename   the filename of the attachment
     * @param sendFrom   the send-from email address (can be null)
     * @param replyTo    the reply-to email address (can be null)
     * @throws IllegalArgumentException the {@code IllegalArgumentException} raised if any issues were encountered
     */
    public void sendMessageWithPdfAttachment(String toString, String subject, String text, byte[] attachment, String filename, String sendFrom, String replyTo) throws IllegalArgumentException {
        sendMessageWithPdfAttachmentToList(Arrays.asList(toString.split(",")), subject, text, attachment, filename, sendFrom, replyTo);
    }

    /**
     * Send an email message with a PDF attachment.
     *
     * @param toList     the email addresses to send the email to as a list of strings
     * @param subject    the subject of the email
     * @param text       the text of the main message of the email
     * @param attachment the attachment PDF as an array of bytes
     * @param filename   the filename of the attachment
     * @param sendFrom   the send-from email address (can be null)
     * @param replyTo    the reply-to email address (can be null)
     * @throws IllegalArgumentException the {@code IllegalArgumentException} raised if any issues were encountered
     */
    public void sendMessageWithPdfAttachmentToList(List<String> toList, String subject, String text, byte[] attachment, String filename, String sendFrom, String replyTo) throws IllegalArgumentException {
        try {
            List<InternetAddress> recipients = new ArrayList<>();
            for (String email : toList) {
                recipients.add(new InternetAddress(email));
            }
            sendMessageWithPdfAttachment(recipients, subject, text, attachment, filename, sendFrom, replyTo);
        } catch (MessagingException e) {
            throw new IllegalArgumentException(e);
        }
    }

}
