package org.nekosoft.pdffer.explorer;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.nekosoft.pdffer.PdfferProducerBean;
import org.nekosoft.pdffer.mail.PdfferMailerBean;
import org.nekosoft.pdffer.registry.PdfferRegistryBean;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.context.annotation.Profile;
import org.springframework.core.io.Resource;
import org.springframework.stereotype.Controller;
import org.springframework.util.FileCopyUtils;
import org.springframework.web.bind.annotation.*;

import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.util.List;
import java.util.Map;

import static java.nio.charset.StandardCharsets.UTF_8;
import static org.springframework.http.MediaType.TEXT_HTML_VALUE;

/**
 * The type Pdffer email explorer controller.
 */
@Controller
@ConditionalOnClass(name = "javax.mail.internet.InternetAddress")
@ConditionalOnBean(PdfferMailerBean.class)
@RequestMapping("explorer")
@Profile("pdffer-explorer")
public class PdfferEmailExplorerController {

    // The template resource for the /email page
    @Value("classpath:org/nekosoft/pdffer/explorer/email.html")
    private Resource emailHtmlTemplate;

    private ObjectMapper mapper;
    private PdfferProducerBean pdffer;
    private PdfferRegistryBean registry;
    private PdfferMailerBean mailer;

    /**
     * Instantiates a new Pdffer email explorer controller.
     *
     * @param mapper   the Jackson mapper in the context
     * @param pdffer   the pdffer
     * @param registry the registry
     * @param mailer   the mailer
     */
    public PdfferEmailExplorerController(ObjectMapper mapper, PdfferProducerBean pdffer, PdfferRegistryBean registry, PdfferMailerBean mailer) {
        this.mapper = mapper;
        this.pdffer = pdffer;
        this.registry = registry;
        this.mailer = mailer;
    }

    /*
     * Takes a Resource object that represents an HTML template and returns a String with the HTML
     * content and any placeholder properly resolved to the corresponding values.
     * No placeholders are available in this module
     */
    private String htmlTemplateAsString(Resource resource, String result) {
        try (Reader reader = new InputStreamReader(resource.getInputStream(), UTF_8)) {
            String htmlTemplate = FileCopyUtils.copyToString(reader);
            StringBuilder pdfTemplateHtml = new StringBuilder();
            List<String> templatePaths = registry.listTemplates();
            for (String path : templatePaths) {
                pdfTemplateHtml.append("<option>");
                pdfTemplateHtml.append(path);
                pdfTemplateHtml.append("</option>\n");
            }
            return htmlTemplate
                    .replace("{{result}}", result)
                    .replace("{{templates}}", pdfTemplateHtml);
        } catch (IOException e) {
            throw new IllegalArgumentException(e);
        }
    }

    /**
     * Controller method mapped to GET /download by default
     *
     * @return the HTML content for the download form
     */
    @GetMapping(value = "email", produces = TEXT_HTML_VALUE)
    @ResponseBody
    public String emailForm() {
        return htmlTemplateAsString(emailHtmlTemplate, "");
     }

    /**
     * Controller method mapped to POST /download by default
     *
     * @param template    the "pdfTemplate" HTML form field with the name of the template to apply                 (only "default" available in this module)
     * @param payload     the JSON text of the data for the PDF template
     * @param sendEmails  the send emails
     * @param mailSubject the mail subject
     * @param mailMessage the mail message
     * @return the bytes of the PDF file that was generated by the framework
     * @throws JsonProcessingException if the payload is not valid JSON text
     */
    @PostMapping(value = "email", produces = TEXT_HTML_VALUE)
    @ResponseBody
    public String processEmailForm(@RequestParam("pdfTemplate") String template, @RequestParam("pdfPayload") String payload, @RequestParam("sendEmails") String sendEmails, @RequestParam("mailSubject") String mailSubject, @RequestParam("mailMessage") String mailMessage) throws JsonProcessingException {
        Map<String,Object> pdfData = mapper.readValue(payload, ((Class<Map<String, Object>>)(Class<?>)Map.class));
        byte[] pdfBytes = pdffer.generatePdfDocument(template, pdfData);
        mailer.sendMessageWithPdfAttachment(sendEmails, mailSubject, mailMessage, pdfBytes, "pdffer.pdf", null, null);
        return htmlTemplateAsString(emailHtmlTemplate, "<p class=\"text-success\">Email with PDF was successfully sent to " + sendEmails + "</p>");
     }

}
