package org.nekosoft.pdffer;

import java.util.Map;

import org.nekosoft.pdffer.registry.PdfferRegistryBean;
import org.nekosoft.pdffer.template.PdfTemplate;
import org.springframework.stereotype.Component;

/**
 * The main Spring bean in PDFfer Core. This uses PDF templates - that must be provided by separate libraries - in
 * order to generate PDFs on the fly when needed. It needs an instance of {@link PdfferRegistryBean}, that will be
 * used when looking for templates.
 *
 */
@Component
public class PdfferProducerBean {

    private final PdfferRegistryBean registry;

    /**
     * Creates a new PdfferProducerBean with the given registry. This is not usually invoked manually. Spring will
     * create an instance of this class and inject the registry.
     *
     * @param registry the registry
     */
    public PdfferProducerBean(PdfferRegistryBean registry) {
        this.registry = registry;
    }

    /**
     * Generates a new PDF with the given template (that must be present in the registry) and the given
     * payload.
     *
     * @param templateName the template name
     * @param data         the payload needed by the template for PDF generation
     * @return the byte [ ]
     */
    public byte[] generatePdfDocument(String templateName, Map<String, Object> data) {
        PdfTemplate template = findTemplate(templateName);
        template.setPdfData(data);
        if (!template.validate()) {
            throw new IllegalArgumentException("PDF Template payload is not valid");
        }
        template.generate();
        return template.getPdfContent();
    }

    /**
     * Looks for a template instance by the given name.
     *
     * @param templateName the template name
     * @return the {@link PdfTemplate} instance corresponding to the given name
     */
    PdfTemplate findTemplate(String templateName) {
        return registry.findTemplate(templateName);
    }

}
