/**********************************************************************
 * Class Rfc3188Validator
 *  
 * Copyright (c) 2011-2014, German National Library / Deutsche Nationalbibliothek
 * Adickesallee 1, D-60322 Frankfurt am Main, Federal Republic of Germany 
 *
 * This program is free software.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 * Kadir Karaca Kocer -- German National Library
 * 
 **********************************************************************/

/* ********************************************************************
 * CHANGELOG:
 * 2012-03-14 Commented and ported to Apache Maven
 * Created on 2011 December by Kadir Karaca Kocer
 **********************************************************************/

package org.nbnResolving.common;

import java.util.Locale;


/**
 * Class for validation of an URN according to RFC.
 * {@link "https://tools.ietf.org/html/rfc3188"}
 * 
 * @author Kadir Karaca Kocer
 */
public class Rfc3188Validator {
	
	/** All two digit country codes according to ISO 3166
	 * {@link "http://www.iso.org/iso/country_codes/iso_3166_code_lists.htm"} */
	public static String[] iso3106Codes = Locale.getISOCountries();
	
	/** All in URN:NBN allowed characters  */
	public static final char[] VALID_CHARS  = {
        '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J',
        'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '-', ':', '_', '.'};
	
	/** URN is VALID */
	public static final int URN_IS_VALID        =   0;
	/** URN is NULL */
	public static final int URN_EQUALS_NULL     = 100;
	/** URN is empty */
	public static final int URN_IS_EMPTY        = 101;
	/** NID too short. See RFC 3288*/
	public static final int NID_TOO_SHORT       = 102;
	/** NID must have at least 4 parts. See RFC 3188*/
	public static final int PART_COUNT_TOO_LOW  = 103;
	/** NID must start with URN:. See RFC 3188*/
	public static final int MUST_START_WITH_URN = 104;
	/** Only NBNs are allowed */
	public static final int NOT_A_NBN           = 105;
	/** Only two digit country codes defined in ISO 3106 are allowed */
	public static final int NOT_TWO_DIGIT_CODE  = 106;
	/** Two digit country code is not in ISO 3106 list */
	public static final int BAD_COUNTRY_CODE    = 107;
	/** Sub-Namespace Code can not be empty */
	public static final int EMPTY_SUBNAMESPACE  = 108;
	/** Assigned NBN string can not be empty */
	public static final int EMPTY_NBN_STRING    = 109;
	/** Not allowed Character in Namespace */
	public static final int INVALID_CHAR_NID    = 110;
	/** Not allowed Character in NBN String */
	public static final int INVALID_CHAR_NBN    = 111;
	
	/**
	 * Method to check if the given URN satisfies the syntax defined in RFC 2141.
	 * Please note that this method checks only NBN namespace defined in RFC 3188:
	 * {@link "https://tools.ietf.org/html/rfc3188"}
	 * All other registered URN namespaces are out of scope of this java class
	 * and must be implemented separetely.
	 * 
	 * @param urn A National Bibliography Number as Uniform Resource Name according to RFC 3188.
	 * @return 0 for a valid URN. An integer in case of a violation against RFC 3188. See the predefined constants.
	 */
	public static int validateUrn(String urn) {
		if (urn == null)   return URN_EQUALS_NULL; //URN can not be NULL
		if (urn.isEmpty()) return URN_IS_EMPTY;
		//URN:NBN:<ISO 3166 country code>:<sub-namespace code>-<assigned NBN string>
		String nbn = urn.trim();
		int seperator = nbn.indexOf("-");
		if (seperator < 10) return NID_TOO_SHORT;
		
		String nid = nbn.substring(0, seperator).toUpperCase();
		//System.out.println("NID: " + nid);
		String[] parts = nid.split(":");
		//System.out.println("Parts count: " + parts.length);
		if (parts.length < 4) return PART_COUNT_TOO_LOW;
		if ((parts[0] == null) || (!parts[0].equals("URN"))) return MUST_START_WITH_URN;
		if ((parts[1] == null) || (!parts[1].equals("NBN"))) return NOT_A_NBN;
		if ((parts[2] == null) || (parts[2].length() != 2))  return NOT_TWO_DIGIT_CODE;
		if ((parts[3] == null) || (parts[3].isEmpty()))      return EMPTY_SUBNAMESPACE;

		//check if Country Code valid!
		boolean valid = false;
		for (int i=0; i<iso3106Codes.length; i++) {
			//System.out.println("Comparing " + parts[2] + " with " + iso3106Codes[i]);
			if (parts[2].equals(iso3106Codes[i])) {
				valid = true;
				break;
			}
		}
		if (!valid) return BAD_COUNTRY_CODE;
		
		String nss = nbn.substring(seperator+1);
		//System.out.println("NSS: " + nss);
		if ((nss == null) || nss.isEmpty()) return EMPTY_NBN_STRING;
		
		for (int i = 3; i < parts.length; i++) {
			if (!charsValid(parts[i])) return INVALID_CHAR_NID;
		}
		
		if (!charsValid(nss.toUpperCase())) return INVALID_CHAR_NBN;
		
		//no violations so far. return ok.
		return URN_IS_VALID;
	}

	/**
	 * Check if only allowed chars are used.
	 * 
	 * @param str String to test.
	 * @return TRUE if all the characters are in the given string in VALID_CHARS list. FALSE otherwise.
	 */
	public static boolean charsValid(String str) {
		for (int i=0; i < str.length(); i++) {
			boolean error = true;
			char c = str.charAt(i);
			for (int j=0; j < VALID_CHARS.length; j++){
				if (c == VALID_CHARS[j]) { //we found the searched character
					error = false;         //good! no error.
				}
			}
			if (error) return false;
		}
		return true;
	}

}
