/**********************************************************************
 * Class DnbUrnValidator
 *  
 * Copyright (c) 2011-2013, German National Library/Deutsche Nationalbibliothek
 * Adickesallee 1, D-60322 Frankfurt am Main, Federal Republic of Germany 
 *
 * This program is free software.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 * Kadir Karaca Kocer -- German National Library
 * 
 **********************************************************************/

/* ********************************************************************
 * CHANGELOG:
 * 2013-05-22 Checking NULL value to prevent NullPointerException
 * 2012-06-12 Checking the length of the URN and its parts
 * 2012-03-14 Commented and ported to Apache Maven
 * Created on 2011 December by Kadir Karaca Kocer
 **********************************************************************/

package org.nbnResolving.common;

import org.nbnResolving.common.Rfc3188Validator;
import org.nbnResolving.common.URNUtils;

/**
 * Class extending the validation of an URN from RFC level
 * to the German National Library URN policy level (more strict).
 * {@link "http://urn.dnb.de"}
 * 
 * @author Kadir Karaca Kocer
 */
public class DnbUrnValidator extends Rfc3188Validator {
    /** Min URN length */
    public static final int MIN_URN_LENGTH         = 16;
    /** Max URN length */
    public static final int MAX_URN_LENGTH         = 254;
    /** Max NAMESPACE length */
    public static final int MAX_NAMESPACE_LENGTH   = 52;
    /** Max UNIQUE PART length */
    public static final int MAX_UNIQUE_PART_LENGTH = 201;
    
    /** URNs must be longer than 15 characters */
    public static final int URN_TOO_SHORT          = 201;    
    /** URNs must be shorter than 254 characters */
    public static final int URN_TOO_LONG           = 202;
    /** Parts of an URN must be separated by a - character */
    public static final int NO_SEPERATOR           = 203;
    /** Namespace must be shorter than 52 characters */
    public static final int NAMESPACE_TOO_LONG     = 204;
    /** Unique ID part can be max 200 characters + checksum digit*/
    public static final int UNIQUE_PART_TOO_LONG   = 205;
    
	/** Checksum must be a number*/
	public static final int CHECKSUM_NOT_A_NUMBER  = 210;
	/** Wrong checksum*/
	public static final int INVALID_CHECKSUM       = 211;
	/** Namespace not registered */
	public static final int UNKNOWN_NAMESPACE      = 212;
	
	/**
	 * Validates if the given URN is compliant German National Library URN policy.
	 * 
	 * @param urn A National Bibliography Number as Uniform Resource Name according to RFC 3188.
	 * @return 0 for a valid URN. An integer in case of a violation against RFC 3188 or
	 * German National Library URN:NBN Policy. See the predefined constants.
	 */
	public static int validateUrn(String urn) {
	    if (urn == null) return Rfc3188Validator.URN_EQUALS_NULL;
	    int length = urn.length();
        if (length < MIN_URN_LENGTH) return URN_TOO_SHORT;
	    if (length > MAX_URN_LENGTH) return URN_TOO_LONG;
	    int pos = urn.indexOf('-');
	    if (pos < 1)  return NO_SEPERATOR;
	    if (pos > MAX_NAMESPACE_LENGTH) return NAMESPACE_TOO_LONG;
	    if (length - pos > MAX_UNIQUE_PART_LENGTH) return UNIQUE_PART_TOO_LONG;
	    
	    //length and parts ok. check if it is valid according to RFC
		int ret = Rfc3188Validator.validateUrn(urn);
		if (ret == 0) {
			//URN is valid according to RFC 3188 syntax. Check its checksum
			String cs = urn.substring(urn.length()-1);
			//System.out.println("Syntax Ok. Checksum string: " + cs);
			int checksumExisting = -1;
			try {
				checksumExisting = Integer.valueOf(cs).intValue();
				//System.out.println("Syntax Ok. Checking the checksum: " + checksumExisting);
				if ((checksumExisting < 0) || (checksumExisting > 9)) return CHECKSUM_NOT_A_NUMBER;	
			} catch (Exception e) {
				return CHECKSUM_NOT_A_NUMBER;
			}
			//Existing checksum is between 0 .. and .. 9 calculate it again and compare
			String nbn = urn.substring(0, urn.length()-1);
			//System.out.println("Calculating checksum for: " + nbn);
			int checksumExpected = URNUtils.getChecksum(nbn);
			//System.out.println("Existing checksum : " + checksumExisting + " Expected checksum: " + checksumExpected);
			if (checksumExisting != checksumExpected) return INVALID_CHECKSUM;
		}
		return ret;
	}
}
