/*
 * Copyright (C) 2015-2022 52°North Spatial Information Research GmbH
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.n52.iceland.i18n;

import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import java.util.Optional;

import javax.inject.Inject;

import org.n52.iceland.i18n.metadata.AbstractI18NMetadata;
import org.n52.janmayen.Producer;
import org.n52.janmayen.component.AbstractComponentRepository;
import org.n52.janmayen.lifecycle.Constructable;

import com.google.common.collect.Maps;

/**
 * I18N DAO repository
 *
 * @author <a href="mailto:c.hollmann@52north.org">Carsten Hollmann</a>
 * @since 1.0.0
 */
public class I18NDAORepository extends AbstractComponentRepository<I18NDAOKey, I18NDAO<?>, I18NDAOFactory>
        implements Constructable {
    private final Map<I18NDAOKey, Producer<I18NDAO<?>>> daos = Maps.newHashMap();

    @Inject
    private Optional<Collection<I18NDAO<?>>> components = Optional.of(Collections.emptyList());
    @Inject
    private Optional<Collection<I18NDAOFactory>> componentFactories = Optional.of(Collections.emptyList());

    @Override
    public void init() {
        this.daos.clear();
        this.daos.putAll(getUniqueProviders(this.components, this.componentFactories));
    }

    /**
     * Get the available DAO
     *
     * @param <T> the meta data type
     * @param c   the meta data class
     *
     * @return the loaded DAO
     */
    @SuppressWarnings("unchecked")
    public <T extends AbstractI18NMetadata> I18NDAO<T> getDAO(Class<T> c) {
        Producer<I18NDAO<?>> producer = daos.get(new I18NDAOKey(c));
        // TODO check for subtypes
        I18NDAO<?> dao = producer == null ? null : producer.get();
        return (I18NDAO<T>) dao;
    }

    public boolean isSupported() {
        if (!daos.isEmpty()) {
            for (Producer<I18NDAO<?>> dao : daos.values()) {
                if (dao.get() != null && dao.get().isSupported()) {
                    return true;
                }
            }
        }
        return false;
    }

}
