/*
 * Decompiled with CFR 0.152.
 */
package org.molgenis.data.postgresql;

import com.google.common.collect.Lists;
import java.text.MessageFormat;
import java.time.Instant;
import java.time.LocalDate;
import java.time.ZoneOffset;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;
import org.apache.commons.lang3.StringUtils;
import org.molgenis.data.Entity;
import org.molgenis.data.MolgenisDataException;
import org.molgenis.data.Query;
import org.molgenis.data.QueryRule;
import org.molgenis.data.Sort;
import org.molgenis.data.UnknownAttributeException;
import org.molgenis.data.meta.AttributeType;
import org.molgenis.data.meta.IllegalAttributeTypeException;
import org.molgenis.data.meta.model.Attribute;
import org.molgenis.data.meta.model.EntityType;
import org.molgenis.data.postgresql.PostgreSqlNameGenerator;
import org.molgenis.data.postgresql.PostgreSqlQueryUtils;
import org.molgenis.data.postgresql.PostgreSqlUtils;
import org.molgenis.data.support.QueryImpl;
import org.molgenis.data.util.AttributeUtils;
import org.molgenis.data.util.EntityTypeUtils;
import org.molgenis.util.UnexpectedEnumException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class PostgreSqlQueryGenerator {
    private static final Logger LOG = LoggerFactory.getLogger(PostgreSqlQueryGenerator.class);
    private static final String UNSPECIFIED_ATTRIBUTE_MSG = "Can't use %s without specifying an attribute";
    static final String ERR_CODE_READONLY_VIOLATION = "23506";

    private PostgreSqlQueryGenerator() {
    }

    private static String getSqlConstraintPrimaryKey(EntityType entityType, Attribute attr) {
        return "CONSTRAINT " + PostgreSqlNameGenerator.getPrimaryKeyName(entityType, attr) + " PRIMARY KEY (" + PostgreSqlNameGenerator.getColumnName(attr) + ')';
    }

    private static String getSqlForeignKey(EntityType entityType, Attribute attr) {
        StringBuilder strBuilder = new StringBuilder("CONSTRAINT ").append(PostgreSqlNameGenerator.getForeignKeyName(entityType, attr)).append(" FOREIGN KEY (").append(PostgreSqlNameGenerator.getColumnName(attr)).append(") REFERENCES ").append(PostgreSqlNameGenerator.getTableName(attr.getRefEntity())).append('(').append(PostgreSqlNameGenerator.getColumnName(attr.getRefEntity().getIdAttribute())).append(')');
        if (attr.getRefEntity().getId().equals(entityType.getId())) {
            strBuilder.append(" DEFERRABLE INITIALLY DEFERRED");
        }
        return strBuilder.toString();
    }

    private static String getSqlUniqueKey(EntityType entityType, Attribute attr) {
        return "CONSTRAINT " + PostgreSqlNameGenerator.getUniqueKeyName(entityType, attr) + " UNIQUE (" + PostgreSqlNameGenerator.getColumnName(attr) + ')';
    }

    private static String getSqlCheckConstraint(EntityType entityType, Attribute attr) {
        if (attr.getDataType() != AttributeType.ENUM) {
            throw new MolgenisDataException(String.format("Check constraint only allowed for attribute type [%s]", AttributeType.ENUM.toString()));
        }
        return "CONSTRAINT " + PostgreSqlNameGenerator.getCheckConstraintName(entityType, attr) + " CHECK (" + PostgreSqlNameGenerator.getColumnName(attr) + " IN (" + attr.getEnumOptions().stream().map(enumOption -> '\'' + enumOption + '\'').collect(Collectors.joining(",")) + "))";
    }

    static String getSqlCreateForeignKey(EntityType entityType, Attribute attr) {
        return "ALTER TABLE " + PostgreSqlNameGenerator.getTableName(entityType) + " ADD " + PostgreSqlQueryGenerator.getSqlForeignKey(entityType, attr);
    }

    static String getSqlDropForeignKey(EntityType entityType, Attribute attr) {
        return "ALTER TABLE " + PostgreSqlNameGenerator.getTableName(entityType) + " DROP CONSTRAINT " + PostgreSqlNameGenerator.getForeignKeyName(entityType, attr);
    }

    static String getSqlCreateUniqueKey(EntityType entityType, Attribute attr) {
        return "ALTER TABLE " + PostgreSqlNameGenerator.getTableName(entityType) + " ADD " + PostgreSqlQueryGenerator.getSqlUniqueKey(entityType, attr);
    }

    static String getSqlDropUniqueKey(EntityType entityType, Attribute attr) {
        return "ALTER TABLE " + PostgreSqlNameGenerator.getTableName(entityType) + " DROP CONSTRAINT " + PostgreSqlNameGenerator.getUniqueKeyName(entityType, attr);
    }

    static String getSqlCreateCheckConstraint(EntityType entityType, Attribute attr) {
        return "ALTER TABLE " + PostgreSqlNameGenerator.getTableName(entityType) + " ADD " + PostgreSqlQueryGenerator.getSqlCheckConstraint(entityType, attr);
    }

    static String getSqlDropCheckConstraint(EntityType entityType, Attribute attr) {
        if (attr.getDataType() != AttributeType.ENUM) {
            throw new MolgenisDataException(String.format("Check constraint only allowed for attribute type [%s]", AttributeType.ENUM.toString()));
        }
        return "ALTER TABLE " + PostgreSqlNameGenerator.getTableName(entityType) + " DROP CONSTRAINT " + PostgreSqlNameGenerator.getCheckConstraintName(entityType, attr);
    }

    static String getSqlSetNotNull(EntityType entityType, Attribute attr) {
        return "ALTER TABLE " + PostgreSqlNameGenerator.getTableName(entityType) + " ALTER COLUMN " + PostgreSqlNameGenerator.getColumnName(attr) + " SET NOT NULL";
    }

    static String getSqlDropNotNull(EntityType entityType, Attribute attr) {
        return "ALTER TABLE " + PostgreSqlNameGenerator.getTableName(entityType) + " ALTER COLUMN " + PostgreSqlNameGenerator.getColumnName(attr) + " DROP NOT NULL";
    }

    static String getSqlSetDataType(EntityType entityType, Attribute attr) {
        return "ALTER TABLE " + PostgreSqlNameGenerator.getTableName(entityType) + " ALTER COLUMN " + PostgreSqlNameGenerator.getColumnName(attr) + " SET DATA TYPE " + PostgreSqlQueryGenerator.getPostgreSqlType(attr) + " USING " + PostgreSqlNameGenerator.getColumnName(attr) + "::" + PostgreSqlQueryGenerator.getPostgreSqlType(attr);
    }

    static String getSqlAddColumn(EntityType entityType, Attribute attr, ColumnMode columnMode) {
        StringBuilder sql = new StringBuilder("ALTER TABLE ");
        String columnSql = PostgreSqlQueryGenerator.getSqlColumn(entityType, attr, columnMode);
        sql.append(PostgreSqlNameGenerator.getTableName(entityType)).append(" ADD ").append(columnSql);
        List<String> sqlTableConstraints = PostgreSqlQueryGenerator.getSqlTableConstraints(entityType, attr);
        if (!sqlTableConstraints.isEmpty()) {
            sqlTableConstraints.forEach(sqlTableConstraint -> sql.append(",ADD ").append((String)sqlTableConstraint));
        }
        return sql.toString();
    }

    static String getSqlDropColumnDefault(EntityType entityType, Attribute attr) {
        return "ALTER TABLE " + PostgreSqlNameGenerator.getTableName(entityType) + " ALTER COLUMN " + PostgreSqlNameGenerator.getColumnName(attr) + " DROP DEFAULT";
    }

    static String getSqlCreateTable(EntityType entityType) {
        List persistedTableAttrs = PostgreSqlQueryUtils.getTableAttributes(entityType).collect(Collectors.toList());
        StringBuilder sql = new StringBuilder("CREATE TABLE ").append(PostgreSqlNameGenerator.getTableName(entityType)).append('(');
        Iterator it = persistedTableAttrs.iterator();
        while (it.hasNext()) {
            Attribute attr = (Attribute)it.next();
            sql.append(PostgreSqlQueryGenerator.getSqlColumn(entityType, attr, ColumnMode.EXCLUDE_DEFAULT_CONSTRAINT));
            if (!it.hasNext()) continue;
            sql.append(',');
        }
        for (Attribute persistedTableAttr : persistedTableAttrs) {
            List<String> sqlTableConstraints = PostgreSqlQueryGenerator.getSqlTableConstraints(entityType, persistedTableAttr);
            if (sqlTableConstraints.isEmpty()) continue;
            sqlTableConstraints.forEach(sqlTableConstraint -> sql.append(',').append((String)sqlTableConstraint));
        }
        sql.append(')');
        return sql.toString();
    }

    static String getSqlCreateFunctionValidateUpdate(EntityType entityType, Collection<Attribute> readonlyTableAttrs) {
        StringBuilder strBuilder = new StringBuilder(512).append("CREATE FUNCTION ").append(PostgreSqlNameGenerator.getFunctionValidateUpdateName(entityType)).append("() RETURNS TRIGGER AS $$\nBEGIN\n");
        String tableName = PostgreSqlNameGenerator.getTableName(entityType);
        String idColName = PostgreSqlNameGenerator.getColumnName(entityType.getIdAttribute());
        readonlyTableAttrs.forEach(attr -> {
            String colName = PostgreSqlNameGenerator.getColumnName(attr);
            strBuilder.append("  IF OLD.").append(colName).append(" <> NEW.").append(colName).append(" THEN\n");
            strBuilder.append("    RAISE EXCEPTION 'Updating read-only column ").append(colName).append(" of table ").append(tableName).append(" with id [%] is not allowed', OLD.").append(idColName).append(" USING ERRCODE = '").append(ERR_CODE_READONLY_VIOLATION).append("';\n");
            strBuilder.append("  END IF;\n");
        });
        strBuilder.append("  RETURN NEW;\nEND;\n$$ LANGUAGE plpgsql;");
        return strBuilder.toString();
    }

    static String getSqlDropFunctionValidateUpdate(EntityType entityType) {
        return "DROP FUNCTION " + PostgreSqlNameGenerator.getFunctionValidateUpdateName(entityType) + "();";
    }

    static String getSqlCreateUpdateTrigger(EntityType entityType, Collection<Attribute> readonlyTableAttrs) {
        StringBuilder strBuilder = new StringBuilder(512).append("CREATE TRIGGER ").append(PostgreSqlNameGenerator.getUpdateTriggerName(entityType)).append(" AFTER UPDATE ON ").append(PostgreSqlNameGenerator.getTableName(entityType)).append(" FOR EACH ROW WHEN (");
        strBuilder.append(readonlyTableAttrs.stream().map(attr -> "OLD." + PostgreSqlNameGenerator.getColumnName(attr) + " IS DISTINCT FROM NEW." + PostgreSqlNameGenerator.getColumnName(attr)).collect(Collectors.joining(" OR ")));
        strBuilder.append(") EXECUTE PROCEDURE ").append(PostgreSqlNameGenerator.getFunctionValidateUpdateName(entityType)).append("();");
        return strBuilder.toString();
    }

    static String getSqlDropUpdateTrigger(EntityType entityType) {
        return "DROP TRIGGER " + PostgreSqlNameGenerator.getUpdateTriggerName(entityType) + " ON " + PostgreSqlNameGenerator.getTableName(entityType);
    }

    static String getSqlCreateJunctionTable(EntityType entityType, Attribute attr) {
        Attribute idAttr = entityType.getIdAttribute();
        StringBuilder sql = new StringBuilder("CREATE TABLE ").append(PostgreSqlNameGenerator.getJunctionTableName(entityType, attr)).append(" (").append(PostgreSqlNameGenerator.getJunctionTableOrderColumnName()).append(" INT,").append(PostgreSqlNameGenerator.getColumnName(idAttr)).append(' ').append(PostgreSqlQueryGenerator.getPostgreSqlType(idAttr)).append(" NOT NULL, ").append(PostgreSqlNameGenerator.getColumnName(attr)).append(' ').append(PostgreSqlQueryGenerator.getPostgreSqlType(attr.getRefEntity().getIdAttribute())).append(" NOT NULL").append(", FOREIGN KEY (").append(PostgreSqlNameGenerator.getColumnName(idAttr)).append(") REFERENCES ").append(PostgreSqlNameGenerator.getTableName(entityType)).append('(').append(PostgreSqlNameGenerator.getColumnName(idAttr)).append(") ON DELETE CASCADE");
        if (attr.getRefEntity().getId().equals(entityType.getId())) {
            sql.append(" DEFERRABLE INITIALLY DEFERRED");
        }
        if (PostgreSqlQueryUtils.isPersistedInPostgreSql(attr.getRefEntity())) {
            sql.append(", FOREIGN KEY (").append(PostgreSqlNameGenerator.getColumnName(attr)).append(") REFERENCES ").append(PostgreSqlNameGenerator.getTableName(attr.getRefEntity())).append('(').append(PostgreSqlNameGenerator.getColumnName(attr.getRefEntity().getIdAttribute())).append(")");
            if (attr.getRefEntity().getId().equals(entityType.getId())) {
                sql.append(" DEFERRABLE INITIALLY DEFERRED");
            }
        }
        AttributeType attrType = attr.getDataType();
        switch (attrType) {
            case CATEGORICAL_MREF: 
            case MREF: {
                sql.append(", UNIQUE (").append(PostgreSqlNameGenerator.getColumnName(idAttr)).append(',').append(PostgreSqlNameGenerator.getColumnName(attr)).append(')');
                break;
            }
            default: {
                throw new IllegalAttributeTypeException(attrType);
            }
        }
        sql.append(", UNIQUE (").append(PostgreSqlNameGenerator.getJunctionTableOrderColumnName()).append(',').append(PostgreSqlNameGenerator.getColumnName(idAttr)).append(')');
        sql.append(')');
        return sql.toString();
    }

    static String getSqlCreateJunctionTableIndex(EntityType entityType, Attribute attr) {
        Attribute idAttr = entityType.getIdAttribute();
        String junctionTableName = PostgreSqlNameGenerator.getJunctionTableName(entityType, attr);
        String junctionTableIndexName = PostgreSqlNameGenerator.getJunctionTableIndexName(entityType, attr, idAttr);
        String idxColumnName = PostgreSqlNameGenerator.getColumnName(idAttr);
        return "CREATE INDEX " + junctionTableIndexName + " ON " + junctionTableName + " (" + idxColumnName + ')';
    }

    static String getSqlDropJunctionTable(EntityType entityType, Attribute attr) {
        return PostgreSqlQueryGenerator.getSqlDropTable(PostgreSqlNameGenerator.getJunctionTableName(entityType, attr));
    }

    static String getSqlDropTable(EntityType entityType) {
        return PostgreSqlQueryGenerator.getSqlDropTable(PostgreSqlNameGenerator.getTableName(entityType));
    }

    static String getSqlDropColumn(EntityType entityType, Attribute attr) {
        return "ALTER TABLE " + PostgreSqlNameGenerator.getTableName(entityType) + " DROP COLUMN " + PostgreSqlNameGenerator.getColumnName(attr);
    }

    static String getSqlInsert(EntityType entityType) {
        StringBuilder sql = new StringBuilder("INSERT INTO ").append(PostgreSqlNameGenerator.getTableName(entityType)).append(" (");
        StringBuilder params = new StringBuilder();
        PostgreSqlQueryUtils.getTableAttributes(entityType).forEach(attr -> {
            sql.append(PostgreSqlNameGenerator.getColumnName(attr)).append(", ");
            params.append("?, ");
        });
        if (sql.charAt(sql.length() - 1) == ' ' && sql.charAt(sql.length() - 2) == ',') {
            sql.setLength(sql.length() - 2);
            params.setLength(params.length() - 2);
        }
        sql.append(") VALUES (").append((CharSequence)params).append(')');
        return sql.toString();
    }

    static String getSqlInsertJunction(EntityType entityType, Attribute attr) {
        String junctionTableName = PostgreSqlNameGenerator.getJunctionTableName(entityType, attr);
        return "INSERT INTO " + junctionTableName + " (" + PostgreSqlNameGenerator.getJunctionTableOrderColumnName() + ',' + PostgreSqlNameGenerator.getColumnName(entityType.getIdAttribute()) + ',' + PostgreSqlNameGenerator.getColumnName(attr) + ") VALUES (?,?,?)";
    }

    static String getSqlDeleteAll(EntityType entityType) {
        return "DELETE FROM " + PostgreSqlNameGenerator.getTableName(entityType);
    }

    static String getSqlDelete(EntityType entityType) {
        return PostgreSqlQueryGenerator.getSqlDelete(PostgreSqlNameGenerator.getTableName(entityType), entityType.getIdAttribute());
    }

    static String getSqlDelete(String tableName, Attribute attr) {
        return "DELETE FROM " + tableName + " WHERE " + PostgreSqlNameGenerator.getColumnName(attr) + " = ?";
    }

    private static boolean isPersistedInOtherTable(Attribute attr) {
        boolean bidirectionalOneToMany = attr.getDataType() == AttributeType.ONE_TO_MANY && attr.isMappedBy();
        return EntityTypeUtils.isMultipleReferenceType((Attribute)attr) || bidirectionalOneToMany;
    }

    static String getSqlJunctionTableSelect(EntityType entityType, Attribute attr, int numOfIds) {
        String idColName = PostgreSqlNameGenerator.getColumnName(entityType.getIdAttribute());
        String refIdColName = PostgreSqlNameGenerator.getColumnName(attr);
        return "SELECT " + idColName + "," + PostgreSqlNameGenerator.getJunctionTableOrderColumnName() + "," + refIdColName + " FROM " + PostgreSqlNameGenerator.getJunctionTableName(entityType, attr) + " WHERE " + idColName + " in (" + IntStream.range(0, numOfIds).mapToObj(x -> "?").collect(Collectors.joining(", ")) + ") ORDER BY " + idColName + "," + PostgreSqlNameGenerator.getJunctionTableOrderColumnName();
    }

    private static <E extends Entity> boolean isDistinctSelectRequired(EntityType entityType, Query<E> q) {
        return PostgreSqlQueryGenerator.isDistinctSelectRequiredRec(entityType, q.getRules());
    }

    private static boolean isDistinctSelectRequiredRec(EntityType entityType, List<QueryRule> queryRules) {
        if (queryRules.isEmpty()) {
            return false;
        }
        for (QueryRule queryRule : queryRules) {
            if (queryRule.getOperator() == QueryRule.Operator.NESTED) {
                if (!PostgreSqlQueryGenerator.isDistinctSelectRequiredRec(entityType, queryRule.getNestedRules())) continue;
                return true;
            }
            String queryRuleField = queryRule.getField();
            if (queryRuleField == null) continue;
            String attrName = StringUtils.split((String)queryRuleField, (char)'.')[0];
            Attribute attr = entityType.getAttribute(attrName);
            if (attr == null) {
                throw new UnknownAttributeException(entityType, attrName);
            }
            if (!PostgreSqlQueryGenerator.isPersistedInOtherTable(attr)) continue;
            return true;
        }
        return false;
    }

    static <E extends Entity> String getSqlSelect(EntityType entityType, Query<E> q, List<Object> parameters, boolean includeMrefs) {
        StringBuilder select = new StringBuilder("SELECT ");
        if (PostgreSqlQueryGenerator.isDistinctSelectRequired(entityType, q)) {
            select.append("DISTINCT ");
        }
        StringBuilder group = new StringBuilder();
        AtomicInteger count = new AtomicInteger();
        Attribute idAttribute = entityType.getIdAttribute();
        PostgreSqlQueryUtils.getPersistedAttributes(entityType).forEach(attr -> {
            if (q.getFetch() == null || q.getFetch().hasField(attr.getName()) || q.getSort() != null && q.getSort().hasField(attr.getName())) {
                if (count.get() > 0) {
                    select.append(", ");
                }
                if (PostgreSqlQueryGenerator.isPersistedInOtherTable(attr)) {
                    if (includeMrefs || attr.getDataType() == AttributeType.ONE_TO_MANY && attr.isMappedBy()) {
                        if (attr.getDataType() == AttributeType.ONE_TO_MANY && attr.isMappedBy()) {
                            Attribute refIdAttr = attr.getRefEntity().getIdAttribute();
                            String mrefSelect = "(SELECT array_agg(" + PostgreSqlNameGenerator.getColumnName(refIdAttr);
                            Sort orderBy = attr.getOrderBy();
                            if (orderBy == null) {
                                orderBy = new Sort(refIdAttr.getName());
                            }
                            mrefSelect = mrefSelect + ' ' + PostgreSqlQueryGenerator.getSqlSort(attr.getRefEntity(), new QueryImpl().sort(orderBy)) + ") FROM " + PostgreSqlNameGenerator.getTableName(attr.getRefEntity()) + " WHERE this." + PostgreSqlNameGenerator.getColumnName(idAttribute) + " = " + PostgreSqlNameGenerator.getTableName(attr.getRefEntity()) + '.' + PostgreSqlNameGenerator.getColumnName(attr.getMappedBy()) + ") AS " + PostgreSqlNameGenerator.getColumnName(attr);
                            select.append(mrefSelect);
                        } else {
                            String mrefSelect = MessageFormat.format("(SELECT array_agg(DISTINCT ARRAY[{0}.{1}::TEXT,{0}.{0}::TEXT]) FROM {2} AS {0} WHERE this.{3} = {0}.{3}) AS {0}", PostgreSqlNameGenerator.getColumnName(attr), PostgreSqlNameGenerator.getJunctionTableOrderColumnName(), PostgreSqlNameGenerator.getJunctionTableName(entityType, attr), PostgreSqlNameGenerator.getColumnName(idAttribute));
                            select.append(mrefSelect);
                        }
                    } else {
                        select.append("NULL AS ").append(PostgreSqlNameGenerator.getColumnName(attr));
                    }
                } else {
                    select.append("this.").append(PostgreSqlNameGenerator.getColumnName(attr));
                    if (group.length() > 0) {
                        group.append(", this.").append(PostgreSqlNameGenerator.getColumnName(attr));
                    } else {
                        group.append("this.").append(PostgreSqlNameGenerator.getColumnName(attr));
                    }
                }
                count.incrementAndGet();
            }
        });
        StringBuilder result = new StringBuilder().append((CharSequence)select).append(PostgreSqlQueryGenerator.getSqlFrom(entityType, q));
        String where = PostgreSqlQueryGenerator.getSqlWhere(entityType, q, parameters, new AtomicInteger());
        if (where.length() > 0) {
            result.append(" WHERE ").append(where);
        }
        result.append(' ').append(PostgreSqlQueryGenerator.getSqlSort(entityType, q));
        if (q.getPageSize() > 0) {
            result.append(" LIMIT ").append(q.getPageSize());
        }
        if (q.getOffset() > 0) {
            result.append(" OFFSET ").append(q.getOffset());
        }
        return result.toString().trim();
    }

    static String getSqlUpdate(EntityType entityType) {
        Attribute idAttribute = entityType.getIdAttribute();
        StringBuilder sql = new StringBuilder("UPDATE ").append(PostgreSqlNameGenerator.getTableName(entityType)).append(" SET ");
        PostgreSqlQueryUtils.getTableAttributes(entityType).forEach(attr -> sql.append(PostgreSqlNameGenerator.getColumnName(attr)).append(" = ?, "));
        if (sql.charAt(sql.length() - 1) == ' ' && sql.charAt(sql.length() - 2) == ',') {
            sql.setLength(sql.length() - 2);
        }
        sql.append(" WHERE ").append(PostgreSqlNameGenerator.getColumnName(idAttribute)).append("= ?");
        return sql.toString();
    }

    static <E extends Entity> String getSqlCount(EntityType entityType, Query<E> q, List<Object> parameters) {
        StringBuilder sqlBuilder = new StringBuilder("SELECT COUNT");
        String idAttribute = PostgreSqlNameGenerator.getColumnName(entityType.getIdAttribute());
        List queryRules = q.getRules();
        if (queryRules == null || queryRules.isEmpty()) {
            sqlBuilder.append("(*) FROM ").append(PostgreSqlNameGenerator.getTableName(entityType));
        } else {
            boolean distinctSelectRequired = PostgreSqlQueryGenerator.isDistinctSelectRequired(entityType, q);
            if (distinctSelectRequired) {
                sqlBuilder.append("(DISTINCT this.").append(idAttribute).append(')');
            } else {
                sqlBuilder.append("(*)");
            }
            String from = PostgreSqlQueryGenerator.getSqlFrom(entityType, q);
            String where = PostgreSqlQueryGenerator.getSqlWhere(entityType, q, parameters, new AtomicInteger());
            sqlBuilder.append(from).append(" WHERE ").append(where);
        }
        return sqlBuilder.toString();
    }

    private static String getSqlColumn(EntityType entityType, Attribute attr, ColumnMode columnMode) {
        StringBuilder sqlBuilder = new StringBuilder(PostgreSqlNameGenerator.getColumnName(attr)).append(' ');
        AttributeType attrType = attr.getDataType();
        switch (attrType) {
            case BOOL: 
            case DATE: 
            case DATE_TIME: 
            case DECIMAL: 
            case EMAIL: 
            case ENUM: 
            case HTML: 
            case HYPERLINK: 
            case INT: 
            case LONG: 
            case SCRIPT: 
            case STRING: 
            case TEXT: {
                sqlBuilder.append(PostgreSqlQueryGenerator.getPostgreSqlType(attr));
                break;
            }
            case CATEGORICAL: 
            case FILE: 
            case XREF: {
                sqlBuilder.append(PostgreSqlQueryGenerator.getPostgreSqlType(attr.getRefEntity().getIdAttribute()));
                break;
            }
            case CATEGORICAL_MREF: 
            case MREF: 
            case ONE_TO_MANY: 
            case COMPOUND: {
                throw new IllegalAttributeTypeException(attrType);
            }
            default: {
                throw new UnexpectedEnumException((Enum)attrType);
            }
        }
        String sqlColumnConstraints = PostgreSqlQueryGenerator.getSqlColumnConstraints(entityType, attr, columnMode);
        if (!sqlColumnConstraints.isEmpty()) {
            sqlBuilder.append(' ').append(sqlColumnConstraints);
        }
        return sqlBuilder.toString();
    }

    static boolean generateSqlColumnDefaultConstraint(Attribute attr) {
        return attr.getDefaultValue() != null && !EntityTypeUtils.isMultipleReferenceType((Attribute)attr);
    }

    private static String getSqlColumnConstraints(EntityType entityType, Attribute attr, ColumnMode columnConstraintsMode) {
        StringBuilder sqlBuilder = new StringBuilder();
        if (!attr.getName().equals(entityType.getIdAttribute().getName()) && !attr.isNillable()) {
            sqlBuilder.append("NOT NULL");
        }
        if (columnConstraintsMode == ColumnMode.INCLUDE_DEFAULT_CONSTRAINT && PostgreSqlQueryGenerator.generateSqlColumnDefaultConstraint(attr)) {
            if (sqlBuilder.length() > 0) {
                sqlBuilder.append(' ');
            }
            sqlBuilder.append("DEFAULT ").append(PostgreSqlQueryGenerator.getSqlDefaulValue(attr));
        }
        return sqlBuilder.toString();
    }

    private static String getSqlDefaulValue(Attribute attribute) {
        return PostgreSqlQueryGenerator.getSqlDefaulValue(attribute, attribute.getDefaultValue());
    }

    private static String getSqlDefaulValue(Attribute attribute, String defaultValueAsString) {
        String sqlDefaultValue;
        Object defaultTypedValue = AttributeUtils.getDefaultTypedValue((Attribute)attribute, (String)defaultValueAsString);
        AttributeType attributeType = attribute.getDataType();
        switch (attributeType) {
            case BOOL: {
                Boolean booleanDefaultValue = (Boolean)defaultTypedValue;
                sqlDefaultValue = booleanDefaultValue != false ? "TRUE" : "FALSE";
                break;
            }
            case CATEGORICAL: 
            case FILE: 
            case XREF: {
                Entity refDefaultValue = (Entity)defaultTypedValue;
                sqlDefaultValue = PostgreSqlQueryGenerator.getSqlDefaulValue(attribute.getRefEntity().getIdAttribute(), refDefaultValue.getIdValue().toString());
                break;
            }
            case DATE: {
                LocalDate dateDefaultValue = (LocalDate)defaultTypedValue;
                sqlDefaultValue = '\'' + dateDefaultValue.toString() + '\'';
                break;
            }
            case DATE_TIME: {
                Instant instantDefaultValue = (Instant)defaultTypedValue;
                sqlDefaultValue = '\'' + instantDefaultValue.truncatedTo(ChronoUnit.SECONDS).atOffset(ZoneOffset.UTC).toString() + '\'';
                break;
            }
            case DECIMAL: {
                Double doubleDefaultValue = (Double)defaultTypedValue;
                sqlDefaultValue = doubleDefaultValue.toString();
                break;
            }
            case EMAIL: 
            case ENUM: 
            case HTML: 
            case HYPERLINK: 
            case SCRIPT: 
            case STRING: 
            case TEXT: {
                sqlDefaultValue = '\'' + (String)defaultTypedValue + '\'';
                break;
            }
            case INT: {
                Integer intDefaultValue = (Integer)defaultTypedValue;
                sqlDefaultValue = intDefaultValue.toString();
                break;
            }
            case LONG: {
                Long longDefaultValue = (Long)defaultTypedValue;
                sqlDefaultValue = longDefaultValue.toString();
                break;
            }
            case CATEGORICAL_MREF: 
            case MREF: 
            case ONE_TO_MANY: 
            case COMPOUND: {
                throw new IllegalAttributeTypeException(attributeType);
            }
            default: {
                throw new UnexpectedEnumException((Enum)attributeType);
            }
        }
        return sqlDefaultValue;
    }

    private static List<String> getSqlTableConstraints(EntityType entityType, Attribute attr) {
        ArrayList tableConstraints = Lists.newArrayList();
        if (attr.getName().equals(entityType.getIdAttribute().getName())) {
            tableConstraints.add(PostgreSqlQueryGenerator.getSqlConstraintPrimaryKey(entityType, attr));
        } else {
            if (EntityTypeUtils.isSingleReferenceType((Attribute)attr) && PostgreSqlQueryUtils.isPersistedInPostgreSql(attr.getRefEntity())) {
                tableConstraints.add(PostgreSqlQueryGenerator.getSqlForeignKey(entityType, attr));
            }
            if (attr.isUnique()) {
                tableConstraints.add(PostgreSqlQueryGenerator.getSqlUniqueKey(entityType, attr));
            }
            if (attr.getDataType() == AttributeType.ENUM) {
                tableConstraints.add(PostgreSqlQueryGenerator.getSqlCheckConstraint(entityType, attr));
            }
        }
        return tableConstraints;
    }

    private static String getSqlDropTable(String tableName) {
        return "DROP TABLE " + tableName;
    }

    static <E extends Entity> String getSqlWhere(EntityType entityType, Query<E> q, List<Object> parameters, AtomicInteger mrefFilterIndex) {
        StringBuilder result = new StringBuilder();
        block18: for (QueryRule r : q.getRules()) {
            Attribute attr = null;
            if (r.getField() != null) {
                attr = entityType.getAttribute(r.getField());
                if (attr == null) {
                    throw new MolgenisDataException(String.format("Unknown attribute [%s]", r.getField()));
                }
                if (PostgreSqlQueryGenerator.isPersistedInOtherTable(attr)) {
                    mrefFilterIndex.incrementAndGet();
                }
            }
            StringBuilder predicate = new StringBuilder();
            QueryRule.Operator operator = r.getOperator();
            switch (operator) {
                case AND: {
                    result.append(" AND ");
                    continue block18;
                }
                case NESTED: {
                    QueryImpl nestedQ = new QueryImpl(r.getNestedRules());
                    result.append('(').append(PostgreSqlQueryGenerator.getSqlWhere(entityType, nestedQ, parameters, mrefFilterIndex)).append(')');
                    continue block18;
                }
                case OR: {
                    result.append(" OR ");
                    continue block18;
                }
                case LIKE: {
                    Objects.requireNonNull(attr, String.format(UNSPECIFIED_ATTRIBUTE_MSG, QueryRule.Operator.LIKE));
                    String columnName = PostgreSqlQueryGenerator.isPersistedInOtherTable(attr) ? PostgreSqlNameGenerator.getFilterColumnName(attr, mrefFilterIndex.get()) : "this." + PostgreSqlNameGenerator.getColumnName(attr);
                    if (EntityTypeUtils.isStringType((Attribute)attr) || EntityTypeUtils.isTextType((Attribute)attr)) {
                        result.append(' ').append(columnName);
                    } else {
                        result.append(" CAST(").append(columnName).append(" as TEXT)");
                    }
                    result.append(" LIKE ?");
                    parameters.add("%" + PostgreSqlUtils.getPostgreSqlQueryValue(r.getValue(), attr) + '%');
                    continue block18;
                }
                case IN: {
                    Objects.requireNonNull(attr, String.format(UNSPECIFIED_ATTRIBUTE_MSG, QueryRule.Operator.IN));
                    Object inValue = r.getValue();
                    if (inValue == null) {
                        throw new MolgenisDataException("Missing value for IN query");
                    }
                    if (!(inValue instanceof Iterable)) {
                        throw new MolgenisDataException(String.format("IN value is of type [%s] instead of [Iterable]", inValue.getClass().getSimpleName()));
                    }
                    StringBuilder in = new StringBuilder();
                    Attribute inAttr = attr;
                    Stream<Object> postgreSqlIds = StreamSupport.stream(((Iterable)inValue).spliterator(), false).map(idValue -> PostgreSqlUtils.getPostgreSqlQueryValue(idValue, inAttr));
                    Iterator it = postgreSqlIds.iterator();
                    while (it.hasNext()) {
                        Object postgreSqlId = it.next();
                        in.append('?');
                        if (it.hasNext()) {
                            in.append(',');
                        }
                        parameters.add(postgreSqlId);
                    }
                    if (PostgreSqlQueryGenerator.isPersistedInOtherTable(attr)) {
                        result.append(PostgreSqlNameGenerator.getFilterColumnName(attr, mrefFilterIndex.get()));
                    } else {
                        result.append("this");
                    }
                    Attribute equalsAttr = attr.isMappedBy() ? attr.getRefEntity().getIdAttribute() : entityType.getAttribute(r.getField());
                    result.append('.').append(PostgreSqlNameGenerator.getColumnName(equalsAttr));
                    result.append(" IN (").append((CharSequence)in).append(')');
                    continue block18;
                }
                case NOT: {
                    result.append(" NOT ");
                    continue block18;
                }
                case RANGE: {
                    Objects.requireNonNull(attr, String.format(UNSPECIFIED_ATTRIBUTE_MSG, QueryRule.Operator.RANGE));
                    Object range = r.getValue();
                    if (range == null) {
                        throw new MolgenisDataException("Missing value for RANGE query");
                    }
                    if (!(range instanceof Iterable)) {
                        throw new MolgenisDataException(String.format("RANGE value is of type [%s] instead of [Iterable]", range.getClass().getSimpleName()));
                    }
                    Iterator rangeValues = ((Iterable)range).iterator();
                    parameters.add(rangeValues.next());
                    parameters.add(rangeValues.next());
                    StringBuilder column = new StringBuilder();
                    if (PostgreSqlQueryGenerator.isPersistedInOtherTable(attr)) {
                        column.append(PostgreSqlNameGenerator.getFilterColumnName(attr, mrefFilterIndex.get()));
                    } else {
                        column.append("this");
                    }
                    column.append('.').append(PostgreSqlNameGenerator.getColumnName(entityType.getAttribute(r.getField())));
                    predicate.append((CharSequence)column).append(" >= ? AND ").append((CharSequence)column).append(" <= ?");
                    result.append((CharSequence)predicate);
                    continue block18;
                }
                case EQUALS: {
                    if (attr == null) {
                        throw new MolgenisDataException("Missing attribute field in EQUALS query rule");
                    }
                    if (PostgreSqlQueryGenerator.isPersistedInOtherTable(attr)) {
                        predicate.append(PostgreSqlNameGenerator.getFilterColumnName(attr, mrefFilterIndex.get()));
                    } else {
                        predicate.append("this");
                    }
                    Attribute equalsAttr = attr.isMappedBy() ? attr.getRefEntity().getIdAttribute() : entityType.getAttribute(r.getField());
                    predicate.append('.').append(PostgreSqlNameGenerator.getColumnName(equalsAttr));
                    if (r.getValue() == null) {
                        predicate.append(" IS NULL ");
                    } else {
                        Object postgreSqlVal = PostgreSqlUtils.getPostgreSqlQueryValue(r.getValue(), attr);
                        if (attr.getDataType() == AttributeType.BOOL) {
                            Boolean bool = (Boolean)postgreSqlVal;
                            if (bool.booleanValue()) {
                                predicate.append(" IS TRUE");
                            } else {
                                predicate.append(" IS FALSE");
                            }
                        } else {
                            predicate.append(" =");
                            predicate.append(" ? ");
                            parameters.add(postgreSqlVal);
                        }
                    }
                    if (!(result.length() <= 0 || result.toString().endsWith(" OR ") || result.toString().endsWith(" AND ") || result.toString().endsWith(" NOT "))) {
                        result.append(" AND ");
                    }
                    result.append((CharSequence)predicate);
                    continue block18;
                }
                case GREATER: 
                case GREATER_EQUAL: 
                case LESS: 
                case LESS_EQUAL: {
                    Objects.requireNonNull(attr, String.format(UNSPECIFIED_ATTRIBUTE_MSG, String.format("%s, %s, %s or %s", QueryRule.Operator.GREATER, QueryRule.Operator.GREATER_EQUAL, QueryRule.Operator.LESS, QueryRule.Operator.LESS_EQUAL)));
                    if (PostgreSqlQueryGenerator.isPersistedInOtherTable(attr)) {
                        predicate.append(PostgreSqlNameGenerator.getFilterColumnName(attr, mrefFilterIndex.get()));
                    } else {
                        predicate.append("this");
                    }
                    predicate.append('.').append(PostgreSqlNameGenerator.getColumnName(entityType.getAttribute(r.getField())));
                    switch (operator) {
                        case GREATER: {
                            predicate.append(" >");
                            break;
                        }
                        case GREATER_EQUAL: {
                            predicate.append(" >=");
                            break;
                        }
                        case LESS: {
                            predicate.append(" <");
                            break;
                        }
                        case LESS_EQUAL: {
                            predicate.append(" <=");
                            break;
                        }
                        default: {
                            throw new RuntimeException(String.format("Unexpected query operator [%s]", operator));
                        }
                    }
                    predicate.append(" ? ");
                    parameters.add(PostgreSqlUtils.getPostgreSqlQueryValue(r.getValue(), attr));
                    if (!(result.length() <= 0 || result.toString().endsWith(" OR ") || result.toString().endsWith(" AND ") || result.toString().endsWith(" NOT "))) {
                        result.append(" AND ");
                    }
                    result.append((CharSequence)predicate);
                    continue block18;
                }
                case DIS_MAX: 
                case FUZZY_MATCH: 
                case FUZZY_MATCH_NGRAM: 
                case SEARCH: 
                case SHOULD: {
                    throw new UnsupportedOperationException(String.format("Query operator [%s] not supported by PostgreSQL repository", operator.toString()));
                }
            }
            throw new UnexpectedEnumException((Enum)operator);
        }
        return result.toString().trim();
    }

    static <E extends Entity> String getSqlSort(EntityType entityType, Query<E> q) {
        Sort sort;
        StringBuilder sortSql = new StringBuilder();
        if (q.getSort() != null && !PostgreSqlQueryGenerator.hasUniqueSortAttribute(entityType, q.getSort())) {
            LOG.debug("Query with sort without unique attribute detected: {}", q);
            sort = new Sort(q.getSort());
            sort.on(entityType.getIdAttribute().getName());
        } else if (q.getSort() == null) {
            LOG.debug("Query without sort detected: {}", q);
            sort = new Sort(entityType.getIdAttribute().getName());
        } else {
            sort = q.getSort();
        }
        for (Sort.Order o : sort) {
            Attribute attr = entityType.getAttribute(o.getAttr());
            sortSql.append(", ").append(PostgreSqlNameGenerator.getColumnName(attr));
            if (o.getDirection().equals((Object)Sort.Direction.DESC)) {
                sortSql.append(" DESC");
                continue;
            }
            sortSql.append(" ASC");
        }
        if (sortSql.length() > 0) {
            sortSql = new StringBuilder("ORDER BY ").append(sortSql.substring(2));
        }
        return sortSql.toString();
    }

    private static boolean hasUniqueSortAttribute(EntityType entityType, Sort sort) {
        for (Sort.Order order : sort) {
            String attributeName = order.getAttr();
            Attribute attribute = entityType.getAttribute(attributeName);
            if (!attribute.isUnique()) continue;
            return true;
        }
        return false;
    }

    private static <E extends Entity> String getSqlFrom(EntityType entityType, Query<E> q) {
        List<Attribute> mrefAttrsInQuery = PostgreSqlQueryGenerator.getJoinQueryAttrs(entityType, q);
        StringBuilder from = new StringBuilder(" FROM ").append(PostgreSqlNameGenerator.getTableName(entityType)).append(" AS this");
        Attribute idAttribute = entityType.getIdAttribute();
        for (int i = 0; i < mrefAttrsInQuery.size(); ++i) {
            Attribute mrefAttr = mrefAttrsInQuery.get(i);
            if (mrefAttr.getDataType() == AttributeType.ONE_TO_MANY && mrefAttr.isMappedBy()) {
                from.append(" LEFT JOIN ").append(PostgreSqlNameGenerator.getTableName(mrefAttr.getRefEntity())).append(" AS ").append(PostgreSqlNameGenerator.getFilterColumnName(mrefAttr, i + 1)).append(" ON (this.").append(PostgreSqlNameGenerator.getColumnName(idAttribute)).append(" = ").append(PostgreSqlNameGenerator.getFilterColumnName(mrefAttr, i + 1)).append('.').append(PostgreSqlNameGenerator.getColumnName(mrefAttr.getMappedBy())).append(')');
                continue;
            }
            from.append(" LEFT JOIN ").append(PostgreSqlNameGenerator.getJunctionTableName(entityType, mrefAttr)).append(" AS ").append(PostgreSqlNameGenerator.getFilterColumnName(mrefAttr, i + 1)).append(" ON (this.").append(PostgreSqlNameGenerator.getColumnName(idAttribute)).append(" = ").append(PostgreSqlNameGenerator.getFilterColumnName(mrefAttr, i + 1)).append('.').append(PostgreSqlNameGenerator.getColumnName(idAttribute)).append(')');
        }
        return from.toString();
    }

    private static <E extends Entity> List<Attribute> getJoinQueryAttrs(EntityType entityType, Query<E> q) {
        ArrayList joinAttrs = Lists.newArrayList();
        PostgreSqlQueryGenerator.getJoinQueryAttrsRec(entityType, q.getRules(), joinAttrs);
        return joinAttrs;
    }

    private static void getJoinQueryAttrsRec(EntityType entityType, List<QueryRule> rules, List<Attribute> joinAttrs) {
        for (QueryRule rule : rules) {
            Attribute attr;
            if (rule.getField() != null && (attr = entityType.getAttribute(rule.getField())) != null && PostgreSqlQueryGenerator.isPersistedInOtherTable(attr)) {
                joinAttrs.add(attr);
            }
            if (rule.getNestedRules() == null || rule.getNestedRules().isEmpty()) continue;
            PostgreSqlQueryGenerator.getJoinQueryAttrsRec(entityType, rule.getNestedRules(), joinAttrs);
        }
    }

    private static String getPostgreSqlType(Attribute attr) {
        AttributeType attrType;
        block12: while (true) {
            attrType = attr.getDataType();
            switch (attrType) {
                case BOOL: {
                    return "boolean";
                }
                case CATEGORICAL: 
                case FILE: 
                case XREF: {
                    attr = attr.getRefEntity().getIdAttribute();
                    continue block12;
                }
                case DATE: {
                    return "date";
                }
                case DATE_TIME: {
                    return "timestamp with time zone";
                }
                case DECIMAL: {
                    return "double precision";
                }
                case EMAIL: 
                case ENUM: 
                case HYPERLINK: 
                case STRING: {
                    return "character varying(255)";
                }
                case HTML: 
                case SCRIPT: 
                case TEXT: {
                    return "text";
                }
                case INT: {
                    return "integer";
                }
                case LONG: {
                    return "bigint";
                }
                case CATEGORICAL_MREF: 
                case MREF: 
                case ONE_TO_MANY: 
                case COMPOUND: {
                    throw new IllegalAttributeTypeException(attrType);
                }
            }
            break;
        }
        throw new UnexpectedEnumException((Enum)attrType);
    }

    static enum ColumnMode {
        INCLUDE_DEFAULT_CONSTRAINT,
        EXCLUDE_DEFAULT_CONSTRAINT;

    }
}

