/*
 * Decompiled with CFR 0.152.
 */
package org.molgenis.data.postgresql;

import java.sql.BatchUpdateException;
import java.sql.SQLException;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.Objects;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javax.sql.DataSource;
import org.molgenis.data.MolgenisDataException;
import org.molgenis.data.meta.AttributeType;
import org.molgenis.data.postgresql.identifier.AttributeDescription;
import org.molgenis.data.postgresql.identifier.EntityTypeDescription;
import org.molgenis.data.postgresql.identifier.EntityTypeRegistry;
import org.molgenis.data.transaction.TransactionExceptionTranslator;
import org.molgenis.data.validation.ConstraintViolation;
import org.molgenis.data.validation.MolgenisValidationException;
import org.postgresql.util.PSQLException;
import org.postgresql.util.ServerErrorMessage;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.support.SQLErrorCodeSQLExceptionTranslator;
import org.springframework.stereotype.Component;
import org.springframework.transaction.TransactionException;

@Component
class PostgreSqlExceptionTranslator
extends SQLErrorCodeSQLExceptionTranslator
implements TransactionExceptionTranslator {
    private static final Logger LOG = LoggerFactory.getLogger(PostgreSqlExceptionTranslator.class);
    private final EntityTypeRegistry entityTypeRegistry;

    @Autowired
    PostgreSqlExceptionTranslator(DataSource dataSource, EntityTypeRegistry entityTypeRegistry) {
        super(Objects.requireNonNull(dataSource));
        this.entityTypeRegistry = Objects.requireNonNull(entityTypeRegistry);
    }

    protected DataAccessException doTranslate(String task, String sql, SQLException ex) {
        DataAccessException dataAccessException = super.doTranslate(task, sql, ex);
        if (dataAccessException == null) {
            return this.doTranslate(ex);
        }
        return this.doTranslate(dataAccessException);
    }

    private MolgenisDataException doTranslate(DataAccessException dataAccessException) {
        Throwable cause = dataAccessException.getCause();
        if (!(cause instanceof PSQLException)) {
            throw new RuntimeException(String.format("Unexpected exception class [%s]", cause.getClass().getSimpleName()));
        }
        PSQLException pSqlException = (PSQLException)cause;
        MolgenisDataException molgenisDataException = this.doTranslate(pSqlException);
        if (molgenisDataException == null) {
            molgenisDataException = new MolgenisDataException((Throwable)dataAccessException);
        }
        return molgenisDataException;
    }

    private MolgenisDataException doTranslate(SQLException sqlException) {
        if (sqlException instanceof BatchUpdateException) {
            sqlException = sqlException.getNextException();
        }
        if (!(sqlException instanceof PSQLException)) {
            throw new RuntimeException(String.format("Unexpected exception class [%s]", sqlException.getClass().getSimpleName()));
        }
        PSQLException pSqlException = (PSQLException)sqlException;
        MolgenisDataException molgenisDataException = this.doTranslate(pSqlException);
        if (molgenisDataException == null) {
            molgenisDataException = new MolgenisDataException((Throwable)sqlException);
        }
        return molgenisDataException;
    }

    private MolgenisDataException doTranslate(PSQLException pSqlException) {
        switch (pSqlException.getSQLState()) {
            case "22007": 
            case "22P02": {
                return PostgreSqlExceptionTranslator.translateInvalidIntegerException(pSqlException);
            }
            case "23502": {
                return this.translateNotNullViolation(pSqlException);
            }
            case "23503": {
                return this.translateForeignKeyViolation(pSqlException);
            }
            case "23505": {
                return this.translateUniqueKeyViolation(pSqlException);
            }
            case "23514": {
                return this.translateCheckConstraintViolation(pSqlException);
            }
            case "2BP01": {
                return this.translateDependentObjectsStillExist(pSqlException);
            }
            case "42703": {
                return PostgreSqlExceptionTranslator.translateUndefinedColumnException(pSqlException);
            }
            case "23506": {
                return this.translateReadonlyViolation(pSqlException);
            }
        }
        return null;
    }

    MolgenisValidationException translateReadonlyViolation(PSQLException pSqlException) {
        Matcher matcher = Pattern.compile("Updating read-only column \"?(.*?)\"? of table \"?(.*?)\"? with id \\[(.*?)] is not allowed").matcher(pSqlException.getServerErrorMessage().getMessage());
        boolean matches = matcher.matches();
        if (!matches) {
            LOG.error("Error translating postgres exception: ", (Throwable)pSqlException);
            throw new RuntimeException("Error translating exception", (Throwable)pSqlException);
        }
        String colName = matcher.group(1);
        String tableName = matcher.group(2);
        String id = matcher.group(3);
        ConstraintViolation constraintViolation = new ConstraintViolation(String.format("Updating read-only attribute '%s' of entity '%s' with id '%s' is not allowed.", this.getAttributeName(tableName, colName), this.getEntityTypeName(tableName), id));
        return new MolgenisValidationException(Collections.singleton(constraintViolation));
    }

    MolgenisValidationException translateDependentObjectsStillExist(PSQLException pSqlException) {
        ServerErrorMessage serverErrorMessage = pSqlException.getServerErrorMessage();
        String detail = serverErrorMessage.getDetail();
        Matcher matcher = Pattern.compile("constraint (.+) on table \"?([^\"]+)\"? depends on table \"?([^\"]+)\"?\n?").matcher(detail);
        LinkedHashMap<String, Set> entityTypeDependencyMap = new LinkedHashMap<String, Set>();
        while (matcher.find()) {
            String tableName = matcher.group(2);
            String dependentTableName = matcher.group(3);
            String entityTypeName = this.getEntityTypeName(tableName);
            String dependentEntityTypeName = this.getEntityTypeName(dependentTableName);
            Set dependentTableNames = entityTypeDependencyMap.computeIfAbsent(dependentEntityTypeName, k -> new LinkedHashSet());
            dependentTableNames.add(entityTypeName);
        }
        if (entityTypeDependencyMap.isEmpty()) {
            LOG.error("Error translating postgres exception: ", (Throwable)pSqlException);
            throw new RuntimeException("Error translating exception", (Throwable)pSqlException);
        }
        Set constraintViolations = entityTypeDependencyMap.entrySet().stream().map(entry -> {
            String message = ((Set)entry.getValue()).size() == 1 ? String.format("Cannot delete entity '%s' because entity '%s' depends on it.", entry.getKey(), ((Set)entry.getValue()).iterator().next()) : String.format("Cannot delete entity '%s' because entities '%s' depend on it.", entry.getKey(), ((Set)entry.getValue()).stream().collect(Collectors.joining(", ")));
            return new ConstraintViolation(message, null);
        }).collect(Collectors.toCollection(LinkedHashSet::new));
        return new MolgenisValidationException(constraintViolations);
    }

    static MolgenisValidationException translateInvalidIntegerException(PSQLException pSqlException) {
        String type;
        String postgreSqlType;
        ServerErrorMessage serverErrorMessage = pSqlException.getServerErrorMessage();
        String message = serverErrorMessage.getMessage();
        Matcher matcher = Pattern.compile("invalid input syntax for \\b(?:type )?\\b(.+?): \"(.*?)\"").matcher(message);
        boolean matches = matcher.matches();
        if (!matches) {
            throw new RuntimeException("Error translating exception", (Throwable)pSqlException);
        }
        switch (postgreSqlType = matcher.group(1)) {
            case "boolean": {
                type = AttributeType.BOOL.toString();
                break;
            }
            case "date": {
                type = AttributeType.DATE.toString();
                break;
            }
            case "timestamp with time zone": {
                type = AttributeType.DATE_TIME.toString();
                break;
            }
            case "double precision": {
                type = AttributeType.DECIMAL.toString();
                break;
            }
            case "integer": {
                type = AttributeType.INT.toString() + " or " + AttributeType.LONG.toString();
                break;
            }
            default: {
                type = postgreSqlType;
            }
        }
        String value = matcher.group(2);
        ConstraintViolation constraintViolation = new ConstraintViolation(String.format("Value [%s] of this entity attribute is not of type [%s].", value, type), null);
        return new MolgenisValidationException(Collections.singleton(constraintViolation));
    }

    MolgenisValidationException translateNotNullViolation(PSQLException pSqlException) {
        ServerErrorMessage serverErrorMessage = pSqlException.getServerErrorMessage();
        String tableName = serverErrorMessage.getTable();
        String message = serverErrorMessage.getMessage();
        Matcher matcher = Pattern.compile("null value in column \"?(.*?)\"? violates not-null constraint").matcher(message);
        boolean matches = matcher.matches();
        if (matches) {
            String columnName = matcher.group(1);
            EntityTypeDescription entityTypeDescription = this.entityTypeRegistry.getEntityTypeDescription(tableName);
            entityTypeDescription.getAttributeDescriptionMap().get((Object)columnName);
            ConstraintViolation constraintViolation = new ConstraintViolation(String.format("The attribute '%s' of entity '%s' can not be null.", this.getAttributeName(tableName, columnName), this.getEntityTypeName(tableName)), null);
            return new MolgenisValidationException(Collections.singleton(constraintViolation));
        }
        matcher = Pattern.compile("column \"(.*?)\" contains null values").matcher(message);
        matches = matcher.matches();
        if (!matches) {
            throw new RuntimeException("Error translating exception", (Throwable)pSqlException);
        }
        String columnName = matcher.group(1);
        ConstraintViolation constraintViolation = new ConstraintViolation(String.format("The attribute '%s' of entity '%s' contains null values.", this.getAttributeName(tableName, columnName), this.getEntityTypeName(tableName)), null);
        return new MolgenisValidationException(Collections.singleton(constraintViolation));
    }

    MolgenisValidationException translateForeignKeyViolation(PSQLException pSqlException) {
        String attrName;
        String constraintViolationMessageTemplate;
        ServerErrorMessage serverErrorMessage = pSqlException.getServerErrorMessage();
        String tableName = serverErrorMessage.getTable();
        String detailMessage = serverErrorMessage.getDetail();
        Matcher m = Pattern.compile("\\((.*?)\\)").matcher(detailMessage);
        if (!m.find()) {
            LOG.error("Error translating postgres exception: ", (Throwable)pSqlException);
            throw new RuntimeException("Error translating exception", (Throwable)pSqlException);
        }
        String colName = m.group(1);
        if (!m.find()) {
            LOG.error("Error translating postgres exception: ", (Throwable)pSqlException);
            throw new RuntimeException("Error translating exception", (Throwable)pSqlException);
        }
        String value = m.group(1);
        if (detailMessage.contains("still referenced from")) {
            constraintViolationMessageTemplate = "Value '%s' for attribute '%s' is referenced by entity '%s'.";
            String refTableName = this.getRefTableFromForeignKeyPsqlException(pSqlException);
            attrName = this.getAttributeName(refTableName, colName);
        } else {
            constraintViolationMessageTemplate = "Unknown xref value '%s' for attribute '%s' of entity '%s'.";
            attrName = this.getAttributeName(tableName, colName);
        }
        ConstraintViolation constraintViolation = new ConstraintViolation(String.format(constraintViolationMessageTemplate, value, attrName, this.getEntityTypeName(tableName)), null);
        return new MolgenisValidationException(Collections.singleton(constraintViolation));
    }

    private String getRefTableFromForeignKeyPsqlException(PSQLException pSqlException) {
        ServerErrorMessage serverErrorMessage = pSqlException.getServerErrorMessage();
        Matcher messageMatcher = Pattern.compile("update or delete on table \"(.*)\" violates foreign key constraint \"(.*)\" on table \"(.*)\"").matcher(serverErrorMessage.getMessage());
        if (!messageMatcher.matches()) {
            LOG.error("Error translating postgres exception: ", (Throwable)pSqlException);
            throw new RuntimeException("Error translating exception", (Throwable)pSqlException);
        }
        return messageMatcher.group(1);
    }

    MolgenisValidationException translateUniqueKeyViolation(PSQLException pSqlException) {
        ServerErrorMessage serverErrorMessage = pSqlException.getServerErrorMessage();
        String tableName = serverErrorMessage.getTable();
        String detailMessage = serverErrorMessage.getDetail();
        Matcher matcher = Pattern.compile("Key \\((.*?)\\)=\\((.*?)\\) already exists.").matcher(detailMessage);
        boolean matches = matcher.matches();
        if (matches) {
            String columnName = matcher.group(1);
            String value = matcher.group(2);
            ConstraintViolation constraintViolation = new ConstraintViolation(String.format("Duplicate value '%s' for unique attribute '%s' from entity '%s'.", value, this.getAttributeName(tableName, columnName), this.getEntityTypeName(tableName)), null);
            return new MolgenisValidationException(Collections.singleton(constraintViolation));
        }
        matcher = Pattern.compile("Key \\((.*?)\\)=\\((.*?)\\) is duplicated.").matcher(detailMessage);
        matches = matcher.matches();
        if (matches) {
            String columnName = matcher.group(1);
            String value = matcher.group(2);
            ConstraintViolation constraintViolation = new ConstraintViolation(String.format("The attribute '%s' of entity '%s' contains duplicate value '%s'.", this.getAttributeName(tableName, columnName), this.getEntityTypeName(tableName), value), null);
            return new MolgenisValidationException(Collections.singleton(constraintViolation));
        }
        LOG.error("Error translating postgres exception: ", (Throwable)pSqlException);
        throw new RuntimeException("Error translating exception", (Throwable)pSqlException);
    }

    MolgenisValidationException translateCheckConstraintViolation(PSQLException pSqlException) {
        ServerErrorMessage serverErrorMessage = pSqlException.getServerErrorMessage();
        String tableName = serverErrorMessage.getTable();
        String constraintName = serverErrorMessage.getConstraint();
        String columnName = constraintName.substring(tableName.length() + 1, constraintName.length() - 4);
        ConstraintViolation constraintViolation = new ConstraintViolation(String.format("Unknown enum value for attribute '%s' of entity '%s'.", this.getAttributeName(tableName, columnName), this.getEntityTypeName(tableName)), null);
        return new MolgenisValidationException(Collections.singleton(constraintViolation));
    }

    static MolgenisValidationException translateUndefinedColumnException(PSQLException pSqlException) {
        ServerErrorMessage serverErrorMessage = pSqlException.getServerErrorMessage();
        String message = serverErrorMessage.getMessage();
        ConstraintViolation constraintViolation = new ConstraintViolation(message);
        return new MolgenisValidationException(Collections.singleton(constraintViolation));
    }

    public MolgenisDataException doTranslate(TransactionException transactionException) {
        Throwable cause = transactionException.getCause();
        if (!(cause instanceof PSQLException)) {
            return null;
        }
        PSQLException pSqlException = (PSQLException)cause;
        return this.doTranslate(pSqlException);
    }

    private String getEntityTypeName(String tableName) {
        EntityTypeDescription entityTypeDescription = this.entityTypeRegistry.getEntityTypeDescription(tableName);
        if (entityTypeDescription == null) {
            throw new RuntimeException(String.format("Unknown entity for table name [%s]", tableName));
        }
        return entityTypeDescription.getId();
    }

    private String getAttributeName(String tableName, String colName) {
        EntityTypeDescription entityTypeDescription = this.entityTypeRegistry.getEntityTypeDescription(tableName);
        if (entityTypeDescription == null) {
            throw new RuntimeException(String.format("Unknown entity for table name [%s]", tableName));
        }
        AttributeDescription attrDescription = (AttributeDescription)entityTypeDescription.getAttributeDescriptionMap().get((Object)colName);
        if (attrDescription == null) {
            throw new RuntimeException(String.format("Unknown attribute for column name [%s]", colName));
        }
        return attrDescription.getName();
    }
}

