///////////////////////////////////////////////////////////////////////////////
// AUTHOR:       Henry Pinkard, henry.pinkard@gmail.com
//
// COPYRIGHT:    University of California, San Francisco, 2015
//
// LICENSE:      This file is distributed under the BSD license.
//               License text is included with the source distribution.
//
//               This file is distributed in the hope that it will be useful,
//               but WITHOUT ANY WARRANTY; without even the implied warranty
//               of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//
//               IN NO EVENT SHALL THE COPYRIGHT OWNER OR
//               CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
//               INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES.
//

package org.micromanager.explore.gui;

import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.table.AbstractTableModel;
import mmcorej.CMMCore;
import org.micromanager.acqj.internal.Engine;
import org.micromanager.explore.ChannelGroupSettings;


/**
 *
 * @author Henry
 */
public class SimpleChannelTableModel extends AbstractTableModel implements TableModelListener {

   private ChannelGroupSettings channels_;
   private final CMMCore core_;
   private boolean selectAll_ = true;
   public static final String[] COLUMN_NAMES = new String[]{
      "Use",
      "Configuration",
      "Exposure",
      "Z-offset (um)" };

   public SimpleChannelTableModel(ChannelGroupSettings channels) {
      core_ = Engine.getCore();
      channels_ = channels;
   }

   public void selectAllChannels() {
      //Alternately select all channels or deselect channels
      channels_.setUseOnAll(selectAll_);
      selectAll_ = !selectAll_;
      fireTableDataChanged();
   }

   public void synchronizeExposures() {
      //Alternately select all channels or deselect channels
      channels_.synchronizeExposures();
      fireTableDataChanged();
   }


   public void setChannelGroup(String group) {
      if (channels_ != null) {
         channels_.updateChannelGroup(group);
      }
   }

   public void setChannels(ChannelGroupSettings channels) {
      channels_ = channels;
   }

   @Override
   public int getRowCount() {
      if (channels_ == null) {
         return 0;
      } else {
         return channels_.getNumChannels();
      }
   }

   @Override
   public int getColumnCount() {
      return COLUMN_NAMES.length ;
   }

   @Override
   public String getColumnName(int columnIndex) {
      return COLUMN_NAMES[columnIndex];
   }

   @Override
   public Object getValueAt(int rowIndex, int columnIndex) {
      //use name exposure, color
      if (columnIndex == 0) {
         return channels_.getChannelListSetting(rowIndex).use_;
      } else if (columnIndex == 1) {
         return channels_.getChannelListSetting(rowIndex).config_;
      } else if (columnIndex == 2) {
         return channels_.getChannelListSetting(rowIndex).exposure_;
      } else if (columnIndex == 3) {
         return channels_.getChannelListSetting(rowIndex).offset_;
      }
      throw new RuntimeException();
   }

   @Override
   public Class getColumnClass(int columnIndex) {
      if (columnIndex == 0) {
         return Boolean.class;
      } else if (columnIndex == 1) {
         return String.class;
      } else if (columnIndex == 2) {
         return Double.class;
      } else if (columnIndex == 3) {
         return Double.class;
      }
      throw new RuntimeException();
   }

   @Override
   public void setValueAt(Object value, int row, int columnIndex) {
      //use name exposure, color  
      int numCamChannels = (int) core_.getNumberOfCameraChannels();

      if (columnIndex == 0) {
         channels_.getChannelListSetting(row).use_ = ((Boolean) value);
         //same for all other channels of the same camera_
         if (numCamChannels > 1) {
            for (int i = (row - row % numCamChannels); i < (row / numCamChannels + 1)
                  * numCamChannels; i++) {
               channels_.getChannelListSetting(i).use_ = ((Boolean) value);
            }
            fireTableDataChanged();
         }
      } else if (columnIndex == 1) {
         //cant edit channel name
      } else if (columnIndex == 2) {
         double val = value instanceof String ? Double.parseDouble((String) value) : (Double) value;
         channels_.getChannelListSetting(row).exposure_ = val;
         //same for all other channels of the same camera_
         if (numCamChannels > 1) {
            for (int i = (row - row % numCamChannels); i < (row / numCamChannels + 1)
                  * numCamChannels; i++) {
               channels_.getChannelListSetting(i).exposure_ = val;
            }
            fireTableDataChanged();
         }
      }
      //Store the newly selected value in preferences
      channels_.storeValuesInPrefs();

   }

   @Override
   public boolean isCellEditable(int nRow, int nCol) {
      return nCol != 1;
   }

   @Override
   public void tableChanged(TableModelEvent e) {
   }

}
