(ns ^:figwheel-hooks ${artifactId}.core
    (:require [cljs-utils.core :as utils :refer [by-id]]
              [cljs-utils.react.hooks :refer [useLens]]
              [cljs.core.match :refer-macros [match]]
              [reitit.frontend :as rf]
              [reitit.frontend.easy :as rfe]
              [${artifactId}.routing-table :as rt]              
              [react :as react]
              ["react-dom/client" :refer [createRoot]]
              [com.stuartsierra.component :as component]
              [cljs.core.async :as async :refer [<!]]
              [system.components.sente :refer [new-channel-socket-client]]
              [taoensso.sente :as sente]
              [goog.dom :as gdom]
              [goog.object :as g]
              [goog.string :as gstring]
              [clojure.string :as str])
    (:require-macros [cljs-utils.compilers.hicada :refer [html]]
                     [cljs.core.async.macros :as a :refer [go-loop]])
    (:import [goog Uri]
             [goog.net XhrIo]
             [goog.ui Zippy]
             [goog.ui Container]
             [goog.ui.Container Orientation]
             [goog.ui Control]
             [goog.ui.Component State]
             [goog.dom TagName]))


;;;

(defn dev-mode? []
  (let [url (.parse Uri js/window.location)
        domain (.getDomain url)]
    (or (= 3125 (.getPort url)) (gstring/startsWith domain "localhost"))))

(if (dev-mode?)
  (enable-console-print!)
  (set! *print-fn*
        (fn [& args]
          (do))))

;;;;; sente

(def sente-client (component/start (new-channel-socket-client js/antiForgeryToken)))
(def ch-chsk       (:ch-chsk sente-client))
(def chsk-send! (:chsk-send! sente-client))
(def chsk-state (:chsk-state sente-client))
(defn fetch [m & fs]
  (if (seq fs)
    (chsk-send! [:websocket/api m] 8000
                (fn [cb-reply]
                  (when (sente/cb-success? cb-reply)
                    (when fs (doseq [f fs] (f cb-reply))))))
    (chsk-send! [:websocket/api m])))
(declare mount)

(defn event-handler [event]
  (match [event]
         [[:chsk/handshake _]] (println "Sente handshake")
         [[:chsk/state [old-state ({:first-open? true} :as new-state) _]]] (do (println "first load" new-state)
                                                                               (mount))
         [[:chsk/state [old-state ({:first-open? false :last-close _} :as new-state) false]]]
         (when (not= (:uid new-state) :sente/nil-uid)
           (println (str "Disconnected. Please sign in again.")))
         [[:chsk/ws-ping]] (println "ping")
         :else (println "Event:" (pr-str event))))

(go-loop [] 
  (let [{:as ev-msg :keys [event]} (<! ch-chsk)]
    (event-handler event)
    (recur)))

;;;;;;

(defn shared []
  (let [[count setCounter] (react/useState 0)]
    (html
     [:*
      [:button {:onClick (fn [_] (fetch {:increment count} #(setCounter (:count %))))} "increment"]
      [:p count]])))

(defn counter-form [props]
  (let [data (js->clj props :keywordize-keys true)]
    (html [:form [:input {:type "number"
                          :name "counter"
                          :onChange (fn [e] (let [val (.-value (.-target e))]
                                             ((:setter data) val)))
                          :value (:count data)}]])))

(defn counter []
  (let [[count setCount] (react/useState 0)]
    (html [:div
           [:p "counter " count]
           [:> counter-form {:count count :setter setCount}]])))

(defn person-form [props]
  (let [{state :state dispatch :dispatch} (js->clj props :keywordize-keys true)]
    (html [:form
           [:input {:type "number"
                    :name "Born"
                    :onChange (fn [e] (let [val (.-value (.-target e))]
                                       (dispatch {:type :born :payload val})))
                    :value (:born state)}]
           [:input {:type "text"
                    :name "name"
                    :placeholder "Jane Doe"
                    :id "name"
                    :onChange (fn [e] (let [val (.-value (.-target e))]
                                       (dispatch {:type :name :payload val})))
                    :value (:user-name state)}]
           [:p 
            [:textarea {:rows 3 :cols 25 :value (:bio state)
                        :onChange (fn [e] (let [val (.-value (.-target e))]
                                       (dispatch {:type :bio :payload val})))}]]])))

(defn person []
  (let [init identity
        [state dispatch] (react/useReducer (fn [state action]
                                             (case (:type action)
                                               :name (assoc state :user-name (:payload action))
                                               :born (assoc state :born (:payload action))
                                               :bio (assoc state :bio (:payload action))
                                               :reset (init (:payload action))))
                                           {:user-name "" :born 1980 :bio ""} init)]
    (react/useEffect (fn [] (fetch {:new-user true} #(dispatch {:type :reset :payload %}))
                       #(println "person component cleaning up")) #js [])
    (html [:div
           [:p (:user-name state) " " (:born state)]
           [:p (:bio state)]
           [:> person-form {:state state :dispatch dispatch}]])))

(defn x [props]
  (let [id (:id (js->clj props :keywordize-keys true))
        [item setItem] (react/useState nil)]
    (react/useEffect (fn [] (fetch {:get-x id} #(setItem %))
                       (fn [] (println "x component cleanup"))) #js [])
    (when item (html [:*
                      [:p (:name item) " " (:quantity item)]
                      [:p (:description item)]]))))

(defn xs []
  (let [[items setItems] (react/useState #js [])]
    (react/useEffect (fn [] (fetch {:get-xs true} #(setItems (:xs %)))
                       (fn [] (println "xs component cleanup"))) #js [])
    (html [:ul (for [item items]
                 [:li {:key (:name item)}
                  [:a {:href (str "/x/" (:id item))} (:name item)]])])))

(defonce app-state (atom {}))

(defn useref []
  (let [input-el (react/useRef nil)]
    (html [:*
           [:input {:type "text" :ref input-el}]
           [:button {:onClick (fn [_] (.focus (.-current input-el)))} "focus"]])))


(defn useref2 []
  (let [v (react/useRef "lisp")]
    (html [:*
           [:p "This demonstrates that mutating a ref doesn't trigger renders."]
           [:p (.-current v)]
           [:form {:onSubmit (fn [e]
                               (.preventDefault e)
                               (println v)
                               (set! (.-current v) "foo"))}
            [:input {:type "submit" :value "Mutate"}]]])))


(defn watcher []
  (let [[height setHeight] (react/useState 0)
        ro (new js/ResizeObserver (fn [entries] (doseq [entry (js->clj entries)
                                                       :let [height (.-height (.-contentRect entry))]]
                                                 (setHeight height))))]
    (js/React.useEffect (fn []
                          (.observe ro (by-id "observee"))
                          #(when (by-id "observee") (.unobserve ro (by-id "observee"))))
                        #js [])
    (html [:p "Observed height is " height])))

(defn child [props]
  (let [measured-ref (:measuredref (js->clj props :keywordize-keys true))
        generate-string #(apply str (take (rand-int 300) (cycle ["a" " " "b"])))
        [demo setDemo] (react/useState (generate-string))
        [show setShow] (react/useState false)]
    (html (if show
            [:div {:id "observee" :ref measured-ref :style #js {:width 300}}
             [:p demo]
             [:button {:onClick (fn [_] (setDemo (generate-string)))} "Change height"]
             [:> watcher {}]]
            [:button {:onClick (fn [_] (setShow true))} "Show child"]))))

(defn useref3 []
  (let [[height setHeight] (react/useState 0)
        measured-ref (react/useCallback (fn [node] (when node
                                                    (setHeight (-> node
                                                                  .getBoundingClientRect
                                                                  .-height
                                                                  Math/round)))) #js [])]
    (html [:*
           [:p "div"]
           [:> child {:measuredref measured-ref}]
           [:h2 "The above div is " height "px height."]])))

(defn use-effect []
  (let [[count setCount] (react/useState 0)]
    (react/useEffect (fn [] (setCount #(inc %))
                       (fn [] (println "use effect component cleanup"))) #js [])
    (html [:*
           [:p "Count: "  count]
           [:p "Passing a function to the setter."]
           [:p "In Dev it is being called twice due to StrictMode."]])))

(defn use-effect2 []
  (let [[count setCount] (react/useState 0)]
    (react/useEffect (fn [] (when (zero? count)
                             (setCount (inc count)))
                       (fn [] (println "use effect2 component cleanup"))) #js [])
    (html [:*
           [:p "Count: "  count]
           [:p "Prevent double calls due to StrictMode."]])))

(defn dialog []
  (let [el (react/useRef nil)]
    (html [:div
           [:dialog {:ref el}
            [:p "hi there"]
            [:button {:onClick #(.close (.-current el))} "close"]]
           [:button {:onClick #(.showModal (.-current el))} "show dialog"]] )))

(defn video-player [props]
  (let [{isPlaying :isPlaying src :src} (js->clj props :keywordize-keys true)
        el (react/useRef nil)]
    (react/useEffect (fn [] (if isPlaying
                             (do (.play (.-current el))
                                (.log js/console "playing"))
                             (do (.pause (.-current el))
                                 (.log js/console "pausing") ))
                       (fn [] (println "use effect2 component cleanup"))) #js [isPlaying])
    (html [:div [:video {:ref el :src src :loop true :playsInline true :width 300}]])))

(defn video []
  (let [[isPlaying setIsPlaying] (react/useState false)
        [text setText] (react/useState "")]
    (html [:div
           [:p "Play with the dependency array in video-player, set it to an empty array, remove it completely, then put \"isPlaying\". Without dependencies, the effect will run at each render (type text in the input to force re-render). With empty dependency, effect will run only once, at mount time."]
           [:input {:type "text" :value text :onChange (fn [e] (setText (.-value (.-target e))))}]
           [:button {:onClick (fn [_] (setIsPlaying (not isPlaying)))} (if isPlaying "Pause" "Play")]
           [:> video-player {:isPlaying isPlaying :src "https://interactive-examples.mdn.mozilla.net/media/cc0-videos/flower.mp4"}]
           [:blockquote "The dependency array can contain multiple dependencies. React will only skip re-running the Effect if all of the dependencies you specify have exactly the same values as they had during the previous render. React compares the dependency values using the Object.is comparison. See the useEffect reference for details."]])))

(defn playground []
  (let [[text setText] (react/useState "a")]
    (react/useEffect (fn [] (let [ontimeout #(.log js/console "⏰ " text)
                                 timeoutid (.setTimeout js/window ontimeout 3000)]
                             (.log js/console "🔵 Schedule \"" text  "\" log")
                             (fn [] (.log js/console "🟡 Cancel \"" text  "\" log")
                               (.clearTimeout js/window timeoutid)))) #js [text])
    (html [:div [:label "what to log" [:input {:value text :onChange (fn [e] (setText (.-value (.-target e))))}]]
           [:h1 text]])))

(defn playground-app []
  (let [[show setShow] (react/useState false)]
    (html [:div
           [:button {:onClick #(setShow (not show))} (str (if show "unmount" "mount") " the component")]
           (when show [:div [:hr]
                       [:> playground {}]])])))

(defn use-ref []
  (let [el (react/useRef nil)]
    (react/useEffect (fn []
                       (when (zero? (.-length (.-children (.-current el))))
                         (let [dom (gdom/createDom TagName/P, nil, "Hi my name is")]
                           (.appendChild (.-current el) dom)))
                       (fn [] (println "use effect3 component cleanup"))) #js [])
    (html [:*
           [:div {:ref el}]])))

(defn use-effect4 []
  (let [el (react/useRef nil)]
    (react/useEffect (fn []
                       (when (zero? (.-length (.-children (.-current el))))
                         (let [header (gdom/createDom TagName/DIV, #js {"style" "background-color:#EEE"}, "Hi my name is")
                               content (gdom/createDom TagName/DIV, nil, "My name is (chka-chka, Slim Shady)")
                               container (gdom/createDom TagName/DIV nil header content)]
                           (Zippy. header content)
                           (.appendChild (.-current el) container)))
                       (fn [] (println "use effect4 component cleanup"))) #js [])
    (html [:*
           [:div {:ref el}]])))

(defn zippy []
  (let [el (react/useRef nil)]
    (react/useEffect (fn []
                       (when (zero? (.-length (.-children (.-current el))))
                         (let [header (gdom/createDom TagName/DIV, #js {"style" "background-color:#EEE"}, "Hi my name is")
                               content (gdom/createDom TagName/DIV, nil, "My name is (chka-chka, Slim Shady)")
                               container (gdom/createDom TagName/DIV nil header content)]
                           (Zippy. header content)
                           (.appendChild (.-current el) container)))
                       (fn [] (println "zippy component cleanup"))) #js [])
    (html [:div {:ref el}])))

(defn challenge1 [props]
  (let [{value :value onChange :onChange} (js->clj props :keywordize-keys true)
        ref (react/useRef nil)]
    (react/useEffect (fn [] (.focus (.-current ref))) #js [])
    (html [:input {:ref ref :value value :onChange onChange}])))


(defn challenge1-app []
  (let [[show setShow] (react/useState false)
        [name setName] (react/useState "Taylor")
        [upper setUpper] (react/useState false)]
    (html [:div [:button {:onClick #(setShow (not show))} (if show "hide" "show")]
           (when show [:div [:> challenge1 {:value name :onChange (fn [e] (setName (.-value (.-target e))))}]
                       [:label "Make it uppercase" [:input {:type "checkbox" :checked upper :onChange (fn [e] (setUpper (.-checked (.-target e))))}]]
                       [:p "Hello " (str (if upper (str/upper-case name) name))]])])))


(defn challenge2 [props]
  (let [{value :value onChange :onChange shouldFocus :shouldFocus} (js->clj props :keywordize-keys true)
        ref (react/useRef nil)]
    (react/useEffect (fn [] (when shouldFocus (.focus (.-current ref)))
                       #(.log js/console "challenge2 cleanup")) #js [shouldFocus])
    (html [:input {:ref ref :value value :onChange onChange}])))

(defn challenge2-app []
  (let [[show setShow] (react/useState false)
        [firstName setFirstName] (react/useState "Taylor")
        [lastName setLastName] (react/useState "Swift")
        name (str firstName " " lastName)]
    (html [:div [:button {:onClick #(setShow (not show))} (if show "hide" "show")]
           (when show [:div 
                       [:label "Enter your first name" [:> challenge2 {:value firstName :shouldFocus true :onChange (fn [e] (setFirstName (.-value (.-target e))))}]]
                       [:label "Enter your last name" [:> challenge2 {:value lastName :shouldFocus false :onChange (fn [e] (setLastName (.-value (.-target e))))}]]
                       [:p "Hello " name]])])))

(defn challenge3 []
  (let [[count setCount] (react/useState 0)]
    (react/useEffect (fn [] (let [onTick #(setCount inc)
                                 interval (.setInterval js/window onTick 1000)]
                             #(do (.clearInterval js/window interval)
                                  (.log js/console "challenge3 cleanup")))) #js [])
    (html [:h1 count])))

(defn challenge3-app []
  (let [[show setShow] (react/useState false)]
    (html [:div [:button {:onClick #(setShow (not show))} (if show "hide" "show")]
           (when show [:> challenge3])])))


(defn challenge4-app []
  (let [[person setPerson] (react/useState "Alice")
        [bio setBio] (react/useState nil)
        fetch-bio (fn [person ignore] (let [delay (if (= person "Bob")
                                                  2000 200)]
                                      (async/go
                                        (async/<! (async/timeout delay))
                                        (when (not @ignore)
                                          (setBio person)))))]
    (react/useEffect (fn []
                       (let [ignore (atom false)]
                         (setBio nil)
                         (fetch-bio person ignore)
                         #(reset! ignore true))) #js [person])
    (html [:div [:select {:value person :onChange (fn [e] (setPerson (.-value (.-target e))))}
                 [:option {:value "Alice"} "Alice"]
                 [:option {:value "Bob"} "Bob"]
                 [:option {:value "Taylor"} "Taylor"]]
           [:hr]
           [:p [:i (if bio bio "Loading...")]]])))

(defn container []
  (let [tags ["Vicky" "Cristina" "Barcelona"]
        nfc (react/useRef nil)
        runonce (react/useRef true)]
    (react/useEffect (fn []
                       (when (.-current runonce)                         
                         (let [container (doto (Container. Orientation/HORIZONTAL)
                                           (.setId "NonFocusableContainer")
                                           (.setFocusable false))]
                           (doseq [tag tags
                                   :let [control (doto (Control. tag)
                                                   (.setId tag)
                                                   (.addClassName "goog-inline-block"))]]
                             (.addChild container control true)
                             (.setSupportedState control State/FOCUSED true))
                           (.render container (.-current nfc))))
                       (fn []
                         (println "Container component cleanup")
                         (set! (.-current runonce) false))) #js [])
    (html [:div {:ref nfc}])))

(defn global []
  (let [saved (useLens app-state :last)
        [name setName] (react/useState "")]
    (html [:*
           [:form {:onSubmit (fn [e]
                               (.preventDefault e)
                               (swap! app-state assoc :last name))}
            [:input {:type "text"
                     :name "name"
                     :placeholder "Jane Doe"
                     :onChange (fn [e] (let [val (.-value (.-target e))]
                                        (setName val)))
                     :value name}]
            [:input {:type "submit" :value "Save"}]]
           [:p "Saved value: " saved]])))

(def themes (clj->js {:light {:foreground "#000000" :background "#eeeeee"}
                      :dark {:foreground "#ffffff" :background "#222222"}}))

(def languages (clj->js {:french {:greeting "Bonjour"}
                         :english {:greeting "Hello"}}))

(def themes-context (react/createContext))

(defn themed-button []
  (let [theme (react/useContext themes-context)]
    (html [:button {:style {:color (g/get theme "foreground") :background (g/get theme "background")}} "button"])))

(defn multilingual-div []
  (let [theme (react/useContext themes-context)]
    (html [:div (.-greeting theme)])))


(defn context []
  (html [:*
         [:p "Context"]
         [:> (.-Provider themes-context) {:value (g/get themes "light")}
          [:> themed-button {}]]
         [:> (.-Provider themes-context) {:value (g/get themes "dark")}
          [:> themed-button {}]]
         [:p "Multilingual"]
         [:> (.-Provider themes-context) {:value (g/get languages "french")}
          [:> multilingual-div {}]]
         [:> (.-Provider themes-context) {:value (g/get languages "english")}
          [:> multilingual-div {}]]]))

(defn file []
  (let [[file setFile] (react/useState nil)
        [status setStatus] (react/useState 0)]
    (html [:*
           [:div
            [:h3 "File upload"]
            (when file
              [:div [:p "Selected file: " (str (.-name file) " " (.-size file) " bytes " (.-type file)  ", last modified " (.-lastModified file))]
               [:button {:onClick (fn [_] (.send XhrIo "/upload"
                                                (fn [e]
                                                  (let [xhr (.-target e)
                                                        resp (.getResponse xhr)]
                                                    (setStatus (.getStatus xhr))
                                                    (println resp (.getStatus xhr))))
                                                "POST"
                                                file
                                                #js {"X-CSRF-Token" js/antiForgeryToken
                                                     "content-type" (.-type file)
                                                     "content-name" (.-name file)}))} "Upload"]])
            [:p "Status: " status]
            [:form [:input {:type "file"
                            :name "file"
                            :onChange (fn [e] (let [files (.-files (.-target e))]
                                               (when (pos? (.-length files))
                                                 (setFile (first files)))))}]]]])))


(defn banner []
  (html [:div
         [:a {:href "/"} "Home | "]
         [:a {:href "/xs"} "Route paths | "]
         [:a {:href "/state"} "Component State | "]         
         [:a {:href "/shared"} "Shared state | "]
         [:a {:href "/useeffect"} "Use effect | "]
         [:a {:href "/challenges"} "Challenges | "]
         [:a {:href "/global"} "Global state | "]
         [:a {:href "/useref"} "Mutable Ref object | "]
         [:a {:href "/context"} "Context | " ]
         [:a {:href "/ui"} "Goog UI | " ]
         [:a {:href "/file"} "File upload"]]))

(declare app-state)
(defn pages [props]
  (println (js->clj props :keywordize-keys true))
  (match [(:view (js->clj props :keywordize-keys true))]
         [:root] (html [:*
                        [:p "Welcome to this demo showcasing Clojurescript interop in React with plain hooks (no wrapper library)."]
                        (when (seq @app-state) [:p "Global state" (pr-str @app-state)])])
         [:global] (html [:*
                          [:p "Global state. You can navigate and come back to find saved value."]
                          [:> global {}]])
         [:shared] (html [:*
                          [:p "Shared state between client and server"]
                          [:> shared {}]]) 
         [:state] (html [:*
                         [:p "Demonstrates useState vs useReducer"]                         
                         [:> counter {}]
                         [:p "useReducer is handy for managing compound objects. In both cases, state is local to the component. If you navigate away, state is reset to default."]
                         [:> person {}]])
         [:xs] (html [:*
                      [:p "Demonstrates how to display a sequence of values with subsequent route path "]
                      [:> xs {}]])
         [:useeffect] (html [:*
                             [:> use-effect]
                             [:> use-effect2]
                             [:> dialog]
                             [:> video]
                             [:> playground-app]])
         [:challenges] (html [:*
                              [:> challenge1-app]
                              [:> challenge2-app]
                              [:> challenge3-app]
                              [:> challenge4-app]])
         [:useref] (html [:*
                          [:> useref]
                          [:> useref2]
                          [:> useref3]])
         [:context] (html [:> context])
         [:ui] (html [:*
                      [:> use-ref]
                      [:> zippy]
                      [:> container]])
         [:file] (html [:> file])
         [{:name :x :id id}] (html [:> x {:id id}])))

(defn router []
  (let [[view setView] (react/useState :root)]
    (react/useEffect (fn []
                       (rfe/start!
                        (rf/router rt/routes)
                        (fn [m]
                          (cond
                            (seq (:path-params m)) (setView (merge (:data m) (:path-params m))) 
                            :else (setView (:name (:data m)))))
                        {:use-fragment false})
                       (fn [] (println "router cleanup"))) #js [])
    (html [:*
           [:> banner]
           [:> pages {:view view}]])))


(defn mount []
  (when-not (some? (:root @app-state))
    (let [tag (by-id "app")
          root (createRoot tag)
          router (html [:> router {}])
          strict-mode (react/createElement react/StrictMode nil router)]
      (if (dev-mode?)
        (do (swap! app-state assoc :root root)
            (.render root strict-mode))
        (.render root router)))))

(defn ^:before-load my-before-reload-callback []
  (println "BEFORE reload!"))

(defn ^:after-load my-after-reload-callback []
  (println "AFTER reload!")
  (.render (:root @app-state) (react/createElement react/StrictMode nil (html [:> router {}]))))





