/*
 * Copyright (c) 2012-2014 Alex de Kruijff
 * Copyright (c) 2014-2015 Specialisterren
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package org.mazarineblue.datasources;

import java.util.HashSet;
import java.util.Set;
import org.mazarineblue.datasources.exceptions.IllegalSourceStateException;

/**
 *
 * @author Alex de Kruijff {@literal <alex.de.kruijff@MazarineBlue.org>}
 */
public class SourceChain
        implements DataSource {

    private DataSource source;
    private SourceChain next;

    public SourceChain() {
    }

    public SourceChain(DataSource source) {
        this.source = source;
    }

    public SourceChain(DataSource source, SourceChain next) {
        this.source = source;
        this.next = next;
    }

    public SourceChain(DataSource... sources) {
        link(sources);
    }

    public SourceChain link(DataSource... sources) {
        SourceChain chain = this;
        for (DataSource source : sources)
            chain = chain.link(source);
        return chain;
    }

    public SourceChain link(DataSource source) {
        if (this.source == null) {
            this.source = source;
            return this;
        }
        return this.next = new SourceChain(source, this.next);
    }

    public SourceChain lastLink() {
        return this.next == null ? this : this.next.lastLink();
    }

    @Override
    public boolean hasNext() {
        boolean found = source.hasNext();
        if (found == true)
            return found;
        return next == null ? false : next.hasNext();
    }

    @Override
    public String next() {
        String first = source.next();
        String second = next == null ? null : next.next();
        return first != null && first.equals("") == false ? first : second;
    }

    @Override
    public String reset() {
        String first = source.reset(), second = null;
        if (next != null)
            second = next.reset();
        if (first == null || first.isEmpty())
            return second;
        return first;
    }

    @Override
    public String getSourceIdentifier() {
        String identifier = source.getSourceIdentifier();
        if (next == null)
            return identifier;
        identifier += ", " + next.getSourceIdentifier();
        return identifier;
    }

    @Override
    public String getLineIdentifier()
            throws IllegalSourceStateException {
        String found = source.getLineIdentifier();
        if (found != null && found.equals("") == false)
            return found;
        return next == null ? null : next.getLineIdentifier();
    }

    @Override
    public Object getData(String column)
            throws IllegalSourceStateException {
        Object obj = source.getData(column);
        if (obj != null)
            return obj;
        return next == null ? null : next.getData(column);
    }

    @Override
    public Object getData(int index)
            throws IllegalSourceStateException {
        Object obj = source.getData(index);
        if (obj != null)
            return obj;
        return next == null ? null : next.getData(index);
    }

    @Override
    public boolean setData(String column, Object value)
            throws Exception {
        if (source.setData(column, value))
            return true;
        return next == null ? false : next.setData(column, value);
    }

    @Override
    public boolean setData(int index, Object value) {
        if (source.setData(index, value))
            return true;
        return next == null ? false : next.setData(index, value);
    }

    @Override
    public String[] getColumns() {
        Set<String> set = new HashSet();
        SourceChain current = this;
        while (current != null) {
            for (String column : source.getColumns())
                set.add(column);
            current = next;
        }
        return set.toArray(new String[0]);
    }
}
