/*
 * Copyright (c) 2012-2014 Alex de Kruijff
 * Copyright (c) 2014-2015 Specialisterren
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package org.mazarineblue.datasources;

import java.util.HashMap;
import java.util.Map;
import org.mazarineblue.datasources.exceptions.IllegalSourceStateException;
import org.mazarineblue.datasources.exceptions.MatrixConfigurationException;

/**
 *
 * @author Alex de Kruijff {@literal <alex.de.kruijff@MazarineBlue.org>}
 */
public class ObjectMatrixSource
        extends MatrixSource {

    private final Object[][] matrix;
    private int row = 0;

    public ObjectMatrixSource(String sourceIdentifier, Object[][] matrix,
                              boolean withHeaders)
            throws MatrixConfigurationException {
        super(sourceIdentifier, initHeaders(matrix, withHeaders));
        this.matrix = initMatrix(matrix, withHeaders);
        this.row = withHeaders ? 1 : 0;
    }

    static private Map<String, Integer> initHeaders(Object[][] matrix,
                                                    boolean withHeaders)
            throws MatrixConfigurationException {
        if (withHeaders == false)
            return null;
        if (matrix.length < 1)
            throw new MatrixConfigurationException(
                    "Matrix must have at least 1 row");
        Map<String, Integer> header = new HashMap<String, Integer>();
        for (int column = 0; column < matrix[0].length; ++column) {
            if (matrix[0][column] instanceof String == false)
                throw new MatrixConfigurationException(
                        "Matrix[0][" + column + "] must be a string");
            header.put((String) matrix[0][column], column);
        }
        return header;
    }

    static private Object[][] initMatrix(Object[][] matrix, boolean withHeaders)
            throws MatrixConfigurationException {
        int n = matrix.length, m = matrix.length == 0 ? 0 : matrix[0].length;
        if (withHeaders)
            --n;
        if (n < 0)
            n = 0;

        for (int row = 1; row < matrix.length; ++row)
            if (matrix[row].length < m)
                throw new MatrixConfigurationException(
                        "Matrix[" + row + "] must have " + m + " columns");

        Object[][] output = new Object[n][m];
        int offset = withHeaders ? 1 : 0;
        for (int row = offset; row < matrix.length; ++row) {
            output[row - offset] = new Object[m];
            for (int column = 0; column < m; ++column)
                output[row - offset][column] = matrix[row][column];
        }
        return output;
    }

    @Override
    public boolean hasNext() {
        return row < matrix.length;
    }

    @Override
    public String next() {
        ++row;
        return getSourceIdentifier() + ":" + row;
    }

    @Override
    public String reset() {
        row = 0;
        return getSourceIdentifier() + ":" + row;
    }

    @Override
    public String getLineIdentifier()
            throws IllegalSourceStateException {
        return getSourceIdentifier() + ":" + row;
    }

    @Override
    public Object getData(int index)
            throws IllegalSourceStateException {
        return matrix[row][index];
    }

    @Override
    public boolean setData(int index, Object value) {
        matrix[row][index] = value;
        return true;
    }
}
