/*
 * Copyright (c) 2012-2014 Alex de Kruijff
 * Copyright (c) 2014-2015 Specialisterren
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package org.mazarineblue.datasources;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.util.HashMap;
import java.util.Map;
import org.mazarineblue.datasources.exceptions.DataValidationException;
import org.mazarineblue.datasources.exceptions.IllegalSourceStateException;

/**
 *
 * @author Alex de Kruijff {@literal <alex.de.kruijff@MazarineBlue.org>}
 */
public class ObjectArraySource
        extends ArraySource {

    private final Map<String, Object> map;

    public ObjectArraySource(String sourceName) {
        this(sourceName, true);
    }

    public ObjectArraySource(String sourceName, DataSource source)
            throws IllegalSourceStateException, DataValidationException {
        this(sourceName, false);
        String mark = source.reset();
        fillArray1(source);
        source.reset();
        fillArray2(source);
        source.reset(mark);
    }

    private void fillArray1(DataSource source)
            throws IllegalSourceStateException, DataValidationException {
        String[] columns = source.getColumns();
        while (source.hasNext()) {
            source.next();
            String firstColumn = source.getData(0, String.class);
            for (String column : columns) {
                Object obj = source.getData(column);
                put(firstColumn + '.' + column, obj);
            }
        }
    }

    private void fillArray2(DataSource source)
            throws IllegalSourceStateException, DataValidationException {
        String[] columns = source.getColumns();
        while (source.hasNext()) {
            source.next();
            String firstColumn = source.getData(0, String.class);
            for (String column : columns) {
                String key = column + '.' + firstColumn;
                Object obj = source.getData(column);
                if (containsKey(key) == false)
                    put(key, obj);
            }
        }
    }

    public ObjectArraySource(String sourceName, boolean initialize) {
        super(sourceName);
        this.map = new HashMap();
        if (initialize)
            next();
    }

    public ObjectArraySource(String sourceName, Map<String, Object> map) {
        this(sourceName, map, false);
    }

    public ObjectArraySource(String sourceName, Map<String, Object> map,
                             boolean initialize) {
        super(sourceName);
        this.map = map;
        if (initialize)
            next();
    }

    public Map<String, Object> fetchMap() {
        Map<String, Object> copy = new HashMap<>();
        copy.putAll(map);
        return copy;
    }

    public void readPlainText(InputStream input)
            throws IOException {
        Reader reader = new InputStreamReader(input);
        BufferedReader convertor = new BufferedReader(reader);
        String str = convertor.readLine();
        while (str != null) {
            String[] arr = str.split(":", 2);
            processPlainText(arr[0], arr[1]);
            str = convertor.readLine();
        }
    }

    private Object processPlainText(String columnKey, Object value) {
        return map.put(columnKey, value);
    }

    public final boolean containsKey(String key) {
        return map.containsKey(key);
    }

    public boolean containsValue(Object value) {
        return map.containsValue(value);
    }

    public final Object put(String columnKey, Object value) {
        return map.put(columnKey, value);
    }

    public void putAll(Map<? extends String, ? extends Object> m) {
        map.putAll(m);
    }

    public Object putIfAbsent(String key, Object value) {
        return map.putIfAbsent(key, value);
    }

    @Override
    public Object getData(String column)
            throws IllegalSourceStateException {
        validateState();
        return map.get(column);
    }

    @Override
    public Object getData(int index) {
        return null;
    }

    @Override
    public boolean setData(String column, Object value) {
        map.put(column, value);
        return true;
    }

    @Override
    public boolean setData(int index, Object value) {
        return false;
    }

    @Override
    public String[] getColumns() {
        return map.keySet().toArray(new String[0]);
    }
}
