/*
 * Copyright (c) 2012-2014 Alex de Kruijff
 * Copyright (c) 2014-2015 Specialisterren
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package org.mazarineblue.datasources;

import org.mazarineblue.datasources.exceptions.DataValidationException;
import org.mazarineblue.datasources.exceptions.IllegalSourceStateException;
import org.mazarineblue.parser.VariableSource;

/**
 * The interface for fetching keys and data from a source.
 *
 * @author Alex de Kruijff {@literal <alex.de.kruijff@MazarineBlue.org>}
 * @see AbstractSource
 * @see ArraySource
 * @see ResourceSource
 * @see ExcelSource
 */
public interface DataSource
        extends VariableSource {

    /**
     * Return true if the data source has more rows.
     *
     * @return true if the data source has more rows.
     */
    public boolean hasNext();

    /**
     * Moves the data source to the next row.
     *
     * @return the line identifier of the next row.
     */
    public String next();

    /**
     * Resets the data source.
     *
     * @return the line identifier from before reset was called or null, if next
     * was not called.
     */
    public String reset();

    default String reset(String mark) {
        String current = reset();
        if (mark == null)
            return current;
        while (mark.equals(current) == false)
            current = next();
        return current;
    }

    /**
     * Returns the source identifier.
     *
     * @return the source identifier
     */
    public String getSourceIdentifier();

    /**
     * Returns the line identifier.
     *
     * @return the line identifier.
     *
     * @throws IllegalSourceStateException when the method could not be called
     * due to the state of the source i.e. next() has not been called or this
     * method is not supported by the data source. when next() has not been
     */
    public String getLineIdentifier()
            throws IllegalSourceStateException;

    /**
     * Return the object at the specified column.
     *
     * @param <T>
     * @param column the location where the object resides.
     * @param objectType the type of the return data
     * @return the object at the specified index or null if it could not be
     * fetched.
     *
     * @throws IllegalSourceStateException when the method could not be called
     * due to the state of the source i.e. next() has not been called or this
     * method is not supported by the data source. when next() has not been
     *
     * @throws DataValidationException when the data object is not of type
     * objectType.
     */
    @SuppressWarnings("unchecked")
    default <T> T getData(String column, Class<T> objectType)
            throws IllegalSourceStateException, DataValidationException {
        Object obj = getData(column);
        if (obj == null)
            return null;
        if (obj.getClass().equals(objectType) == false)
            throw new DataValidationException(
                    "Data is of wrong type at column " + column);
        return (T) obj;
    }

    /**
     * Return the object at the specified column.
     *
     * @param column the location where the object resides.
     * @return the object at the specified index or null if it could not be
     * fetched.
     *
     * @throws IllegalSourceStateException when the method could not be called
     * due to the state of the source i.e. next() has not been called or this
     * method is not supported by the data source. when next() has not been
     */
    @Override
    public Object getData(String column)
            throws IllegalSourceStateException;

    @SuppressWarnings("unchecked")
    default <T> T getData(int index, Class<T> objectType)
            throws IllegalSourceStateException, DataValidationException {
        Object obj = getData(index);
        if (obj == null)
            return null;
        if (obj.getClass().equals(objectType) == false)
            throw new DataValidationException(
                    "Data is of wrong type at index " + index);
        return (T) obj;
    }

    public Object getData(int index)
            throws IllegalSourceStateException;

    /**
     * Return the object at the specified index.
     *
     * @param column the location where the object should be stored.
     * @param value the object to store.
     * @return the object at the specified index or null if it could not be
     * fetched.
     *
     * @throws IllegalSourceStateException when the method could not be called
     * due to the state of the source i.e. next() has not been called or this
     * method is not supported by the data source. when next() has not been
     *
     * @throws DataValidationException when the data object is not of type
     * objectType.
     */
    @Override
    public boolean setData(String column, Object value)
            throws Exception;

    /**
     * Sets the specified value to the specivied column.
     *
     * @param index the location where the object should be stored.
     * @param value the object to store.
     * @return the object at the specified index or null if it could not be
     * fetched.
     */
    public boolean setData(int index, Object value);

    /**
     * Returns an array of supported columns.
     *
     * @return an array of supported columns or an empty array when none are
     * supported by this data source.
     */
    public String[] getColumns();
}
