/*
 * Copyright 2010 by Thomas Mauch
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * $Id: ISvnEntry.java 972 2010-11-22 12:20:16Z magic $
 */
package org.magicwerk.brownies.svn;

import java.util.Map;

import org.magicwerk.brownies.core.files.filemodel.IEntry;


/**
 * Representation of a file or directory in a SVN repository.
 * 
 * @author Thomas Mauch
 * @version $Id: ISvnEntry.java 972 2010-11-22 12:20:16Z magic $
 */
public interface ISvnEntry extends IEntry {
	/**
	 * Gets the revision this entry was fetched.
	 * 
	 * @return the revision
	 */
	public long getRevision();
	
    /**
     * Gets the last changed revision of this entry.
     * 
     * @return 	the revision of this entry when it was last changed 
     */
    public long getCommitRevision();
    
    /**
     * Retrieves the name of the author who last changed this entry.
     * 
     * @return 	the last author's name.
     */
    public String getCommitAuthor();


	/**
	 * Returns properties of this entry.
	 * The property value is stored as byte[].
	 * Note that only user defined properties are returned, but not
	 * SVN system properties.
	 * 
	 * @return user defined properties 
	 */
	public Map<String, Object> getProperties();
	
	/**
	 * Return property as byte array.
	 * 
	 * @param name property name
	 * @return	byte array or null if the property does not exist
	 */
	public byte[] getProperty(String name);
	
	/**
	 * Return property as string by interpreting the byte array as UTF-8 string.
	 * 
	 * @param name property name
	 * @return	String or null if the property does not exist
	 */
	public String getPropertyAsString(String name);
	
	/**
	 * Returns properties of this entry.
	 * The property value is stored as byte[].
	 * Note that only SVN system properties are returned, but not
	 * user defined properties.
	 * 
	 * @return user defined properties 
	 */
	public Map<String, Object> getSystemProperties();

	/**
	 * Return system property as byte array.
	 * System properties are automatically maintained by SVN have a svn: prefix.
	 * 
	 * @param name property name
	 * @return	byte array or null if the property does not exist
	 */
	public byte[] getSystemProperty(String name);
	
	/**
	 * Return system property as string by interpreting the byte array as UTF-8 string.
	 * System properties are automatically maintained by SVN have a svn: prefix.
	 * 
	 * @param name property name
	 * @return	String or null if the property does not exist
	 */
	public String getSystemPropertyAsString(String name);
}
