/*
 * Copyright 2010 by Thomas Mauch
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * $Id$
 */
package org.magicwerk.brownies.svn.wc;

import java.io.File;
import java.util.List;

import org.magicwerk.brownies.collections.GapList;
import org.magicwerk.brownies.collections.IList;
import org.magicwerk.brownies.core.exceptions.WrapperException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.tmatesoft.svn.cli.svn.SVN;
import org.tmatesoft.svn.core.SVNCancelException;
import org.tmatesoft.svn.core.SVNException;
import org.tmatesoft.svn.core.SVNLogEntry;
import org.tmatesoft.svn.core.SVNPropertyValue;
import org.tmatesoft.svn.core.wc.ISVNEventHandler;
import org.tmatesoft.svn.core.wc.ISVNStatusHandler;
import org.tmatesoft.svn.core.wc.SVNClientManager;
import org.tmatesoft.svn.core.wc.SVNEvent;
import org.tmatesoft.svn.core.wc.SVNEventAction;
import org.tmatesoft.svn.core.wc.SVNInfo;
import org.tmatesoft.svn.core.wc.SVNPropertyData;
import org.tmatesoft.svn.core.wc.SVNStatus;
import org.tmatesoft.svn.core.wc.SVNStatusType;

/**
 * Tools for working with Subversion in a working copy.
 *
 * @author Thomas Mauch
 * @version $Id$
 */
public class SvnWcTools {

	static class SvnCli extends SVN {

		boolean success;

		public boolean execute(String... args) {
			super.run(args);
			return success;
		}

		@Override
		public void failure() {
			// Must be overridden as default implementation calls System.exit 
			success = false;
		}

		@Override
		public void success() {
			// Must be overridden as default implementation calls System.exit 
			success = true;
		}
	}

	static final Logger LOG = LoggerFactory.getLogger(SvnWcTools.class);

	//

	/**
	 * Execute a SVN command through a CLI.
	 * 
	 * @param args command line arguments
	 * @return true for successful execution, false for error
	 */
	public static boolean execute(String... args) {
		return new SvnCli().execute(args);
	}

	// --- svn prop get ---

	/**
	 * @param file
	 *            file name
	 * @param property
	 *            property name
	 * @return true if the property is set, false otherwise
	 */
	public static boolean hasProperty(String file, String property) {
		SVNPropertyValue value = getPropertyValue(file, property);
		return value != null;
	}

	/**
	 * @param file
	 *            file name
	 * @param property
	 *            property name
	 * @return string value of the property, null if the property is not set
	 */
	public static String getPropertyAsString(String file, String property) {
		SVNPropertyValue value = getPropertyValue(file, property);
		if (value == null) {
			return null;
		} else {
			return SVNPropertyValue.getPropertyAsString(value);
		}
	}

	/**
	 * @param file
	 *            file name
	 * @param property
	 *            property name
	 * @return SVNPropertyValue if the property is set, null otherwise
	 */
	public static SVNPropertyValue getPropertyValue(String file, String property) {
		SvnPropGetCommand propGet = new SvnPropGetCommand().setFile(file).setPropName(property);
		SVNPropertyData data = propGet.getProperty();
		if (data == null) {
			return null;
		} else {
			return data.getValue();
		}
	}

	// --- svn status ---

	/**
	 * Returns true if the item is known by the SVN working copy.
	 *
	 * @param file 	file path
	 * @return 		true if the item is known by the SVN working copy
	 */
	public static boolean isVersioned(String file) {
		SvnStatusCommand propStatus = new SvnStatusCommand(file);
		SVNStatusType statusType = propStatus.getCommitStatus();
		return statusType != null && statusType != SVNStatusType.STATUS_UNVERSIONED;
	}

	/**
	 * Returns information about log information about all commits.
	 *
	 * @param file	file path
	 * @return		list with log information
	 */
	public static IList<SVNLogEntry> getLogEntries(String file) {
		SvnLogCommand cmd = new SvnLogCommand().setPath(file);
		return cmd.getLogEntries();
	}

	public static SVNLogEntry getLogEntries(String file, long rev) {
		SvnLogCommand cmd = new SvnLogCommand().setPath(file).setRevision("" + rev);
		return cmd.getLogEntries().getSingleOrNull();
	}

	/**
	 * Get SVN info for file.
	 * 
	 * @param file path to file
	 * @return	SVNInfo for file, null if file does not exist or is not under version control
	 */
	public static SVNInfo getInfo(String file) {
		SvnInfoCommand cmd = new SvnInfoCommand().setPath(file);
		return cmd.getInfos().getSingle();
	}

	public static List<SVNInfo> getInfo(String file, boolean recursive) {
		SvnInfoCommand cmd = new SvnInfoCommand().setPath(file).setRecursive(recursive);
		return cmd.getInfos();
	}

	// TODO
	// Note that the handling of non-existing and non-versioned files
	// is not very consistent:
	// - svn status -v file.txt: shows unmodified status
	// - svn status -v unknown.txt: shows unversioned status (?)
	// - svn status -v non-existing.txt: shows no output
	// - svn status -v dir/file.txt: svn: warning: 'dir\file.txt' is not a working copy
	// - svn status -v dir/non-existing.txt: svn: warning: 'dir\file.txt' is not a working copy
	//

	/**
	 * Returns status of item in working copy.
	 *
	 * @param file file path
	 * @return SVNStatus if the file exists, null otherwise
	 */
	public static SVNStatus getStatus(String file) {
		LOG.debug("SvnWcTools.getStatus {}", file);

		SvnStatusCommand statusCmd = new SvnStatusCommand(file);
		return statusCmd.getStatus();
	}

	/**
	 * Returns commit status of item in working copy.
	 * Note that for an unversioned file null, STATUS_UNVERSIONED or STATUS_NONE can be returned.
	 *
	 * @param file file path
	 * @return SVNStatusType if the file exists, null otherwise
	 */
	public static SVNStatusType getStatusType(String file) {
		return getStatusType(getStatus(file));
	}

	public static SVNStatusType getStatusType(SVNStatus status) {
		if (status == null) {
			return null;
		} else {
			return status.getCombinedNodeAndContentsStatus();
		}
	}

	public static long getCommitRevision(SVNStatus status) {
		if (status.isVersioned() && status.getCommittedRevision().isValid()) {
			return status.getCommittedRevision().getNumber();
		} else {
			return -1;
		}
	}

	public static long getRevision(SVNStatus status) {
		if (status.isVersioned() && status.getRevision().isValid()) {
			return status.getRevision().getNumber();
		} else {
			return -1;
		}
	}

	/**
	 * Returns status of items in directory of working copy.
	 *
	 * @param dir		dir path
	 * @param recursive	true to recurse into subdirectories
	 * @return 			list with status information for all retrieved files
	 */
	public static IList<SVNStatus> getSvnStatusList(String dir, boolean recursive) {
		SVNClientManager clientManager = SVNClientManager.newInstance();
		boolean isRemote = false;
		boolean isReportUnmodified = true;
		boolean isIncludeIgnored = false;
		boolean isCollectParentExternals = false;

		StatusHandler statusHandler = new StatusHandler();
		try {
			clientManager.getStatusClient().doStatus(new File(dir), recursive, isRemote, isReportUnmodified,
					isIncludeIgnored, isCollectParentExternals, statusHandler);
		} catch (SVNException e) {
			throw new WrapperException(e);
		}
		return statusHandler.statusList;
	}

	static class StatusHandler implements ISVNStatusHandler, ISVNEventHandler {

		IList<SVNStatus> statusList = GapList.create();

		@Override
		public void handleStatus(SVNStatus status) {
			statusList.add(status);
		}

		/*
		 * This is an implementation for
		 * ISVNEventHandler.handleEvent(SVNEvent event, double progress)
		 */
		@Override
		public void handleEvent(SVNEvent event, double progress) {
			/*
			 * Gets the current action. An action is represented by SVNEventAction.
			 * In case of a status operation a current action can be determined via
			 * SVNEvent.getAction() and SVNEventAction.STATUS_-like constants.
			 */
			SVNEventAction action = event.getAction();
			/*
			 * Print out the revision against which the status was performed. This
			 * event is dispatched when the SVNStatusClient.doStatus() was invoked
			 * with the flag remote set to true - that is for a local status it
			 * won't be dispatched.
			 */
			if (action == SVNEventAction.STATUS_COMPLETED) {
				System.out.println("Status against revision:  " + event.getRevision());
			}

		}

		/*
		 * Should be implemented to check if the current operation is cancelled. If
		 * it is, this method should throw an SVNCancelException.
		 */
		@Override
		public void checkCancelled() throws SVNCancelException {
		}

	}

}
