/*
 * Copyright 2011 by Thomas Mauch
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * $Id$
 */
package org.magicwerk.brownies.svn.wc;

import org.tmatesoft.svn.cli.AbstractSVNCommand;
import org.tmatesoft.svn.cli.svn.SVNStatusCommand;
import org.tmatesoft.svn.core.SVNException;
import org.tmatesoft.svn.core.wc.SVNStatus;
import org.tmatesoft.svn.core.wc.SVNStatusType;

/**
 * Implementation of the SVN command "status".
 *
 * @author Thomas Mauch
 * @version $Id$
 */
public class SvnStatusCommand extends SvnCommand {
	private String file;
	private SVNStatus status;


	/**
	 * Create object to execute command "svn status".
	 *
	 * @param file	file to request status
	 */
	public SvnStatusCommand(String file) {
		this.file = file;
	}

    /**
     * Returns commit status of item in working copy.
     *
     * @return	commit status of item in working copy
     */
    public SVNStatusType getCommitStatus() {
    	return getCommitStatus(getStatus());
    }

    /**
     * Returns SVN status type determined using both contents
     * and properties status.
     * If the status of contents is normal but the properties has been
     * changed, the modified status is returned.
     *
     * @param status 	SVN status
     * @return			determined SVN status type
     */
    public static SVNStatusType getCommitStatus(SVNStatus status) {
    	SVNStatusType statusType = null;
    	if (status != null) {
    		statusType = status.getCombinedNodeAndContentsStatus();
    		if (statusType == SVNStatusType.STATUS_NORMAL) {
    			statusType = status.getPropertiesStatus();
    			if (statusType != SVNStatusType.STATUS_NORMAL) {
    				statusType = SVNStatusType.STATUS_MODIFIED;
    			}
    		}
    	}
		return statusType;
    }

    /**
     * Returns status of item in working copy.
     *
     * @return	status of item in working copy
     */
    public SVNStatus getStatus() {
		String[] args = { "status", "-v", file };
    	run(args);
    	return status;
    }

    @Override
    protected AbstractSVNCommand getCommand() {
        return new SVNStatusCommand() {
			@Override
		    public void handleStatus(SVNStatus status) throws SVNException {
				SvnStatusCommand.this.status = status;
				super.handleStatus(status);
			}
        };
    }
}
