/*
 * Copyright 2011 by Thomas Mauch
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * $Id: SvnPropGetCommand.java 3145 2016-04-13 13:55:20Z origo $
 */
package org.magicwerk.brownies.svn.wc;

import java.util.List;

import org.magicwerk.brownies.collections.GapList;
import org.magicwerk.brownies.collections.IList;
import org.magicwerk.brownies.svn.SvnException;
import org.tmatesoft.svn.cli.AbstractSVNCommand;
import org.tmatesoft.svn.cli.svn.SVNInfoCommand;
import org.tmatesoft.svn.core.SVNException;
import org.tmatesoft.svn.core.wc.SVNInfo;

/**
 * Implementation of the SVN command "info".
 *
 * @author Thomas Mauch
 * @version $Id: SvnPropGetCommand.java 3145 2016-04-13 13:55:20Z origo $
 */
public class SvnInfoCommand extends SvnCommand {
	String file;
	boolean recursive;

	IList<SVNInfo> infos;

	public SvnInfoCommand setPath(String file) {
		this.file = file;
		return this;
	}

	public SvnInfoCommand setRecursive(boolean recursive) {
		this.recursive = recursive;
		return this;
	}

	@Override
	public List<String> buildArgs() {
		IList<String> args = GapList.create();
		// Use --verbose to get affected paths
		args.addArray("info", file);
		if (recursive) {
			args.add("-R");
		}
		return args;
	}

	/**
	 * Returns info data.
	 * 
	 * @return	info data
	 */
	public IList<SVNInfo> getInfos() {
		try {
			infos = GapList.create();
			run();
		} catch (SvnException e) {
			SVNException se = (SVNException) e.getCause();
			int code = se.getErrorMessage().getErrorCode().getCode();
			if (code == 2009) {
				return null;
			}
		}
		return infos;
	}

	@Override
	protected AbstractSVNCommand getCommand() {
		return new SVNInfoCommand() {
			@Override
			protected void printInfo(SVNInfo info) {
				monitor.updated("Getting info for " + getFilePath(info));
				infos.add(info);
			}
		};
	}

	public static String getFilePath(SVNInfo info) {
		return info.getFile().getPath();
	}

	public static String getUrlPath(SVNInfo info) {
		return info.getURL().getPath();
	}

}
