/*
 * Copyright 2011 by Thomas Mauch
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * $Id$
 */
package org.magicwerk.brownies.svn.wc;

import java.io.OutputStream;
import java.io.PrintStream;
import java.util.List;

import org.magicwerk.brownies.collections.GapList;
import org.magicwerk.brownies.core.context.IMonitor;
import org.magicwerk.brownies.core.context.Monitor;
import org.magicwerk.brownies.core.print.PrintTools;
import org.magicwerk.brownies.core.streams.NullPrintStream;
import org.magicwerk.brownies.core.strings.StringPrinter;
import org.magicwerk.brownies.core.strings.escape.StringListEscaper;
import org.magicwerk.brownies.svn.SvnException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.tmatesoft.svn.cli.AbstractSVNCommand;
import org.tmatesoft.svn.cli.AbstractSVNCommandEnvironment;
import org.tmatesoft.svn.cli.svn.SVNCommandEnvironment;
import org.tmatesoft.svn.core.SVNException;

/**
 * Abstract base class for command implementations like SvnStatusCommand.
 *
 * @author Thomas Mauch
 * @version $Id$
 */
public abstract class SvnCommand extends org.tmatesoft.svn.cli.svn.SVN {

	static final Logger LOG = LoggerFactory.getLogger(SvnCommand.class);

	static final StringListEscaper argsEscaper = StringListEscaper.DosListEscaper();

	SVNException exception;

	IMonitor monitor = Monitor.get();

	List<String> args;

	protected void run() {
		List<String> cmdArgs;
		if (args != null) {
			cmdArgs = args;
		} else {
			cmdArgs = buildArgs();
		}
		String[] cmdArgsArray = cmdArgs.toArray(new String[] {});
		run(cmdArgsArray);
	}

	protected List<String> buildArgs() {
		return null;
	}

	public SvnCommand setArgs(String... args) {
		this.args = GapList.create(args);
		return this;
	}

	public SvnCommand setArgs(List<String> args) {
		this.args = GapList.create(args);
		return this;
	}

	public SvnCommand setArgLine(String argLine) {
		this.args = argsEscaper.decode(argLine);
		return this;
	}

	/**
	 * Run SVN command with specified arguments.
	 * 
	 * @param args  arguments used to execute this command
	 */
	@Override
	public void run(String[] args) {
		LOG.debug("Executing {}", PrintTools.toString(args));

		super.run(args);
		if (exception != null) {
			String cmd = new StringPrinter().setElemMarker(" ").addAll(GapList.create(args)).toString();
			String msg = "Command svn " + cmd + " failed";
			throw new SvnException(msg, exception);
		}
	}

	/**
	 * @return SVNCommand needed to execute this command
	 */
	protected abstract AbstractSVNCommand getCommand();

	@Override
	protected void registerCommands() {
		AbstractSVNCommand.registerCommand(getCommand());
	}

	protected OutputStream getOutputStream() {
		return NullPrintStream.INSTANCE;
	}

	@Override
	protected AbstractSVNCommandEnvironment createCommandEnvironment() {
		return new SVNCommandEnvironment(getProgramName(), new PrintStream(getOutputStream()), NullPrintStream.INSTANCE, null) {
			@Override
			public boolean run() {
				getCommand().init(this);
				try {
					getCommand().run();
				} catch (SVNException e) {
					exception = e;
					return false;
				} finally {
					getOut().flush();
					getErr().flush();
				}
				return true;
			}

			@Override
			public void dispose() {
				SvnCommand.this.dispose();
				super.dispose();
			}
		};
	}

	protected void dispose() {
	}

	@Override
	public void success() {
		// Method success() must be overridden as it will call System.exit(0)
	}

	@Override
	public void failure() {
		// Method failure() must be overridden as it will call System.exit(1)
	}

}
