/*
 * Copyright 2011 by Thomas Mauch
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * $Id: SvnLogCommand.java 3204 2016-06-02 18:22:18Z origo $
 */
package org.magicwerk.brownies.svn.wc;

import java.io.FileOutputStream;
import java.io.OutputStream;

import org.magicwerk.brownies.collections.GapList;
import org.magicwerk.brownies.collections.IList;
import org.magicwerk.brownies.core.StreamTools;
import org.magicwerk.brownies.core.exceptions.WrapperException;
import org.tmatesoft.svn.cli.AbstractSVNCommand;
import org.tmatesoft.svn.cli.svn.SVNCatCommand;

/**
 * Implementation of the SVN command "cat".
 *
 * @author Thomas Mauch
 * @version $Id: SvnLogCommand.java 3204 2016-06-02 18:22:18Z origo $
 */
public class SvnCatCommand extends SvnCommand {
	// Config
	private String path;
	private long revision;
	private String target;
	// State
	private FileOutputStream fileOutputStream;

	/**
	 * Create object to execute command "svn cat".
	 *
	 * @param path		path to cat
	 * @param target	target file to receive content of path (null for standard output)
	 */
	public SvnCatCommand(String path, long revision, String target) {
		this.path = path;
		this.revision = revision;
		this.target = target;
	}

	@Override
	protected OutputStream getOutputStream() {
		try {
			if (target != null) {
				fileOutputStream = new FileOutputStream(target);
				return fileOutputStream;
			} else {
				return System.out;
			}
		} catch (Exception e) {
			throw new WrapperException(e);
		}
	}

	@Override
	protected void dispose() {
		StreamTools.close(fileOutputStream);
	}

	/**
	 * Execute SVN cat command.
	 */
	public void execute() {
		IList<String> args = GapList.create();
		args.add("cat");
		if (revision != -1) {
			args.addArray("-r", String.valueOf(revision));
		}
		args.add(path);
		run(args.toArray(new String[] {}));
	}

	@Override
	protected AbstractSVNCommand getCommand() {
		return new SVNCatCommand();
	}
}
