/*
 * Copyright 2010 by Thomas Mauch
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * $Id: SvnEntry.java 1046 2011-01-09 22:10:10Z magic $
 */
package org.magicwerk.brownies.svn;

import java.time.Instant;
import java.time.ZonedDateTime;
import java.util.Map;
import java.util.TreeMap;

import org.magicwerk.brownies.core.CharsetTools;
import org.magicwerk.brownies.core.files.PathTools;
import org.magicwerk.brownies.core.files.filemodel.IDirectory;
import org.magicwerk.brownies.core.files.filemodel.IEntry;
import org.magicwerk.brownies.core.time.TimeTools;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.tmatesoft.svn.core.ISVNDirEntryHandler;
import org.tmatesoft.svn.core.SVNDirEntry;
import org.tmatesoft.svn.core.SVNException;
import org.tmatesoft.svn.core.SVNNodeKind;
import org.tmatesoft.svn.core.SVNProperties;
import org.tmatesoft.svn.core.SVNProperty;
import org.tmatesoft.svn.core.SVNPropertyValue;

/**
 * Representation of a file or directory in a SVN repository.
 * 
 * @author Thomas Mauch
 * @version $Id: SvnEntry.java 1046 2011-01-09 22:10:10Z magic $
 */
public abstract class SvnEntry implements ISvnEntry {
	/** Logger */
	static final Logger LOG = LoggerFactory.getLogger(SvnEntry.class);

	private SvnSession session;
	private String path;
	private long revision;
	private Map<String, Object> properties;
	private Map<String, Object> systemProperties;
	private SVNDirEntry info;

	/**
	 * Protected constructor.
	 */
	protected SvnEntry(SvnSession session, String path, long revision, SVNDirEntry info) {
		this.session = session;
		this.path = path;
		this.revision = revision;
		this.info = info;
	}

	SVNDirEntry getInfo() {
		if (info == null) {
			info = session.readInfo(path, revision);
			SVNNodeKind nodeKind = info.getKind();
			if (getType() == IEntry.Type.FILE && nodeKind != SVNNodeKind.FILE || getType() == IEntry.Type.DIRECTORY && nodeKind != SVNNodeKind.DIR) {
				throw new IllegalStateException();
			}
		}
		return info;
	}

	Map<String, Object> readProperties() {
		if (properties == null) {
			initProperties();
		}
		return properties;
	}

	Map<String, Object> readSystemProperties() {
		if (properties == null) {
			initProperties();
		}
		return systemProperties;
	}

	/**
	 * Initializes properties and systemProperties.
	 */
	void initProperties() {
		try {
			SVNProperties svnProperties = new SVNProperties();
			switch (getType()) {
			case DIRECTORY:
				LOG.debug("SVN: getDir properties ({}, {})", path, revision);
				getSession().getRepository().getDir(path, revision, svnProperties, (ISVNDirEntryHandler) null);
				LOG.debug("SVN: getDir properties done");
				break;
			case FILE:
				LOG.debug("SVN: getFile properties ({}, {})", path, revision);
				getSession().getRepository().getFile(path, revision, svnProperties, null);
				LOG.debug("SVN: getFile properties done");
				break;
			default:
				throw new AssertionError();
			}

			this.properties = new TreeMap<String, Object>();
			this.systemProperties = new TreeMap<String, Object>();
			for (Object obj : svnProperties.asMap().entrySet()) {
				Map.Entry<?, ?> entry = (Map.Entry<?, ?>) obj;
				String name = (String) entry.getKey();
				SVNPropertyValue value = (SVNPropertyValue) entry.getValue();
				if (SVNProperty.isRegularProperty(name)) {
					this.properties.put(name, SVNPropertyValue.getPropertyAsBytes(value));
				} else {
					this.systemProperties.put(name, SVNPropertyValue.getPropertyAsBytes(value));
				}
			}
		} catch (SVNException e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * @return the repository
	 */
	@Override
	public SvnSession getSession() {
		return session;
	}

	/**
	 * The path of a SVN entry always contains the repository root path as prefix.
	 */
	@Override
	public String getPath() {
		// TODO why should 'svn://localhost/Test' be added?
		//return PathTools.getPath(session.getRepositoryRoot(), path); 
		return path;
	}

	@Override
	public long getRevision() {
		return revision;
	}

	@Override
	public long getCommitRevision() {
		return getInfo().getRevision();
	}

	@Override
	public String getCommitAuthor() {
		return getInfo().getAuthor();
	}

	@Override
	public Map<String, Object> getProperties() {
		return readProperties();
	}

	@Override
	public Map<String, Object> getSystemProperties() {
		return readSystemProperties();
	}

	@Override
	public byte[] getProperty(String name) {
		return (byte[]) readProperties().get(name);
	}

	@Override
	public String getPropertyAsString(String name) {
		byte[] data = getProperty(name);
		return CharsetTools.getString(data);
	}

	@Override
	public byte[] getSystemProperty(String name) {
		return (byte[]) readSystemProperties().get(name);
	}

	@Override
	public String getSystemPropertyAsString(String name) {
		byte[] data = getSystemProperty(name);
		if (data == null) {
			// svn:entry:checksum may be null
			// TODO is null for files with length 0?
			return null;
		}
		return CharsetTools.getString(data);
	}

	@Override
	public ZonedDateTime getLastModifiedDate() {
		//svn:entry:committed-date=2009-03-17T21:45:52.625000Z
		String dateStr = getSystemPropertyAsString("svn:entry:committed-date");
		Instant d = Instant.parse(dateStr);
		//return date; TODO

		ZonedDateTime date = TimeTools.toZonedDateTime(getInfo().getDate());
		return date;
	}

	@Override
	public String getName() {
		return PathTools.getName(path);
	}

	@Override
	public IDirectory getParent() {
		// Note that we cannot use path directly as it does not contain
		// the repository root
		String path = getPath();

		String root = session.getRepositoryRoot();
		if (path.equals(root)) {
			return null;
		}
		String parentPath = PathTools.getDir(path);
		return session.getDirectory(parentPath);
	}

	@Override
	public String toString() {
		return getPath() + " [" + getType().toString() + "]";
	}

	//--- Eclipse generated methods ---

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((path == null) ? 0 : path.hashCode());
		result = prime * result + ((session == null) ? 0 : session.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		SvnEntry other = (SvnEntry) obj;
		if (path == null) {
			if (other.path != null)
				return false;
		} else if (!path.equals(other.path))
			return false;
		if (session == null) {
			if (other.session != null)
				return false;
		} else if (!session.equals(other.session))
			return false;
		return true;
	}

}
