/*
 * Copyright 2010 by Thomas Mauch
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * $Id: SvnDirectory.java 1046 2011-01-09 22:10:10Z magic $
 */
package org.magicwerk.brownies.svn;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import org.magicwerk.brownies.core.files.PathTools;
import org.magicwerk.brownies.core.files.filemodel.IEntry;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.tmatesoft.svn.core.SVNDirEntry;
import org.tmatesoft.svn.core.SVNException;
import org.tmatesoft.svn.core.SVNNodeKind;

/**
 * Representation of a directory in a SVN repository.
 * 
 * @author Thomas Mauch
 * @version $Id: SvnDirectory.java 1046 2011-01-09 22:10:10Z magic $
 */
public class SvnDirectory extends SvnEntry implements ISvnDirectory {
	/** Logger */
	private static Logger LOG = LoggerFactory.getLogger(SvnDirectory.class);

	/**
	 * Constructor.
	 * 
	 * @param connection
	 * @param path
	 * @param revision
	 * @param info
	 */
	SvnDirectory(SvnSession connection, String path, long revision, SVNDirEntry info) {
		super(connection, path, revision, info);
	}

	@Override
	public Type getType() {
		return Type.DIRECTORY;
	}

	@Override
	public List<IEntry> list() {
		try {
			/*
			 * Gets the contents of the directory specified by path at the latest
			 * revision (for this purpose -1 is used here as the revision number to
			 * mean HEAD-revision) getDir returns a Collection of SVNDirEntry
			 * elements. SVNDirEntry represents information about the directory
			 * entry. Here this information is used to get the entry name, the name
			 * of the person who last changed this entry, the number of the revision
			 * when it was last changed and the entry type to determine whether it's
			 * a directory or a file. If it's a directory listEntries steps into a
			 * next recursion to display the contents of this directory. The third
			 * parameter of getDir is null and means that a user is not interested
			 * in directory properties. The fourth one is null, too - the user
			 * doesn't provide its own Collection instance and uses the one returned
			 * by getDir.
			 */

			List<IEntry> list = new ArrayList<IEntry>();

			LOG.debug("SVN: getDir entries ({}, {})", getPath(), getRevision());
			Collection<?> entries = getSession().getRepository().getDir(getSession().getLocalPath(getPath()), getRevision(), null, (Collection<?>) null);
			LOG.debug("SVN: getDir entries done");

			Iterator<?> iterator = entries.iterator();
			while (iterator.hasNext()) {
				SVNDirEntry entry = (SVNDirEntry) iterator.next();
				String path = PathTools.getPath(getSession().getLocalPath(getPath()), entry.getName());
				// TODO construction of SvnDirectory and SvnFile should happen
				// with SVNDirEntry which already contains additional information like size 
				if (entry.getKind() == SVNNodeKind.DIR) {
					SvnDirectory dir = new SvnDirectory(getSession(), path, getRevision(), entry);
					list.add(dir);
				} else if (entry.getKind() == SVNNodeKind.FILE) {
					SvnFile file = new SvnFile(getSession(), path, getRevision(), entry);
					list.add(file);
				}
			}
			return list;
		} catch (SVNException e) {
			throw new RuntimeException(e);
		}
	}

}
