/*
 * Copyright 2010 by Thomas Mauch
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * $Id: SvnTools.java 1046 2011-01-09 22:10:10Z magic $
 */
package org.magicwerk.brownies.svn;

import java.io.InputStream;
import java.util.Map;

import org.magicwerk.brownies.core.ObjectTools;
import org.magicwerk.brownies.core.StreamTools;
import org.tmatesoft.svn.core.SVNProperty;

/**
 * Tools for working with Subversion.
 *
 * @author Thomas Mauch
 * @version $Id: SvnTools.java 1046 2011-01-09 22:10:10Z magic $
 */
public class SvnTools {

	/**
	 * Determines whether this is text or binary file.
	 * The decision is based on the property svn:mime-type.
	 *
	 * @param file	file
	 * @return true for text file, false for binary file.
	 */
	public static boolean isTextType(ISvnFile file) {
		String mimeType = (String) file.getSystemProperties().get(SVNProperty.MIME_TYPE);
		boolean isTextType = SVNProperty.isTextMimeType(mimeType);
		return isTextType;
	}

	/**
	 * Compares the content of the two given files for equality.
	 * File name and file properties are ignored.
	 *
	 * @param session	SVN session
	 * @param path1		path of first file to compare
	 * @param rev1		revision of first file to compare
	 * @param path2	`	path of second file to compare
	 * @param rev2		revision of second file to compare
	 * @return			true if the content of the two files is equals,
	 * 					false otherwise
	 */
	public static boolean isContentEqual(SvnSession session, String path1, long rev1, String path2, long rev2) {
		SvnFile file1 = session.getFile(path1, rev1);
		SvnFile file2 = session.getFile(path2, rev2);
		return isContentEqual(file1, file2);
	}

	/**
	 * Compares the content of the two given files for equality of content.
	 * File name and properties are ignored.
	 *
	 * @param file1		first file to compare
	 * @param file2	`	second file to compare
	 * @return			true if the content of the two files is equal,
	 * 					false otherwise
	 */
	public static boolean isContentEqual(ISvnFile file1, ISvnFile file2) {
		InputStream content1 = file1.getContent();
		InputStream content2 = file2.getContent();
		return StreamTools.diff(content1, content2) == 0;
	}

	/**
	 * Compares the content of the two specified entries for equality of properties.
	 * Entry name and content are ignored.
	 *
	 * @param entry1	first entry to compare
	 * @param entry2	second entry to compare
	 * @return			true if the content of the two entries is equal,
	 * 					false otherwise
	 */
	public static boolean isPropertiesEqual(ISvnEntry entry1, ISvnEntry entry2) {
		Map<?, ?> properties1 = entry1.getProperties();
		Map<?, ?> properties2 = entry2.getProperties();
		return ObjectTools.equals(properties1, properties2);
	}

}
