/*
 * Copyright 2010 by Thomas Mauch
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * $Id: SvnSession.java 1046 2011-01-09 22:10:10Z magic $
 */
package org.magicwerk.brownies.svn;

import java.util.ArrayList;
import java.util.List;

import org.magicwerk.brownies.core.StringTools;
import org.magicwerk.brownies.core.files.filemodel.IEntry;
import org.magicwerk.brownies.core.files.filemodel.ISession;
import org.magicwerk.brownies.svn.log.LogEntry;
import org.magicwerk.brownies.svn.log.Path;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.tmatesoft.svn.core.ISVNLogEntryHandler;
import org.tmatesoft.svn.core.SVNAuthenticationException;
import org.tmatesoft.svn.core.SVNDirEntry;
import org.tmatesoft.svn.core.SVNException;
import org.tmatesoft.svn.core.SVNLogEntry;
import org.tmatesoft.svn.core.SVNLogEntryPath;
import org.tmatesoft.svn.core.SVNNodeKind;
import org.tmatesoft.svn.core.SVNURL;
import org.tmatesoft.svn.core.auth.BasicAuthenticationManager;
import org.tmatesoft.svn.core.auth.ISVNAuthenticationManager;
import org.tmatesoft.svn.core.internal.io.dav.DAVRepositoryFactory;
import org.tmatesoft.svn.core.internal.io.fs.FSRepositoryFactory;
import org.tmatesoft.svn.core.internal.io.svn.SVNRepositoryFactoryImpl;
import org.tmatesoft.svn.core.io.SVNRepository;
import org.tmatesoft.svn.core.io.SVNRepositoryFactory;

/**
 * Connection to a Subversion repository.
 *
 * @author Thomas Mauch
 * @version $Id: SvnSession.java 1046 2011-01-09 22:10:10Z magic $
 */
public class SvnSession implements ISession, ISvnSession {
	/** Logger */
	private static Logger LOG = LoggerFactory.getLogger(SvnSession.class);

	static {
		setupLibrary();
	}

	/** Access to SVN repository. */
	private SVNRepository repository;
	/** Path of repository root */
	private String repositoryRoot;
	/** Default revision to use (-1 means HEAD) */
	private long defaultRevision = -1;
	/** Username used for connection */
	private String username;
	/** Password used for connection */
	private String password;

	/**
	 * Connect to a SVN repository.
	 *
	 * @param url		URL of repository (this can be the full path
	 * 					referencing an item under SVN control)
	 * @param username	username to connect
	 * @param password	password to connect
	 * @throws			RuntimeException if connect fails
	 */
	public SvnSession(String url, String username, String password) {
		this.username = username;
		this.password = password;

		try {
			/*
			 * Creates an instance of SVNRepository to work with the repository.
			 * All user's requests to the repository are relative to the
			 * repository location used to create this SVNRepository.
			 * SVNURL is a wrapper for URL strings that refer to repository locations.
			 */
			repository = SVNRepositoryFactory.create(SVNURL.parseURIEncoded(url));
			ISVNAuthenticationManager authManager = new BasicAuthenticationManager(username, password);
			repository.setAuthenticationManager(authManager);
			SVNURL svnUrl = repository.getRepositoryRoot(true);
			repositoryRoot = svnUrl.toDecodedString();
			repository.setLocation(svnUrl, false);
		} catch (SVNAuthenticationException e) {
			throw new SvnAuthException(e);
		} catch (SVNException e) {
			throw new SvnException(e);
		}
	}

	/**
	 * Close connection to SVN repository.
	 */
	public void close() {
		repository.closeSession();
	}

	public String getUsername() {
		return username;
	}

	public String getPassword() {
		return password;
	}

	/**
	 * Connect to a SVN repository and gets stored directory.
	 * The created SvnSession can be accessed by calling getSession()
	 * on the directory.
	 *
	 * @param url		URL of directory in repository
	 * @param username	username to connect
	 * @param password	password to connect
	 * @return			SVN directory
	 * @throws			RuntimeException if connect fails
	 */
	public static SvnDirectory getDirectory(String url, String username, String password) {
		SvnSession conn = new SvnSession(url, username, password);
		String root = conn.getRepositoryRoot();
		String dir = StringTools.removeHeadIf(url, root);
		return conn.getDirectory(dir);
	}

	/**
	 * Connect to a SVN repository and gets stored file.
	 * The created SvnSession can be accessed by calling getSession()
	 * on the file.
	 *
	 * @param url		URL of file in repository
	 * @param username	username to connect
	 * @param password	password to connect
	 * @param revision	revision to retrieve (-1 for HEAD)
	 * @return			SVN file
	 * @throws			RuntimeException if connect fails
	 */
	public static SvnFile getFile(String url, String username, String password, long revision) {
		SvnSession conn = new SvnSession(url, username, password);
		String root = conn.getRepositoryRoot();
		String file = StringTools.removeHeadIf(url, root);
		return conn.getFile(file, revision);
	}

	/**
	 * @return the default revision
	 */
	public long getDefaultRevision() {
		return defaultRevision;
	}

	/**
	 * @param defaultRevision the default revision to set
	 */
	public void setDefaultRevision(long defaultRevision) {
		this.defaultRevision = defaultRevision;
	}

	/**
	 * Returns object for direct access to SVNRepository of SVNKit
	 *
	 * @return	SVNRepository
	 */
	public SVNRepository getRepository() {
		return repository;
	}

	String getLocalPath(String path) {
		if (StringTools.isEmpty(path) || path.startsWith("/")) {
			return path;
		}
		String localPath = StringTools.removeHead(path, repositoryRoot);
		if (localPath == null) {
			throw new IllegalArgumentException("Path " + path + " does not match root " + repositoryRoot);
		}
		return localPath;
	}

	/**
	 * @return root path of repository
	 */
	@Override
	public String getRepositoryRoot() {
		return repositoryRoot;
	}

	@Override
	public long getHeadRevision() {
		try {
			return repository.getLatestRevision();
		} catch (SVNException svne) {
			throw new SvnException(svne);
		}
	}

	@Override
	public SvnFile getFile(String path) {
		return getFile(path, defaultRevision);
	}

	@Override
	public SvnDirectory getDirectory(String path) {
		return getDirectory(path, defaultRevision);
	}

	public boolean isDirectory(String path) {
		return isDirectory(path, defaultRevision);
	}

	public boolean isFile(String path) {
		return isFile(path, defaultRevision);
	}

	/**
	 * Checks whether the given directory exists.
	 *
	 * @param path		path of directory
	 * @param revision	revision of directory (use -1 for head revision)
	 * @return			true if directory exists, false otherwise
	 * @throws			RuntimeException if a SVN error occurs
	 */
	public boolean isDirectory(String path, long revision) {
		path = getLocalPath(path);
		try {
			SVNNodeKind nodeKind = repository.checkPath(path, revision);
			return nodeKind == SVNNodeKind.DIR;
		} catch (SVNException svne) {
			throw new SvnException(svne);
		}
	}

	/**
	 * Checks whether the given file exists.
	 *
	 * @param path		path of file
	 * @param revision	revision of file (use -1 for head revision)
	 * @return			true if file exists, false otherwise
	 * @throws			RuntimeException if a SVN error occurs
	 */
	public boolean isFile(String path, long revision) {
		path = getLocalPath(path);
		try {
			SVNNodeKind nodeKind = repository.checkPath(path, revision);
			return nodeKind == SVNNodeKind.FILE;
		} catch (SVNException svne) {
			throw new SvnException(svne);
		}
	}

	@Override
	public SvnDirectory getRoot() {
		return getRoot(defaultRevision);
	}

	@Override
	public SvnDirectory getRoot(long revision) {
		return getDirectory(repositoryRoot, revision);
	}

	@Override
	public IEntry.Type getType(String path) {
		return getType(path, defaultRevision);
	}

	/**
	 * @param path
	 * @param revision
	 * @return			info or null
	 */
	SVNDirEntry readInfo(String path, long revision) {
		try {
			LOG.debug("SVN: readInfo ({}, {})", path, revision);
			SVNDirEntry info = repository.info(path, revision);
			LOG.debug("SVN: readInfo done");
			return info;
		} catch (NullPointerException e) {
			// NOTE: It can happen a NPE is thrown
			// (erroneous behavior of SVN server?):
			//Exception in thread "main" java.lang.NullPointerException
			//at org.tmatesoft.svn.core.internal.io.dav.DAVRepository.createDirEntry(DAVRepository.java:1282)
			//at org.tmatesoft.svn.core.internal.io.dav.DAVRepository.info(DAVRepository.java:750)
			//at org.magicwerk.brownies.svn.SvnSession.readInfo(SvnSession.java:250)
			//at org.magicwerk.brownies.svn.SvnSession.getType(SvnSession.java:268)
			return null;
		} catch (SVNException e) {
			throw new SvnException(e);
		}
	}

	private SvnFile readFile(String path, long revision, SVNDirEntry info) {
		return new SvnFile(this, path, revision, info);
	}

	private SvnDirectory readDirectory(String path, long revision, SVNDirEntry info) {
		return new SvnDirectory(this, path, revision, info);
	}

	@Override
	public IEntry.Type getType(String path, long revision) {
		path = getLocalPath(path);
		SVNDirEntry info = readInfo(path, revision);
		if (info == null) {
			return null;
		}
		SVNNodeKind nodeKind = info.getKind();
		if (nodeKind == SVNNodeKind.FILE) {
			return IEntry.Type.FILE;
		} else if (nodeKind == SVNNodeKind.DIR) {
			return IEntry.Type.DIRECTORY;
		} else {
			throw new AssertionError();
		}
	}

	@Override
	public SvnEntry getEntry(String path) {
		return getEntry(path, defaultRevision);
	}

	@Override
	public SvnEntry getEntry(String path, long revision) {
		path = getLocalPath(path);
		SVNDirEntry info = readInfo(path, revision);
		if (info == null) {
			throw new RuntimeException("No entry found at " + path + " (revision " + revision + ")");
		}
		SVNNodeKind nodeKind = info.getKind();
		if (nodeKind == SVNNodeKind.DIR) {
			return readDirectory(path, revision, info);
		} else if (nodeKind == SVNNodeKind.FILE) {
			return readFile(path, revision, info);
		} else {
			throw new AssertionError();
		}
	}

	public SvnEntry getEntry(String path, long revision, IEntry.Type type) {
		switch (type) {
		case DIRECTORY:
			return readDirectory(path, revision, null);
		case FILE:
			return readFile(path, revision, null);
		default:
			throw new AssertionError();
		}
	}

	/**
	 * Return SvnFile reference to requested file.
	 *
	 * @param path		Path to file
	 * @param revision	revision number (use -1 for head revision)
	 * @return			reference to requested file
	 * @throws			RuntimeException if no file entry is found
	 */
	@Override
	public SvnFile getFile(String path, long revision) {
		path = getLocalPath(path);
		SVNDirEntry info = readInfo(path, revision);
		if (info == null || info.getKind() != SVNNodeKind.FILE) {
			throw new SvnException("No file found at " + path + " (revision " + revision + ")");
		}
		return readFile(path, revision, info);
	}

	/**
	 * Return SvnDir reference to requested directory.
	 *
	 * @param path		path to directory
	 * @param revision	revision number (use -1 for head revision)
	 * @return			reference to requested directory
	 * @throws			RuntimeException if no directory entry is found
	 */
	@Override
	public SvnDirectory getDirectory(String path, long revision) {
		path = getLocalPath(path);
		SVNDirEntry info = readInfo(path, revision);
		if (info == null || info.getKind() != SVNNodeKind.DIR) {
			throw new SvnException("No directory found at " + path + " (revision " + revision + ")");
		}
		return readDirectory(path, revision, info);
	}

	public LogEntry getLogEntry(long rev) {
		return getLogEntries(rev, rev, null).get(0);
	}

	public LogEntry getLogEntry(long rev, String path) {
		return getLogEntries(rev, rev, path).get(0);
	}

	public List<LogEntry> getLogEntries(long startRev, long endRev) {
		return getLogEntries(startRev, endRev, null);
	}

	public List<LogEntry> getLogEntries(long startRev, long endRev, String path) {
		try {
			SVNURL location = null;
			String locationUri = null;
			if (path != null) {
				location = repository.getLocation();
				locationUri = location.toString();
				location = location.appendPath(path, false);
				repository.setLocation(location, false);
			}

			final List<LogEntry> logEntries = new ArrayList<LogEntry>();
			repository.log(new String[] { "" }, startRev, endRev, true, true, new ISVNLogEntryHandler() {
				@Override
				public void handleLogEntry(SVNLogEntry sle) {
					logEntries.add(toLogEntry(sle));
				}
			});

			if (path != null) {
				location = SVNURL.parseURIEncoded(locationUri);
				repository.setLocation(location, false);
			}
			return logEntries;
		} catch (SVNException e) {
			throw new SvnException(e);
		}
	}

	static LogEntry toLogEntry(SVNLogEntry sle) {
		LogEntry le = new LogEntry();
		le.setRevision((int) sle.getRevision());
		le.setAuthor(sle.getAuthor());
		le.setDate(sle.getDate());
		le.setMessage(sle.getMessage());

		ArrayList<Path> paths = new ArrayList<Path>();
		for (SVNLogEntryPath slep : sle.getChangedPaths().values()) {
			Path path = new Path();
			path.setAction(slep.getType());
			path.setPath(slep.getPath());
			path.setCopyFromPath(slep.getCopyPath());
			path.setCopyFromRev((int) slep.getCopyRevision());
			paths.add(path);
		}
		le.setRevisionPaths(paths);
		return le;
	}

	/**
	 * Initializes the library to work with a repository via
	 * different protocols.
	 */
	private static void setupLibrary() {
		// For using over http:// and https://
		DAVRepositoryFactory.setup();

		// For using over svn:// and svn+xxx://
		SVNRepositoryFactoryImpl.setup();

		// For using over file:///
		FSRepositoryFactory.setup();
	}

}
