/*
 * Decompiled with CFR 0.152.
 */
package org.locationtech.geowave.core.geotime.index.dimension;

import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import java.nio.ByteBuffer;
import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.TimeZone;
import org.locationtech.geowave.core.index.FloatCompareUtils;
import org.locationtech.geowave.core.index.StringUtils;
import org.locationtech.geowave.core.index.VarintUtils;
import org.locationtech.geowave.core.index.dimension.bin.BinRange;
import org.locationtech.geowave.core.index.dimension.bin.BinValue;
import org.locationtech.geowave.core.index.dimension.bin.BinningStrategy;
import org.locationtech.geowave.core.index.sfc.data.NumericData;
import org.locationtech.geowave.core.index.sfc.data.NumericRange;

public class TemporalBinningStrategy
implements BinningStrategy {
    protected static final long MILLIS_PER_DAY = 86400000L;
    private static final NumberFormat TWO_DIGIT_NUMBER = NumberFormat.getIntegerInstance();
    private Unit unit;
    private String timezone;

    public TemporalBinningStrategy() {
        TWO_DIGIT_NUMBER.setMinimumIntegerDigits(2);
        TWO_DIGIT_NUMBER.setMaximumIntegerDigits(2);
    }

    public TemporalBinningStrategy(Unit unit) {
        this(unit, "GMT");
    }

    public TemporalBinningStrategy(Unit unit, String timezone) {
        TWO_DIGIT_NUMBER.setMinimumIntegerDigits(2);
        TWO_DIGIT_NUMBER.setMaximumIntegerDigits(2);
        this.unit = unit;
        this.timezone = timezone;
    }

    public double getBinMin() {
        return 0.0;
    }

    public double getBinMax() {
        return this.getBinSizeMillis() - 1L;
    }

    public BinValue getBinnedValue(double value) {
        Calendar epochCal = Calendar.getInstance(TimeZone.getTimeZone(this.timezone));
        epochCal.setTimeInMillis((long)value);
        this.setToEpoch(epochCal);
        Calendar valueCal = Calendar.getInstance(TimeZone.getTimeZone(this.timezone));
        valueCal.setTimeInMillis((long)value);
        return new BinValue(this.getBinId(valueCal), (double)(valueCal.getTimeInMillis() - epochCal.getTimeInMillis()));
    }

    private long getBinSizeMillis() {
        long binSizeMillis = 86400000L;
        switch (this.unit) {
            case DECADE: {
                binSizeMillis *= 3653L;
                break;
            }
            default: {
                binSizeMillis *= 366L;
                break;
            }
            case MONTH: {
                binSizeMillis *= 31L;
                break;
            }
            case WEEK: {
                binSizeMillis *= 7L;
                break;
            }
            case DAY: {
                break;
            }
            case HOUR: {
                binSizeMillis /= 24L;
                break;
            }
            case MINUTE: {
                binSizeMillis /= 1440L;
            }
        }
        return binSizeMillis;
    }

    @SuppressFBWarnings(value={"SF_SWITCH_FALLTHROUGH", "SF_SWITCH_NO_DEFAULT"}, justification="Fallthrough intentional for time parsing; default case is provided")
    protected void setToEpoch(Calendar value) {
        switch (this.unit) {
            case DECADE: {
                value.set(1, value.get(1) / 10 * 10);
            }
            default: {
                value.set(2, value.getActualMinimum(2));
            }
            case MONTH: {
                value.set(5, value.getActualMinimum(5));
            }
            case DAY: {
                value.set(11, value.getActualMinimum(11));
            }
            case HOUR: {
                value.set(12, value.getActualMinimum(12));
            }
            case MINUTE: {
                value.set(13, value.getActualMinimum(13));
                value.set(14, value.getActualMinimum(14));
                break;
            }
            case WEEK: {
                value.set(7, value.getActualMinimum(7));
                value.set(11, value.getActualMinimum(11));
                value.set(12, value.getActualMinimum(12));
                value.set(13, value.getActualMinimum(13));
                value.set(14, value.getActualMinimum(14));
            }
        }
    }

    public int getFixedBinIdSize() {
        switch (this.unit) {
            default: {
                return 4;
            }
            case MONTH: {
                return 7;
            }
            case WEEK: {
                return 7;
            }
            case DAY: {
                return 10;
            }
            case HOUR: {
                return 13;
            }
            case MINUTE: 
        }
        return 16;
    }

    private byte[] getBinId(Calendar value) {
        switch (this.unit) {
            default: {
                return StringUtils.stringToBinary((String)Integer.toString(value.get(1)));
            }
            case MONTH: {
                return StringUtils.stringToBinary((String)(Integer.toString(value.get(1)) + "_" + TWO_DIGIT_NUMBER.format(value.get(2))));
            }
            case WEEK: {
                return StringUtils.stringToBinary((String)(Integer.toString(value.get(1)) + "_" + TWO_DIGIT_NUMBER.format(value.get(3))));
            }
            case DAY: {
                return StringUtils.stringToBinary((String)(Integer.toString(value.get(1)) + "_" + TWO_DIGIT_NUMBER.format(value.get(2)) + "_" + TWO_DIGIT_NUMBER.format(value.get(5))));
            }
            case HOUR: {
                return StringUtils.stringToBinary((String)(Integer.toString(value.get(1)) + "_" + TWO_DIGIT_NUMBER.format(value.get(2)) + "_" + TWO_DIGIT_NUMBER.format(value.get(5)) + "_" + TWO_DIGIT_NUMBER.format(value.get(11))));
            }
            case MINUTE: 
        }
        return StringUtils.stringToBinary((String)(Integer.toString(value.get(1)) + "_" + TWO_DIGIT_NUMBER.format(value.get(2)) + "_" + TWO_DIGIT_NUMBER.format(value.get(5)) + "_" + TWO_DIGIT_NUMBER.format(value.get(11)) + "_" + TWO_DIGIT_NUMBER.format(value.get(12))));
    }

    @SuppressFBWarnings(value={"SF_SWITCH_FALLTHROUGH", "SF_SWITCH_NO_DEFAULT"}, justification="Fallthrough intentional for time parsing")
    private Calendar getStartEpoch(byte[] binId) {
        String str = StringUtils.stringFromBinary((byte[])binId);
        Calendar cal = Calendar.getInstance(TimeZone.getTimeZone(this.timezone));
        switch (this.unit) {
            case MINUTE: {
                int minute = Integer.parseInt(str.substring(14, 16));
                cal.set(12, minute);
            }
            case HOUR: {
                int hour = Integer.parseInt(str.substring(11, 13));
                cal.set(11, hour);
            }
            case DAY: {
                int day = Integer.parseInt(str.substring(8, 10));
                cal.set(5, day);
            }
            case MONTH: {
                int month = Integer.parseInt(str.substring(5, 7));
                cal.set(2, month);
            }
            default: {
                int year = Integer.parseInt(str.substring(0, 4));
                cal.set(1, year);
                break;
            }
            case DECADE: {
                int decade = Integer.parseInt(str.substring(0, 4));
                decade = decade / 10 * 10;
                cal.set(1, decade);
                break;
            }
            case WEEK: {
                int yr = Integer.parseInt(str.substring(0, 4));
                cal.set(1, yr);
                int weekOfYear = Integer.parseInt(str.substring(5, 7));
                cal.set(3, weekOfYear);
            }
        }
        this.setToEpoch(cal);
        return cal;
    }

    public BinRange[] getNormalizedRanges(NumericData range) {
        long startEpochMillis;
        if (range == null || range.getMax() < range.getMin()) {
            return new BinRange[0];
        }
        Calendar startEpoch = Calendar.getInstance(TimeZone.getTimeZone(this.timezone));
        long binSizeMillis = this.getBinSizeMillis();
        startEpoch.setTimeInMillis((long)range.getMin());
        this.setToEpoch(startEpoch);
        long epochIterator = startEpochMillis = startEpoch.getTimeInMillis();
        ArrayList<BinRange> bins = new ArrayList<BinRange>();
        boolean firstBin = (long)range.getMin() != startEpochMillis;
        boolean lastBin = false;
        do {
            long startMillis;
            boolean fullExtent;
            long endMillis;
            Calendar nextEpochCal = Calendar.getInstance(TimeZone.getTimeZone(this.timezone));
            nextEpochCal.setTimeInMillis(epochIterator + (long)((double)binSizeMillis * 1.5));
            this.setToEpoch(nextEpochCal);
            long nextEpoch = nextEpochCal.getTimeInMillis();
            long maxOfBin = nextEpoch - 1L;
            Calendar cal = Calendar.getInstance(TimeZone.getTimeZone(this.timezone));
            cal.setTimeInMillis(epochIterator);
            if ((long)range.getMax() <= maxOfBin) {
                lastBin = true;
                endMillis = (long)range.getMax();
                fullExtent = FloatCompareUtils.checkDoublesEqual((double)range.getMax(), (double)maxOfBin);
            } else {
                endMillis = maxOfBin;
                boolean bl = fullExtent = !firstBin;
            }
            if (firstBin) {
                startMillis = (long)range.getMin();
                firstBin = false;
            } else {
                startMillis = epochIterator;
            }
            bins.add(new BinRange(this.getBinId(cal), (double)(startMillis - epochIterator), (double)(endMillis - epochIterator), fullExtent));
            epochIterator = nextEpoch;
        } while (!lastBin);
        return bins.toArray(new BinRange[bins.size()]);
    }

    public byte[] toBinary() {
        byte[] timeZone = StringUtils.stringToBinary((String)this.timezone);
        ByteBuffer binary = ByteBuffer.allocate(this.timezone.length() + VarintUtils.signedIntByteLength((int)this.unit.calendarEnum));
        VarintUtils.writeSignedInt((int)this.unit.calendarEnum, (ByteBuffer)binary);
        binary.put(timeZone);
        return binary.array();
    }

    public void fromBinary(byte[] bytes) {
        ByteBuffer buffer = ByteBuffer.wrap(bytes);
        int unitCalendarEnum = VarintUtils.readSignedInt((ByteBuffer)buffer);
        byte[] timeZoneName = new byte[buffer.remaining()];
        buffer.get(timeZoneName);
        this.unit = Unit.getUnit(unitCalendarEnum);
        this.timezone = StringUtils.stringFromBinary((byte[])timeZoneName);
    }

    public int hashCode() {
        int prime = 31;
        int result = 1;
        String className = this.getClass().getName();
        result = 31 * result + (className == null ? 0 : className.hashCode());
        result = 31 * result + (this.timezone == null ? 0 : this.timezone.hashCode());
        result = 31 * result + (this.unit == null ? 0 : this.unit.calendarEnum);
        return result;
    }

    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (this.getClass() != obj.getClass()) {
            return false;
        }
        TemporalBinningStrategy other = (TemporalBinningStrategy)obj;
        if (this.timezone == null ? other.timezone != null : !this.timezone.equals(other.timezone)) {
            return false;
        }
        return !(this.unit == null ? other.unit != null : this.unit.calendarEnum != other.unit.calendarEnum);
    }

    public NumericRange getDenormalizedRanges(BinRange binnedRange) {
        Calendar startofEpoch = this.getStartEpoch(binnedRange.getBinId());
        long startOfEpochMillis = startofEpoch.getTimeInMillis();
        long minMillis = startOfEpochMillis + (long)binnedRange.getNormalizedMin();
        long maxMillis = startOfEpochMillis + (long)binnedRange.getNormalizedMax();
        return new NumericRange((double)minMillis, (double)maxMillis);
    }

    public static enum Unit {
        MINUTE(12),
        HOUR(11),
        DAY(5),
        WEEK(3),
        MONTH(2),
        YEAR(1),
        DECADE(-1);

        private final int calendarEnum;

        private Unit(int calendarEnum) {
            this.calendarEnum = calendarEnum;
        }

        public int toCalendarEnum() {
            return this.calendarEnum;
        }

        public static Unit getUnit(int calendarEnum) {
            for (Unit u : Unit.values()) {
                if (u.calendarEnum != calendarEnum) continue;
                return u;
            }
            throw new IllegalArgumentException("Calendar enum '" + calendarEnum + "' not found as a valid unit ");
        }

        public static Unit fromString(String code) {
            for (Unit output : Unit.values()) {
                if (!output.toString().equalsIgnoreCase(code)) continue;
                return output;
            }
            return null;
        }
    }
}

