/*
 * Decompiled with CFR 0.152.
 */
package org.locationtech.geowave.analytic;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Random;
import java.util.UUID;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.commons.math3.geometry.Vector;
import org.apache.commons.math3.geometry.euclidean.twod.Vector2D;
import org.geotools.feature.simple.SimpleFeatureBuilder;
import org.geotools.feature.simple.SimpleFeatureTypeBuilder;
import org.geotools.referencing.CRS;
import org.locationtech.geowave.adapter.vector.FeatureDataAdapter;
import org.locationtech.geowave.analytic.distance.CoordinateCircleDistanceFn;
import org.locationtech.geowave.analytic.distance.DistanceFn;
import org.locationtech.geowave.core.geotime.index.SpatialDimensionalityTypeProvider;
import org.locationtech.geowave.core.geotime.index.SpatialOptions;
import org.locationtech.geowave.core.store.api.DataStore;
import org.locationtech.geowave.core.store.api.DataTypeAdapter;
import org.locationtech.geowave.core.store.api.Index;
import org.locationtech.geowave.core.store.api.Writer;
import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.GeometryFactory;
import org.locationtech.jts.geom.LineString;
import org.locationtech.jts.geom.Point;
import org.opengis.feature.simple.SimpleFeature;
import org.opengis.feature.type.AttributeDescriptor;
import org.opengis.feature.type.GeometryType;
import org.opengis.referencing.FactoryException;
import org.opengis.referencing.cs.CoordinateSystem;
import org.opengis.referencing.cs.CoordinateSystemAxis;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GeometryDataSetGenerator {
    static final Logger LOGGER = LoggerFactory.getLogger(GeometryDataSetGenerator.class);
    private final Random rand = new Random();
    private final GeometryFactory geoFactory = new GeometryFactory();
    private final DistanceFn<SimpleFeature> distanceFunction;
    private final SimpleFeatureBuilder builder;
    private SimpleFeature minFeature;
    private double[] minAxis;
    private double[] maxAxis;
    private CoordinateSystem coordSystem;
    private boolean includePolygons = true;

    public GeometryDataSetGenerator(DistanceFn<SimpleFeature> distanceFunction, SimpleFeatureBuilder builder) {
        this.distanceFunction = distanceFunction;
        this.builder = builder;
        this.init();
    }

    public boolean isIncludePolygons() {
        return this.includePolygons;
    }

    public void setIncludePolygons(boolean includePolygons) {
        this.includePolygons = includePolygons;
    }

    public SimpleFeature getCorner() {
        return this.minFeature;
    }

    public Geometry getBoundingRegion() {
        int[] adder = new int[]{1, 2, -1, 2};
        int num = 0;
        int addCnt = 0;
        int dims = this.coordSystem.getDimension();
        int coords = (int)Math.pow(dims, 2.0);
        Coordinate[] coordinates = new Coordinate[coords + 1];
        for (int i = 0; i < coords; ++i) {
            coordinates[i] = new Coordinate();
            for (int j = 0; j < dims; ++j) {
                boolean isMin = (num >> j) % 2 == 0;
                coordinates[i].setOrdinate(j, isMin ? this.minAxis[j] : this.maxAxis[j]);
            }
            num += adder[addCnt];
            addCnt = (addCnt + 1) % 4;
        }
        coordinates[coords] = coordinates[0];
        return this.geoFactory.createPolygon(coordinates);
    }

    private double[] createRange(double factor, double[] minAxis, double[] maxAxis) {
        double[] range = new double[minAxis.length];
        for (int i = 0; i < minAxis.length; ++i) {
            range[i] = (maxAxis[i] - minAxis[i]) * factor;
        }
        return range;
    }

    private Pair<double[], double[]> gridCellBounds(double minCenterDistanceFactor, double[] minAxis, double[] maxAxis) {
        double[] range = this.createRange(1.0, minAxis, maxAxis);
        double[] min = new double[range.length];
        double[] max = new double[range.length];
        for (int i = 0; i < range.length; ++i) {
            min[i] = Math.max(minAxis[i] + minCenterDistanceFactor * ((double)this.rand.nextInt(Integer.MAX_VALUE) % (range[i] / minCenterDistanceFactor)), minAxis[i]);
            max[i] = Math.min(min[i] + minCenterDistanceFactor * range[i], maxAxis[i]);
        }
        return Pair.of((Object)min, (Object)max);
    }

    public void writeToGeoWave(DataStore dataStore, List<SimpleFeature> featureData) throws IOException {
        Index index = SpatialDimensionalityTypeProvider.createIndexFromOptions((SpatialOptions)new SpatialOptions());
        FeatureDataAdapter adapter = new FeatureDataAdapter(featureData.get(0).getFeatureType());
        SimpleFeatureBuilder featureBuilder = new SimpleFeatureBuilder(featureData.get(0).getFeatureType());
        LOGGER.info("Writing " + featureData.size() + " records to " + adapter.getFeatureType().getTypeName());
        dataStore.addType((DataTypeAdapter)adapter, new Index[]{index});
        try (Writer writer = dataStore.createWriter(adapter.getTypeName());){
            for (SimpleFeature feature : featureData) {
                writer.write((Object)feature);
                featureBuilder.reset();
            }
        }
    }

    public List<SimpleFeature> generatePointSet(double minCenterDistanceFactor, double outlierFactor, int numberOfCenters, int minSetSize) {
        return this.generatePointSet(minCenterDistanceFactor, outlierFactor, numberOfCenters, minSetSize, this.minAxis, this.maxAxis);
    }

    public List<SimpleFeature> generatePointSet(LineString line, double distanceFactor, int points) {
        ArrayList<SimpleFeature> pointSet = new ArrayList<SimpleFeature>();
        for (Point point : CurvedDensityDataGeneratorTool.generatePoints(line, distanceFactor, points)) {
            pointSet.add(this.createFeatureWithGeometry((Geometry)point));
        }
        return pointSet;
    }

    public List<SimpleFeature> generatePointSet(double minCenterDistanceFactor, double outlierFactor, int numberOfCenters, int minSetSize, double[] minAxis, double[] maxAxis) {
        ArrayList<SimpleFeature> pointSet = new ArrayList<SimpleFeature>();
        ArrayList<double[]> minForCenter = new ArrayList<double[]>();
        ArrayList<double[]> maxForCenter = new ArrayList<double[]>();
        double[] range = this.createRange(minCenterDistanceFactor, minAxis, maxAxis);
        if (numberOfCenters >= minSetSize) {
            LOGGER.error("The number of centers passed much be less than the minimum set size");
            throw new IllegalArgumentException("The number of centers passed much be less than the minimum set size");
        }
        double minDistance = this.computeMinDistance(minCenterDistanceFactor, minAxis, maxAxis);
        while (pointSet.size() < numberOfCenters) {
            Iterator axis = this.gridCellBounds(minCenterDistanceFactor, minAxis, maxAxis);
            SimpleFeature nextFeature = this.createNewFeature((double[])axis.getLeft(), (double[])axis.getRight());
            if (!this.isFarEnough(nextFeature, pointSet, minDistance)) continue;
            pointSet.add(nextFeature);
        }
        for (SimpleFeature center : pointSet) {
            double[] centerMinAxis = new double[this.coordSystem.getDimension()];
            double[] centerMaxAxis = new double[this.coordSystem.getDimension()];
            Geometry geo = (Geometry)center.getDefaultGeometry();
            Coordinate centerCoord = geo.getCentroid().getCoordinate();
            for (int i = 0; i < centerMinAxis.length; ++i) {
                centerMinAxis[i] = centerCoord.getOrdinate(i) - range[i] / 2.0;
                centerMaxAxis[i] = centerCoord.getOrdinate(i) + range[i] / 2.0;
            }
            minForCenter.add(centerMinAxis);
            maxForCenter.add(centerMaxAxis);
        }
        int clusterdItemsCount = (int)Math.ceil((double)minSetSize * (1.0 - outlierFactor));
        while (pointSet.size() < clusterdItemsCount) {
            int centerPos = this.rand.nextInt(Integer.MAX_VALUE) % minForCenter.size();
            pointSet.add(this.createNewFeature((double[])minForCenter.get(centerPos), (double[])maxForCenter.get(centerPos)));
        }
        while (pointSet.size() < minSetSize) {
            pointSet.add(this.createNewFeature(minAxis, maxAxis));
        }
        return pointSet;
    }

    public List<SimpleFeature> addRandomNoisePoints(List<SimpleFeature> pointSet, int minSetSize, double[] minAxis, double[] maxAxis) {
        while (pointSet.size() < minSetSize) {
            pointSet.add(this.createNewFeature(minAxis, maxAxis));
        }
        return pointSet;
    }

    private void init() {
        this.coordSystem = this.builder.getFeatureType().getCoordinateReferenceSystem().getCoordinateSystem();
        this.minAxis = new double[this.coordSystem.getDimension()];
        this.maxAxis = new double[this.coordSystem.getDimension()];
        for (int i = 0; i < this.coordSystem.getDimension(); ++i) {
            CoordinateSystemAxis axis = this.coordSystem.getAxis(i);
            this.minAxis[i] = axis.getMinimumValue();
            this.maxAxis[i] = axis.getMaximumValue();
        }
        int dims = this.coordSystem.getDimension();
        Coordinate coordinate = new Coordinate();
        for (int i = 0; i < dims; ++i) {
            coordinate.setOrdinate(i, this.minAxis[i]);
        }
        this.minFeature = this.createFeatureWithGeometry((Geometry)this.geoFactory.createPoint(coordinate));
    }

    private boolean isFarEnough(SimpleFeature feature, List<SimpleFeature> set, double minDistance) {
        for (SimpleFeature setItem : set) {
            if (!(this.distanceFunction.measure(feature, setItem) < minDistance)) continue;
            return false;
        }
        return true;
    }

    private double computeMinDistance(double minCenterDistanceFactor, double[] minAxis, double[] maxAxis) {
        assert (minCenterDistanceFactor < 0.75);
        int dims = this.coordSystem.getDimension();
        Coordinate coordinate = new Coordinate();
        for (int i = 0; i < dims; ++i) {
            coordinate.setOrdinate(i, minAxis[i]);
        }
        SimpleFeature minFeature = this.createFeatureWithGeometry((Geometry)this.geoFactory.createPoint(coordinate));
        coordinate = new Coordinate();
        for (int i = 0; i < dims; ++i) {
            coordinate.setOrdinate(i, maxAxis[i]);
        }
        SimpleFeature maxFeature = this.createFeatureWithGeometry((Geometry)this.geoFactory.createPoint(coordinate));
        return minCenterDistanceFactor * this.distanceFunction.measure(minFeature, maxFeature);
    }

    private SimpleFeature createNewFeature(double[] minAxis, double[] maxAxis) {
        int dims = this.coordSystem.getDimension();
        int shapeSize = this.includePolygons ? this.rand.nextInt(Integer.MAX_VALUE) % 5 + 1 : 1;
        Coordinate[] shape = new Coordinate[shapeSize > 2 ? shapeSize + 1 : shapeSize];
        double[] constrainedMaxAxis = Arrays.copyOf(maxAxis, maxAxis.length);
        double[] constrainedMinAxis = Arrays.copyOf(minAxis, minAxis.length);
        for (int s = 0; s < shapeSize; ++s) {
            Coordinate coordinate = new Coordinate();
            for (int i = 0; i < dims; ++i) {
                coordinate.setOrdinate(i, constrainedMinAxis[i] + this.rand.nextDouble() * (constrainedMaxAxis[i] - constrainedMinAxis[i]));
            }
            shape[s] = coordinate;
            if (s != 0) continue;
            this.constrain(coordinate, constrainedMaxAxis, constrainedMinAxis);
        }
        if (shapeSize > 2) {
            shape[shapeSize] = shape[0];
            return this.createFeatureWithGeometry(this.geoFactory.createLinearRing(shape).convexHull());
        }
        if (shapeSize == 2) {
            return this.createFeatureWithGeometry((Geometry)this.geoFactory.createLineString(shape));
        }
        return this.createFeatureWithGeometry((Geometry)this.geoFactory.createPoint(shape[0]));
    }

    public GeometryFactory getFactory() {
        return this.geoFactory;
    }

    private void constrain(Coordinate coordinate, double[] constrainedMaxAxis, double[] constrainedMinAxis) {
        for (int i = 0; i < constrainedMaxAxis.length; ++i) {
            double range = (constrainedMaxAxis[i] - constrainedMinAxis[i]) * 0.001;
            constrainedMaxAxis[i] = Math.min(coordinate.getOrdinate(i) + range, constrainedMaxAxis[i]);
            constrainedMinAxis[i] = Math.max(coordinate.getOrdinate(i) - range, constrainedMinAxis[i]);
        }
    }

    private SimpleFeature createFeatureWithGeometry(Geometry geometry) {
        Object[] values = new Object[this.builder.getFeatureType().getAttributeCount()];
        for (int i = 0; i < values.length; ++i) {
            AttributeDescriptor desc = this.builder.getFeatureType().getDescriptor(i);
            if (desc.getType() instanceof GeometryType) {
                values[i] = geometry;
                continue;
            }
            Class binding = desc.getType().getBinding();
            if (!String.class.isAssignableFrom(binding)) continue;
            values[i] = UUID.randomUUID().toString();
        }
        return this.builder.buildFeature(UUID.randomUUID().toString(), values);
    }

    private static SimpleFeatureBuilder getBuilder(String name) throws FactoryException {
        SimpleFeatureTypeBuilder typeBuilder = new SimpleFeatureTypeBuilder();
        typeBuilder.setName(name);
        typeBuilder.setCRS(CRS.decode((String)"EPSG:4326", (boolean)true));
        typeBuilder.add("geom", Geometry.class);
        typeBuilder.add("name", String.class);
        typeBuilder.add("count", Long.class);
        return new SimpleFeatureBuilder(typeBuilder.buildFeatureType());
    }

    public static class CurvedDensityDataGeneratorTool {
        private static final CoordinateCircleDistanceFn DISTANCE_FN = new CoordinateCircleDistanceFn();

        private CurvedDensityDataGeneratorTool() {
        }

        public static final List<Point> generatePoints(LineString line, double distanceFactor, int points) {
            ArrayList<Point> results = new ArrayList<Point>();
            Coordinate lastCoor = null;
            double distanceTotal = 0.0;
            double[] distancesBetweenCoords = new double[line.getCoordinates().length - 1];
            int i = 0;
            for (Coordinate coor : line.getCoordinates()) {
                if (lastCoor != null) {
                    distancesBetweenCoords[i] = Math.abs(DISTANCE_FN.measure(lastCoor, coor));
                    distanceTotal += distancesBetweenCoords[i++];
                }
                lastCoor = coor;
            }
            lastCoor = null;
            i = 0;
            for (Coordinate coor : line.getCoordinates()) {
                if (lastCoor != null) {
                    results.addAll(CurvedDensityDataGeneratorTool.generatePoints(line.getFactory(), CurvedDensityDataGeneratorTool.toVec(coor), CurvedDensityDataGeneratorTool.toVec(lastCoor), distanceFactor, (int)((double)points * (distancesBetweenCoords[i++] / distanceTotal))));
                }
                lastCoor = coor;
            }
            return results;
        }

        private static final List<Point> generatePoints(GeometryFactory factory, Vector2D coordinateOne, Vector2D coordinateTwo, double distanceFactor, int points) {
            ArrayList<Point> results = new ArrayList<Point>();
            Random rand = new Random();
            Vector2D originVec = coordinateTwo.subtract((Vector)coordinateOne);
            for (int i = 0; i < points; ++i) {
                double factor = rand.nextDouble();
                Vector2D projectionPoint = originVec.scalarMultiply(factor);
                double direction = rand.nextGaussian() * distanceFactor;
                Vector2D orthogonal = new Vector2D(originVec.getY(), -originVec.getX());
                results.add(factory.createPoint(CurvedDensityDataGeneratorTool.toCoordinate(orthogonal.scalarMultiply(direction).add((Vector)projectionPoint).add((Vector)coordinateOne))));
            }
            return results;
        }

        public static Coordinate toCoordinate(Vector2D vec) {
            return new Coordinate(vec.getX(), vec.getY());
        }

        public static Vector2D toVec(Coordinate coor) {
            return new Vector2D(coor.x, coor.y);
        }
    }
}

