package org.liveontologies.protege.explanation.justification;

/*-
 * #%L
 * Protege Justification Explanation
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2016 - 2017 Live Ontologies Project
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */

import static java.util.stream.Collectors.toList;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicLong;

import org.semanticweb.owlapi.model.AddAxiom;
import org.semanticweb.owlapi.model.AxiomType;
import org.semanticweb.owlapi.model.IRI;
import org.semanticweb.owlapi.model.OWLAnnotationAssertionAxiom;
import org.semanticweb.owlapi.model.OWLAnnotationPropertyDomainAxiom;
import org.semanticweb.owlapi.model.OWLAnnotationPropertyRangeAxiom;
import org.semanticweb.owlapi.model.OWLAsymmetricObjectPropertyAxiom;
import org.semanticweb.owlapi.model.OWLAxiom;
import org.semanticweb.owlapi.model.OWLAxiomVisitor;
import org.semanticweb.owlapi.model.OWLClass;
import org.semanticweb.owlapi.model.OWLClassAssertionAxiom;
import org.semanticweb.owlapi.model.OWLClassExpression;
import org.semanticweb.owlapi.model.OWLDataPropertyAssertionAxiom;
import org.semanticweb.owlapi.model.OWLDataPropertyDomainAxiom;
import org.semanticweb.owlapi.model.OWLDataPropertyExpression;
import org.semanticweb.owlapi.model.OWLDataPropertyRangeAxiom;
import org.semanticweb.owlapi.model.OWLDatatypeDefinitionAxiom;
import org.semanticweb.owlapi.model.OWLDeclarationAxiom;
import org.semanticweb.owlapi.model.OWLDifferentIndividualsAxiom;
import org.semanticweb.owlapi.model.OWLDisjointClassesAxiom;
import org.semanticweb.owlapi.model.OWLDisjointDataPropertiesAxiom;
import org.semanticweb.owlapi.model.OWLDisjointObjectPropertiesAxiom;
import org.semanticweb.owlapi.model.OWLDisjointUnionAxiom;
import org.semanticweb.owlapi.model.OWLEntity;
import org.semanticweb.owlapi.model.OWLEquivalentClassesAxiom;
import org.semanticweb.owlapi.model.OWLEquivalentDataPropertiesAxiom;
import org.semanticweb.owlapi.model.OWLEquivalentObjectPropertiesAxiom;
import org.semanticweb.owlapi.model.OWLFunctionalDataPropertyAxiom;
import org.semanticweb.owlapi.model.OWLFunctionalObjectPropertyAxiom;
import org.semanticweb.owlapi.model.OWLHasKeyAxiom;
import org.semanticweb.owlapi.model.OWLImportsDeclaration;
import org.semanticweb.owlapi.model.OWLIndividual;
import org.semanticweb.owlapi.model.OWLInverseFunctionalObjectPropertyAxiom;
import org.semanticweb.owlapi.model.OWLInverseObjectPropertiesAxiom;
import org.semanticweb.owlapi.model.OWLIrreflexiveObjectPropertyAxiom;
import org.semanticweb.owlapi.model.OWLNegativeDataPropertyAssertionAxiom;
import org.semanticweb.owlapi.model.OWLNegativeObjectPropertyAssertionAxiom;
import org.semanticweb.owlapi.model.OWLObject;
import org.semanticweb.owlapi.model.OWLObjectPropertyAssertionAxiom;
import org.semanticweb.owlapi.model.OWLObjectPropertyDomainAxiom;
import org.semanticweb.owlapi.model.OWLObjectPropertyExpression;
import org.semanticweb.owlapi.model.OWLObjectPropertyRangeAxiom;
import org.semanticweb.owlapi.model.OWLOntology;
import org.semanticweb.owlapi.model.OWLOntologyChange;
import org.semanticweb.owlapi.model.OWLOntologyCreationException;
import org.semanticweb.owlapi.model.OWLOntologyManager;
import org.semanticweb.owlapi.model.OWLProperty;
import org.semanticweb.owlapi.model.OWLPropertyAxiom;
import org.semanticweb.owlapi.model.OWLReflexiveObjectPropertyAxiom;
import org.semanticweb.owlapi.model.OWLRuntimeException;
import org.semanticweb.owlapi.model.OWLSameIndividualAxiom;
import org.semanticweb.owlapi.model.OWLSubAnnotationPropertyOfAxiom;
import org.semanticweb.owlapi.model.OWLSubClassOfAxiom;
import org.semanticweb.owlapi.model.OWLSubDataPropertyOfAxiom;
import org.semanticweb.owlapi.model.OWLSubObjectPropertyOfAxiom;
import org.semanticweb.owlapi.model.OWLSubPropertyChainOfAxiom;
import org.semanticweb.owlapi.model.OWLSymmetricObjectPropertyAxiom;
import org.semanticweb.owlapi.model.OWLTransitiveObjectPropertyAxiom;
import org.semanticweb.owlapi.model.SWRLRule;

import uk.ac.manchester.cs.owl.explanation.ordering.EntailedAxiomTree;
import uk.ac.manchester.cs.owl.explanation.ordering.ExplanationOrderer;
import uk.ac.manchester.cs.owl.explanation.ordering.ExplanationTree;
import uk.ac.manchester.cs.owl.explanation.ordering.Tree;

/**
 * Provides ordering and indenting of explanations based on various ordering
 * heuristics.
 *
 * @author Matthew Horridge, The University Of Manchester, Bio-Health
 *         Informatics Group, Date: 11-Jan-2008
 */

public class ProtegeExplanationOrderer implements ExplanationOrderer {

	private Set<OWLAxiom> currentExplanation_;
	private final Map<OWLEntity, Set<OWLAxiom>> lhs2AxiomMap_;
	private final Map<OWLAxiom, Set<OWLEntity>> entitiesByAxiomRHS_;
	private final SeedExtractor seedExtractor_;
	private final OWLOntologyManager man_;
	private OWLOntology ont_;
	private final Map<OWLObject, Set<OWLAxiom>> mappedAxioms_;
	private final Set<OWLAxiom> consumedAxioms_;
	private final Set<AxiomType<?>> passTypes_;
	private OWLEntity currentSource_;
	private OWLEntity currentTarget_;

	/**
	 * @param m
	 *            the manager to use
	 */
	public ProtegeExplanationOrderer(OWLOntologyManager m) {
		currentExplanation_ = Collections.emptySet();
		lhs2AxiomMap_ = new HashMap<>();
		entitiesByAxiomRHS_ = new HashMap<>();
		seedExtractor_ = new SeedExtractor();
		man_ = m;
		mappedAxioms_ = new HashMap<>();
		passTypes_ = new HashSet<>();
		// I'm not sure what to do with disjoint classes yet. At the
		// moment, we just shove them at the end at the top level.
		passTypes_.add(AxiomType.DISJOINT_CLASSES);
		consumedAxioms_ = new HashSet<>();
	}

	private void reset() {
		lhs2AxiomMap_.clear();
		entitiesByAxiomRHS_.clear();
		consumedAxioms_.clear();
	}

	@SuppressWarnings("deprecation")
	@Override
	public ExplanationTree getOrderedExplanation(OWLAxiom entailment,
			Set<OWLAxiom> axioms) {
		currentExplanation_ = new HashSet<>(axioms);
		buildIndices();
		ExplanationTree root = new EntailedAxiomTree(entailment);
		currentSource_ = seedExtractor_.getSource(entailment);
		insertChildren(currentSource_, root);
		currentTarget_ = seedExtractor_.getTarget(entailment);
		Set<OWLAxiom> axs = root.getUserObjectClosure();
		final Set<OWLAxiom> targetAxioms = new HashSet<>();
		if (currentTarget_ != null) {
			if (currentTarget_.isOWLClass()) {
				targetAxioms
						.addAll(ont_.getAxioms(currentTarget_.asOWLClass()));
			}
			if (currentTarget_.isOWLObjectProperty()) {
				targetAxioms.addAll(
						ont_.getAxioms(currentTarget_.asOWLObjectProperty()));
			}
			if (currentTarget_.isOWLDataProperty()) {
				targetAxioms.addAll(
						ont_.getAxioms(currentTarget_.asOWLDataProperty()));
			}
			if (currentTarget_.isOWLNamedIndividual()) {
				targetAxioms.addAll(
						ont_.getAxioms(currentTarget_.asOWLNamedIndividual()));
			}
		}
		List<OWLAxiom> rootAxioms = axioms.stream()
				.filter(ax -> !axs.contains(ax)).collect(toList());
		Collections.sort(rootAxioms, new Comparator<OWLAxiom>() {

			@Override
			public int compare(OWLAxiom o1, OWLAxiom o2) {
				if (targetAxioms.contains(o1)) {
					return 1;
				}
				if (targetAxioms.contains(o2)) {
					return -1;
				}
				return 0;
			}
		});
		for (OWLAxiom ax : rootAxioms) {
			root.addChild(new ExplanationTree(ax));
		}
		return root;
	}

	private List<OWLEntity> getRHSEntitiesSorted(OWLAxiom ax) {
		Collection<OWLEntity> entities = getRHSEntities(ax);
		List<OWLEntity> sortedEntities = new ArrayList<>(entities);
		Collections.sort(sortedEntities, propertiesFirstComparator);
		return sortedEntities;
	}

	@SuppressWarnings("deprecation")
	private void insertChildren(OWLEntity entity, ExplanationTree tree) {
		Set<OWLAxiom> currentPath = new HashSet<>(
				tree.getUserObjectPathToRoot());
		Set<? extends OWLAxiom> axioms = Collections.emptySet();
		if (entity != null) {
			if (entity.isOWLClass()) {
				axioms = ont_.getAxioms(entity.asOWLClass());
			} else if (entity.isOWLObjectProperty()) {
				axioms = ont_.getAxioms(entity.asOWLObjectProperty());
			} else if (entity.isOWLDataProperty()) {
				axioms = ont_.getAxioms(entity.asOWLDataProperty());
			} else if (entity.isOWLNamedIndividual()) {
				axioms = ont_.getAxioms(entity.asOWLNamedIndividual());
			}
			for (OWLAxiom ax : axioms) {
				if (passTypes_.contains(ax.getAxiomType())) {
					continue;
				}
				Set<OWLAxiom> mapped = getIndexedSet(entity, mappedAxioms_,
						true);
				if (consumedAxioms_.contains(ax) || mapped.contains(ax)
						|| currentPath.contains(ax)) {
					continue;
				}
				mapped.add(ax);
				consumedAxioms_.add(ax);
				ExplanationTree child = new ExplanationTree(ax);
				tree.addChild(child);
				for (OWLEntity ent : getRHSEntitiesSorted(ax)) {
					insertChildren(ent, child);
				}
			}
		}
		sortChildrenAxioms(tree);
	}

	private static Comparator<Tree<OWLAxiom>> comparator = new OWLAxiomTreeComparator();

	private void sortChildrenAxioms(ExplanationTree tree) {
		tree.sortChildren(comparator);
	}

	private static AtomicLong randomstart = new AtomicLong(
			System.currentTimeMillis());

	private void buildIndices() {
		reset();
		AxiomMapBuilder builder = new AxiomMapBuilder();
		for (OWLAxiom ax : currentExplanation_) {
			ax.accept(builder);
		}
		try {
			if (ont_ != null) {
				man_.removeOntology(ont_);
			}
			ont_ = man_.createOntology(IRI.create("http://www.semanticweb.org/",
					"ontology" + randomstart.incrementAndGet()));
			List<OWLOntologyChange> changes = new ArrayList<>();
			for (OWLAxiom ax : currentExplanation_) {
				changes.add(new AddAxiom(ont_, ax));
				ax.accept(builder);
			}
			man_.applyChanges(changes);
		} catch (OWLOntologyCreationException e) {
			throw new OWLRuntimeException(e);
		}
	}

	/**
	 * A utility method that obtains a set of axioms that are indexed by some
	 * object.
	 *
	 * @param obj
	 *            The object that indexed the axioms
	 * @param map
	 *            The map that provides the index structure
	 * @param addIfEmpty
	 *            A flag that indicates whether an empty set of axiom should be
	 *            added to the index if there is not value present for the
	 *            indexing object.
	 * @return A set of axioms (may be empty)
	 */
	private static <K, E> Set<E> getIndexedSet(K obj, Map<K, Set<E>> map,
			boolean addIfEmpty) {
		Set<E> values = map.get(obj);
		if (values == null) {
			values = new HashSet<>();
			if (addIfEmpty) {
				map.put(obj, values);
			}
		}
		return values;
	}

	/**
	 * Gets axioms that have a LHS corresponding to the specified entity.
	 *
	 * @param lhs
	 *            The entity that occurs on the left hand side of the axiom.
	 * @return A set of axioms that have the specified entity as their left hand
	 *         side.
	 */
	protected Set<OWLAxiom> getAxiomsForLHS(OWLEntity lhs) {
		return getIndexedSet(lhs, lhs2AxiomMap_, true);
	}

	private Collection<OWLEntity> getRHSEntities(OWLAxiom axiom) {
		return getIndexedSet(axiom, entitiesByAxiomRHS_, true);
	}

	protected void indexAxiomsByRHSEntities(OWLObject rhs, OWLAxiom axiom) {
		getIndexedSet(axiom, entitiesByAxiomRHS_, true)
				.addAll(rhs.getSignature());
	}

	private static final class PropertiesFirstComparator
			implements Comparator<OWLObject> {

		public PropertiesFirstComparator() {
		}

		@Override
		public int compare(OWLObject o1, OWLObject o2) {
			if (o1 instanceof OWLProperty) {
				return -1;
			} else {
				if (o1.equals(o2)) {
					return 0;
				}
				return 1;
			}
		}
	}

	private static PropertiesFirstComparator propertiesFirstComparator = new PropertiesFirstComparator();

	// ///////////////////////////////////////////////////////////////////////////////////////////////////////////
	// ///////////////////////////////////////////////////////////////////////////////////////////////////////////
	/** tree comparator. */
	private static final class OWLAxiomTreeComparator
			implements Comparator<Tree<OWLAxiom>>, Serializable {

		private static final long serialVersionUID = 30406L;

		public OWLAxiomTreeComparator() {
		}

		@Override
		public int compare(Tree<OWLAxiom> o1, Tree<OWLAxiom> o2) {
			OWLAxiom ax1 = o1.getUserObject();
			OWLAxiom ax2 = o2.getUserObject();
			// Equivalent classes axioms always come last
			if (ax1 instanceof OWLEquivalentClassesAxiom) {
				return 1;
			}
			if (ax2 instanceof OWLEquivalentClassesAxiom) {
				return -1;
			}
			if (ax1 instanceof OWLPropertyAxiom) {
				return -1;
			}
			int childCount1 = o1.getChildCount();
			childCount1 = childCount1 > 0 ? 0 : 1;
			int childCount2 = o2.getChildCount();
			childCount2 = childCount2 > 0 ? 0 : 1;
			int diff = childCount1 - childCount2;
			if (diff != 0) {
				return diff;
			}
			if (ax1 instanceof OWLSubClassOfAxiom
					&& ax2 instanceof OWLSubClassOfAxiom) {
				OWLSubClassOfAxiom sc1 = (OWLSubClassOfAxiom) ax1;
				OWLSubClassOfAxiom sc2 = (OWLSubClassOfAxiom) ax2;
				return sc1.getSuperClass().compareTo(sc2.getSuperClass());
			}
			return 1;
		}
	}

	@SuppressWarnings("unused")
	private static class SeedExtractor implements OWLAxiomVisitor {

		private OWLEntity source;
		private OWLEntity target;

		public SeedExtractor() {
		}

		public OWLEntity getSource(OWLAxiom axiom) {
			axiom.accept(this);
			return source;
		}

		public OWLEntity getTarget(OWLAxiom axiom) {
			axiom.accept(this);
			return target;
		}

		@Override
		public void visit(OWLSubClassOfAxiom axiom) {
			if (!axiom.getSubClass().isAnonymous()) {
				source = axiom.getSubClass().asOWLClass();
			}
			if (!axiom.getSuperClass().isOWLNothing()) {
				OWLClassExpression classExpression = axiom.getSuperClass();
				if (!classExpression.isAnonymous()) {
					target = classExpression.asOWLClass();
				}
			}
		}

		@Override
		public void visit(OWLNegativeObjectPropertyAssertionAxiom axiom) {
		}

		@Override
		public void visit(OWLAsymmetricObjectPropertyAxiom axiom) {
		}

		@Override
		public void visit(OWLReflexiveObjectPropertyAxiom axiom) {
		}

		@Override
		public void visit(OWLDisjointClassesAxiom axiom) {
			for (OWLClassExpression ce : axiom.getClassExpressions()) {
				if (!ce.isAnonymous()) {
					if (source == null) {
						source = ce.asOWLClass();
					} else if (target == null) {
						target = ce.asOWLClass();
					} else {
						break;
					}
				}
			}
		}

		@Override
		public void visit(OWLDataPropertyDomainAxiom axiom) {
		}

		public void visit(OWLImportsDeclaration axiom) {
		}

		@Override
		public void visit(OWLObjectPropertyDomainAxiom axiom) {
		}

		@Override
		public void visit(OWLEquivalentObjectPropertiesAxiom axiom) {
		}

		@Override
		public void visit(OWLNegativeDataPropertyAssertionAxiom axiom) {
		}

		@Override
		public void visit(OWLDifferentIndividualsAxiom axiom) {
		}

		@Override
		public void visit(OWLDisjointDataPropertiesAxiom axiom) {
		}

		@Override
		public void visit(OWLDisjointObjectPropertiesAxiom axiom) {
		}

		@Override
		public void visit(OWLObjectPropertyRangeAxiom axiom) {
		}

		@Override
		public void visit(OWLObjectPropertyAssertionAxiom axiom) {
		}

		@Override
		public void visit(OWLFunctionalObjectPropertyAxiom axiom) {
		}

		@Override
		public void visit(OWLSubObjectPropertyOfAxiom axiom) {
			if (!axiom.getSubProperty().isAnonymous()) {
				source = axiom.getSubProperty().asOWLObjectProperty();
			}
			if (!axiom.getSuperProperty().isAnonymous()) {
				target = axiom.getSuperProperty().asOWLObjectProperty();
			}
		}

		@Override
		public void visit(OWLDisjointUnionAxiom axiom) {
		}

		@Override
		public void visit(OWLDeclarationAxiom axiom) {
		}

		@Override
		public void visit(OWLAnnotationAssertionAxiom axiom) {
		}

		@Override
		public void visit(OWLSymmetricObjectPropertyAxiom axiom) {
		}

		@Override
		public void visit(OWLDataPropertyRangeAxiom axiom) {
		}

		@Override
		public void visit(OWLFunctionalDataPropertyAxiom axiom) {
		}

		@Override
		public void visit(OWLEquivalentDataPropertiesAxiom axiom) {
		}

		@Override
		public void visit(OWLClassAssertionAxiom axiom) {
			if (!axiom.getClassExpression().isAnonymous()) {
				source = axiom.getIndividual().asOWLNamedIndividual();
				target = axiom.getClassExpression().asOWLClass();
			}
		}

		@Override
		public void visit(OWLEquivalentClassesAxiom axiom) {
			for (OWLClass cls : axiom.getNamedClasses()) {
				if (source == null) {
					source = cls;
				} else if (target == null) {
					target = cls;
				} else {
					break;
				}
			}
		}

		@Override
		public void visit(OWLDataPropertyAssertionAxiom axiom) {
		}

		@Override
		public void visit(OWLTransitiveObjectPropertyAxiom axiom) {
		}

		@Override
		public void visit(OWLIrreflexiveObjectPropertyAxiom axiom) {
		}

		@Override
		public void visit(OWLSubDataPropertyOfAxiom axiom) {
		}

		@Override
		public void visit(OWLInverseFunctionalObjectPropertyAxiom axiom) {
		}

		@Override
		public void visit(OWLSameIndividualAxiom axiom) {
		}

		@Override
		public void visit(OWLSubPropertyChainOfAxiom axiom) {
		}

		@Override
		public void visit(OWLInverseObjectPropertiesAxiom axiom) {
		}

		@Override
		public void visit(SWRLRule rule) {
		}

		@Override
		public void visit(OWLHasKeyAxiom axiom) {
		}

		@Override
		public void visit(OWLAnnotationPropertyDomainAxiom axiom) {
		}

		@Override
		public void visit(OWLAnnotationPropertyRangeAxiom axiom) {
		}

		@Override
		public void visit(OWLSubAnnotationPropertyOfAxiom axiom) {
		}

		@Override
		public void visit(OWLDatatypeDefinitionAxiom axiom) {
		}
	}

	// ///////////////////////////////////////////////////////////////////////////////////////////////////////////
	// ///////////////////////////////////////////////////////////////////////////////////////////////////////////
	/** A visitor that indexes axioms by their left and right hand sides. */
	@SuppressWarnings("unused")
	private class AxiomMapBuilder implements OWLAxiomVisitor {

		public AxiomMapBuilder() {
		}

		@Override
		public void visit(OWLSubClassOfAxiom axiom) {
			if (!axiom.getSubClass().isAnonymous()) {
				getAxiomsForLHS(axiom.getSubClass().asOWLClass()).add(axiom);
				indexAxiomsByRHSEntities(axiom.getSuperClass(), axiom);
			}
		}

		@Override
		public void visit(OWLNegativeObjectPropertyAssertionAxiom axiom) {
		}

		@Override
		public void visit(OWLAsymmetricObjectPropertyAxiom axiom) {
			if (!axiom.getProperty().isAnonymous()) {
				getAxiomsForLHS(axiom.getProperty().asOWLObjectProperty())
						.add(axiom);
			}
		}

		@Override
		public void visit(OWLReflexiveObjectPropertyAxiom axiom) {
			if (!axiom.getProperty().isAnonymous()) {
				getAxiomsForLHS(axiom.getProperty().asOWLObjectProperty())
						.add(axiom);
			}
		}

		@Override
		public void visit(OWLDisjointClassesAxiom axiom) {
			for (OWLClassExpression desc : axiom.getClassExpressions()) {
				if (!desc.isAnonymous()) {
					getAxiomsForLHS(desc.asOWLClass()).add(axiom);
				}
				indexAxiomsByRHSEntities(desc, axiom);
			}
		}

		@Override
		public void visit(OWLDataPropertyDomainAxiom axiom) {
			getAxiomsForLHS(axiom.getProperty().asOWLDataProperty()).add(axiom);
			indexAxiomsByRHSEntities(axiom.getDomain(), axiom);
		}

		public void visit(OWLImportsDeclaration axiom) {
		}

		@Override
		public void visit(OWLObjectPropertyDomainAxiom axiom) {
			if (!axiom.getProperty().isAnonymous()) {
				getAxiomsForLHS(axiom.getProperty().asOWLObjectProperty())
						.add(axiom);
			}
			indexAxiomsByRHSEntities(axiom.getDomain(), axiom);
		}

		@Override
		public void visit(OWLEquivalentObjectPropertiesAxiom axiom) {
			for (OWLObjectPropertyExpression prop : axiom.getProperties()) {
				if (!prop.isAnonymous()) {
					getAxiomsForLHS(prop.asOWLObjectProperty()).add(axiom);
				}
				indexAxiomsByRHSEntities(prop, axiom);
			}
		}

		@Override
		public void visit(OWLNegativeDataPropertyAssertionAxiom axiom) {
		}

		@Override
		public void visit(OWLDifferentIndividualsAxiom axiom) {
			for (OWLIndividual ind : axiom.getIndividuals()) {
				if (!ind.isAnonymous()) {
					getAxiomsForLHS(ind.asOWLNamedIndividual()).add(axiom);
					indexAxiomsByRHSEntities(ind, axiom);
				}
			}
		}

		@Override
		public void visit(OWLDisjointDataPropertiesAxiom axiom) {
			for (OWLDataPropertyExpression prop : axiom.getProperties()) {
				getAxiomsForLHS(prop.asOWLDataProperty()).add(axiom);
				indexAxiomsByRHSEntities(prop, axiom);
			}
		}

		@Override
		public void visit(OWLDisjointObjectPropertiesAxiom axiom) {
			for (OWLObjectPropertyExpression prop : axiom.getProperties()) {
				if (!prop.isAnonymous()) {
					getAxiomsForLHS(prop.asOWLObjectProperty()).add(axiom);
				}
				indexAxiomsByRHSEntities(prop, axiom);
			}
		}

		@Override
		public void visit(OWLObjectPropertyRangeAxiom axiom) {
			if (!axiom.getProperty().isAnonymous()) {
				getAxiomsForLHS(axiom.getProperty().asOWLObjectProperty())
						.add(axiom);
			}
			indexAxiomsByRHSEntities(axiom.getRange(), axiom);
		}

		@Override
		public void visit(OWLObjectPropertyAssertionAxiom axiom) {
		}

		@Override
		public void visit(OWLFunctionalObjectPropertyAxiom axiom) {
			if (!axiom.getProperty().isAnonymous()) {
				getAxiomsForLHS(axiom.getProperty().asOWLObjectProperty())
						.add(axiom);
			}
		}

		@Override
		public void visit(OWLSubObjectPropertyOfAxiom axiom) {
			if (!axiom.getSubProperty().isAnonymous()) {
				getAxiomsForLHS(axiom.getSubProperty().asOWLObjectProperty())
						.add(axiom);
			}
			indexAxiomsByRHSEntities(axiom.getSuperProperty(), axiom);
		}

		@Override
		public void visit(OWLDisjointUnionAxiom axiom) {
			getAxiomsForLHS(axiom.getOWLClass()).add(axiom);
		}

		@Override
		public void visit(OWLDeclarationAxiom axiom) {
		}

		@Override
		public void visit(OWLAnnotationAssertionAxiom axiom) {
		}

		@Override
		public void visit(OWLSymmetricObjectPropertyAxiom axiom) {
			if (!axiom.getProperty().isAnonymous()) {
				getAxiomsForLHS(axiom.getProperty().asOWLObjectProperty())
						.add(axiom);
			}
		}

		@Override
		public void visit(OWLDataPropertyRangeAxiom axiom) {
			if (!axiom.getProperty().isAnonymous()) {
				getAxiomsForLHS(axiom.getProperty().asOWLDataProperty())
						.add(axiom);
			}
			indexAxiomsByRHSEntities(axiom.getRange(), axiom);
		}

		@Override
		public void visit(OWLFunctionalDataPropertyAxiom axiom) {
			if (!axiom.getProperty().isAnonymous()) {
				getAxiomsForLHS(axiom.getProperty().asOWLDataProperty())
						.add(axiom);
			}
		}

		@Override
		public void visit(OWLEquivalentDataPropertiesAxiom axiom) {
			for (OWLDataPropertyExpression prop : axiom.getProperties()) {
				getAxiomsForLHS(prop.asOWLDataProperty()).add(axiom);
				indexAxiomsByRHSEntities(prop, axiom);
			}
		}

		@Override
		public void visit(OWLClassAssertionAxiom axiom) {
			if (!axiom.getIndividual().isAnonymous()) {
				getAxiomsForLHS(axiom.getIndividual().asOWLNamedIndividual())
						.add(axiom);
				indexAxiomsByRHSEntities(axiom.getClassExpression(), axiom);
			}
		}

		@Override
		public void visit(OWLEquivalentClassesAxiom axiom) {
			for (OWLClassExpression desc : axiom.getClassExpressions()) {
				if (!desc.isAnonymous()) {
					getAxiomsForLHS(desc.asOWLClass()).add(axiom);
				}
				indexAxiomsByRHSEntities(desc, axiom);
			}
		}

		@Override
		public void visit(OWLDataPropertyAssertionAxiom axiom) {
			indexAxiomsByRHSEntities(axiom.getSubject(), axiom);
		}

		@Override
		public void visit(OWLTransitiveObjectPropertyAxiom axiom) {
			if (!axiom.getProperty().isAnonymous()) {
				getAxiomsForLHS(axiom.getProperty().asOWLObjectProperty())
						.add(axiom);
			}
		}

		@Override
		public void visit(OWLIrreflexiveObjectPropertyAxiom axiom) {
			if (!axiom.getProperty().isAnonymous()) {
				getAxiomsForLHS(axiom.getProperty().asOWLObjectProperty())
						.add(axiom);
			}
		}

		@Override
		public void visit(OWLSubDataPropertyOfAxiom axiom) {
			getAxiomsForLHS(axiom.getSubProperty().asOWLDataProperty())
					.add(axiom);
			indexAxiomsByRHSEntities(axiom.getSuperProperty(), axiom);
		}

		@Override
		public void visit(OWLInverseFunctionalObjectPropertyAxiom axiom) {
			if (!axiom.getProperty().isAnonymous()) {
				getAxiomsForLHS(axiom.getProperty().asOWLObjectProperty())
						.add(axiom);
			}
		}

		@Override
		public void visit(OWLSameIndividualAxiom axiom) {
			for (OWLIndividual ind : axiom.getIndividuals()) {
				if (!ind.isAnonymous()) {
					getAxiomsForLHS(ind.asOWLNamedIndividual()).add(axiom);
					indexAxiomsByRHSEntities(ind, axiom);
				}
			}
		}

		@Override
		public void visit(OWLSubPropertyChainOfAxiom axiom) {
		}

		@Override
		public void visit(OWLDatatypeDefinitionAxiom axiom) {
		}

		@Override
		public void visit(OWLInverseObjectPropertiesAxiom axiom) {
			if (!axiom.getFirstProperty().isAnonymous()) {
				getAxiomsForLHS(axiom.getFirstProperty().asOWLObjectProperty())
						.add(axiom);
			}
			indexAxiomsByRHSEntities(axiom.getFirstProperty(), axiom);
			indexAxiomsByRHSEntities(axiom.getSecondProperty(), axiom);
		}

		@Override
		public void visit(SWRLRule rule) {
		}

		@Override
		public void visit(OWLHasKeyAxiom axiom) {
			if (!axiom.getClassExpression().isAnonymous()) {
				indexAxiomsByRHSEntities(
						axiom.getClassExpression().asOWLClass(), axiom);
			}
		}

		@Override
		public void visit(OWLAnnotationPropertyDomainAxiom axiom) {
		}

		@Override
		public void visit(OWLAnnotationPropertyRangeAxiom axiom) {
		}

		@Override
		public void visit(OWLSubAnnotationPropertyOfAxiom axiom) {
		}
	}
}