/*
 * Decompiled with CFR 0.152.
 */
package org.liquigraph.core.io.lock;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.Map;
import java.util.UUID;
import java.util.function.Supplier;
import org.liquigraph.core.exception.LiquigraphLockException;
import org.liquigraph.core.io.lock.ShutdownTask;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LiquigraphLock {
    private static final Logger LOGGER = LoggerFactory.getLogger(LiquigraphLock.class);
    private final UUID uuid = UUID.randomUUID();
    private final Map<Connection, Boolean> connections = new IdentityHashMap<Connection, Boolean>();
    private final Thread task = new Thread(new ShutdownTask(this));
    private final Supplier<Connection> connectionSupplier;

    public LiquigraphLock(Supplier<Connection> connection) {
        this.connectionSupplier = connection;
    }

    void acquire(Connection connection) {
        if (this.addConnection(connection)) {
            LOGGER.debug("Acquiring lock {} on database", (Object)this.uuid);
            this.addShutdownHook();
            this.ensureLockUniqueness(connection);
            this.tryWriteLock(connection);
        }
    }

    void release(Connection connection) {
        if (this.removeConnection(connection)) {
            LOGGER.debug("Releasing lock {} from database", (Object)this.uuid);
            this.removeShutdownHook();
            this.releaseLock(connection);
        }
    }

    void cleanup() {
        for (Connection connection : new HashSet<Connection>(this.connections.keySet())) {
            this.release(connection);
        }
    }

    private boolean addConnection(Connection connection) {
        boolean wasEmpty = this.connections.isEmpty();
        this.connections.put(connection, Boolean.TRUE);
        return wasEmpty;
    }

    private boolean removeConnection(Connection connection) {
        if (this.connections.isEmpty()) {
            return false;
        }
        this.connections.remove(connection);
        return this.connections.isEmpty();
    }

    private void addShutdownHook() {
        Runtime.getRuntime().addShutdownHook(this.task);
    }

    private void removeShutdownHook() {
        Runtime.getRuntime().removeShutdownHook(this.task);
    }

    private void ensureLockUniqueness(Connection connection) {
        if (this.lockConstraintsExists(connection)) {
            return;
        }
        try (Statement statement = connection.createStatement();){
            statement.execute("CREATE CONSTRAINT ON (lock:__LiquigraphLock) ASSERT lock.name IS UNIQUE");
            connection.commit();
        }
        catch (SQLException e) {
            throw new LiquigraphLockException("Could not ensure __LiquigraphLock uniqueness\n\tPlease make sure your instance is in a clean state\n\tNo more than 1 lock should be there simultaneously!", e);
        }
    }

    private boolean lockConstraintsExists(Connection connection) {
        boolean result = false;
        try (Statement statement = connection.createStatement();
             ResultSet resultSet = statement.executeQuery("CALL db.constraints() YIELD description WHERE description CONTAINS '__LiquigraphLock' AND description CONTAINS 'UNIQUE' RETURN SIZE(COLLECT(description)) > 0 AS result");){
            if (resultSet.next()) {
                result = resultSet.getBoolean("result");
            }
            connection.commit();
        }
        catch (SQLException e) {
            throw new LiquigraphLockException("Could not verify  __LiquigraphLock uniqueness constraint existence", e);
        }
        return result;
    }

    private void tryWriteLock(Connection connection) {
        try (PreparedStatement statement = connection.prepareStatement("CREATE (:__LiquigraphLock {name:'John', uuid:?})");){
            statement.setString(1, this.uuid.toString());
            statement.execute();
            connection.commit();
        }
        catch (SQLException e) {
            throw new LiquigraphLockException("Cannot create __LiquigraphLock lock\n\tLikely another Liquigraph execution is going on or has crashed.", e);
        }
    }

    private void releaseLock(Connection connection) {
        String deleteLockQuery = "MATCH (lock:__LiquigraphLock {uuid:?}) DELETE lock";
        try (PreparedStatement statement = connection.prepareStatement(deleteLockQuery);){
            statement.setString(1, this.uuid.toString());
            statement.execute();
            connection.commit();
        }
        catch (SQLException firstAttemptException) {
            LOGGER.info("Failed to remove __LiquigraphLock. Trying again with new connection", (Throwable)firstAttemptException);
            try (PreparedStatement statement2 = this.connectionSupplier.get().prepareStatement(deleteLockQuery);){
                statement2.setString(1, this.uuid.toString());
                statement2.execute();
                connection.commit();
            }
            catch (SQLException secondAttemptException) {
                LOGGER.error("Cannot remove __LiquigraphLock during cleanup.", (Throwable)secondAttemptException);
            }
        }
    }
}

