/*
 * Decompiled with CFR 0.152.
 */
package org.liquigraph.core.io;

import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import org.liquigraph.core.exception.Throwables;
import org.liquigraph.core.io.ChangelogWriter;
import org.liquigraph.core.io.ConditionPrinter;
import org.liquigraph.core.model.Changeset;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ChangelogFileWriter
implements ChangelogWriter {
    private static final Logger LOGGER = LoggerFactory.getLogger(ChangelogFileWriter.class);
    private final ConditionPrinter conditionPrinter;
    private final String database;
    private final File outputFile;

    public ChangelogFileWriter(ConditionPrinter conditionPrinter, String database, File outputFile) {
        this.conditionPrinter = conditionPrinter;
        this.database = database;
        this.outputFile = outputFile;
    }

    @Override
    public void write(Collection<Changeset> changelogsToInsert) {
        try {
            LOGGER.info("About to dry-run Liquigraph. Results in file {}", (Object)this.outputFile.getAbsolutePath());
            Path path = this.outputFile.toPath();
            this.reinitializeFile(path);
            if (changelogsToInsert.isEmpty()) {
                this.writeNothingToPersist(path);
                return;
            }
            this.writeHeaderMaybe(path);
            for (Changeset changeset : changelogsToInsert) {
                this.writeChangeset(changeset, path);
            }
        }
        catch (IOException e) {
            throw Throwables.propagate(e);
        }
    }

    private void reinitializeFile(Path path) throws IOException {
        Files.deleteIfExists(path);
        Files.createFile(path, new FileAttribute[0]);
    }

    private void writeNothingToPersist(Path path) throws IOException {
        String nothingToDoMessage = String.format("//Liquigraph%s: nothing to persist!", this.databaseInstanceOrEmpty());
        Files.write(path, Collections.singletonList(nothingToDoMessage), StandardCharsets.UTF_8, new OpenOption[0]);
    }

    private void writeHeaderMaybe(Path path) throws IOException {
        if (this.database == null) {
            return;
        }
        String header = String.format("//Liquigraph (instance: %s)", this.database);
        Files.write(path, Collections.singletonList(header), StandardCharsets.UTF_8, new OpenOption[0]);
    }

    private String databaseInstanceOrEmpty() {
        return this.database == null ? "" : " (instance " + this.database + ")";
    }

    private void writeChangeset(Changeset changeset, Path path) throws IOException {
        Files.write(path, this.conditionPrinter.print(changeset.getPrecondition()), StandardCharsets.UTF_8, StandardOpenOption.APPEND);
        Files.write(path, this.changesetToString(changeset), StandardCharsets.UTF_8, StandardOpenOption.APPEND);
        Files.write(path, this.conditionPrinter.print(changeset.getPostcondition()), StandardCharsets.UTF_8, StandardOpenOption.APPEND);
    }

    private Collection<String> changesetToString(Changeset changeset) {
        ArrayList<String> lines = new ArrayList<String>();
        lines.add(String.format("//Liquigraph changeset[author: %s, id: %s]", changeset.getAuthor(), changeset.getId()));
        lines.add(String.format("//Liquigraph changeset[executionContexts: %s]", this.flatten(changeset.getExecutionsContexts())));
        lines.addAll(changeset.getQueries());
        return lines;
    }

    private String flatten(Collection<String> executionsContexts) {
        if (executionsContexts.isEmpty()) {
            return "none declared";
        }
        return String.join((CharSequence)",", executionsContexts);
    }
}

