package org.linkedopenactors.rdfpub.client;

import java.io.StringWriter;
import java.util.List;

import org.eclipse.rdf4j.model.IRI;
import org.eclipse.rdf4j.model.Model;
import org.eclipse.rdf4j.model.Resource;
import org.eclipse.rdf4j.model.Value;
import org.eclipse.rdf4j.model.util.Models;
import org.eclipse.rdf4j.query.BindingSet;
import org.eclipse.rdf4j.rio.RDFFormat;
import org.eclipse.rdf4j.rio.Rio;
import org.linkedopenactors.ns.rdfpub.RDFPUB;

import de.naturzukunft.rdf4j.utils.ModelLogger;
import de.naturzukunft.rdf4j.vocabulary.AS;
import lombok.extern.slf4j.Slf4j;

@Slf4j
public class SimpleProfile {

	private RdfPubClient rdfPubClient;
	private Model profileModel;
	
	public SimpleProfile(RdfPubClient rdfPubClient) {
		this.rdfPubClient = rdfPubClient;
		profileModel = rdfPubClient.getProfile().orElseThrow(()->new RuntimeException("no profile for " + this.rdfPubClient.getActorId()));
	}
	
	public Model outbox(String authToken) {
		return collection(AS.outbox, authToken);
	}
	
	public Model getStatementsOutbox(String authToken, Resource subj, IRI pred, Value obj) {
		return rdfPubClient.getStatementsOutbox(authToken, subj, pred, obj);
	}
	
	public Model getStatementsPublic(Resource subj, IRI pred, Value obj) {
		return rdfPubClient.getStatementsPublic(subj, pred, obj);
	}
	
	public Model inbox(String authToken) {
		return collection(AS.inbox, authToken);
	}

	public IRI actorId() {
		return this.rdfPubClient.getActorId();
	}
	
	public Model collection(IRI collectionId, String authToken) {
		IRI collectionIri = Models.getPropertyIRI(profileModel, this.rdfPubClient.getActorId(), collectionId).orElseThrow(()->new RuntimeException("no "+collectionId+" for " + this.rdfPubClient.getActorId()));
		return rdfPubClient.read(collectionIri, authToken).orElseThrow(()->new RuntimeException("no "+collectionId+" content for " + collectionIri));		
	}

	public IRI outboxSparqlEndpoint() {
		return Models.getPropertyIRI(profileModel, this.rdfPubClient.getActorId(), RDFPUB.OUTBOX_SPARQL).orElseThrow(()->new RuntimeException("no '" + RDFPUB.OUTBOX_SPARQL + "' for '" + this.rdfPubClient.getActorId() + "'"));
	}
	
	public List<BindingSet> queryOutbox(String query, String authToken) {
		return rdfPubClient.tupleQueryOutbox(query, authToken);
	}
	
	public Model constructFromOutbox(String query, String authToken) {
		return rdfPubClient.graphQueryOutbox(query, authToken);
	}

	public IRI getOutboxSparqlEndpoint() {
		IRI sparqlEndpoint = Models.getPropertyIRI(profileModel, this.rdfPubClient.getActorId(), RDFPUB.OUTBOX_SPARQL).orElseThrow(()->{
			ModelLogger.error(log, profileModel, "profile: ");
			return new RuntimeException("no '" + RDFPUB.OUTBOX_SPARQL + "' for '" + this.rdfPubClient.getActorId() + "'");
			});
		return sparqlEndpoint;
	}
	
	public IRI getPublicSparqlEndpoint() {
		IRI sparqlEndpoint = Models.getPropertyIRI(profileModel, this.rdfPubClient.getActorId(), RDFPUB.PUBLIC_SPARQL).orElseThrow(()->{
			ModelLogger.error(log, profileModel, "profile: ");
			return new RuntimeException("no '" + RDFPUB.PUBLIC_SPARQL + "' for '" + this.rdfPubClient.getActorId() + "'");
			});
		return sparqlEndpoint;
	}

	public IRI getActorId() {
		return rdfPubClient.getActorId();
	}
	
	public String toString() {
		StringWriter sw = new StringWriter();
		if (profileModel != null) {
			Rio.write(profileModel, sw, RDFFormat.TURTLE);
		}
		return sw.toString();
	}
	
	public Model asModel() {
		return profileModel;
	}
}
