package org.linkedopenactors.rdfpub.client;

import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import java.time.Duration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Stream;

import org.eclipse.rdf4j.model.IRI;
import org.eclipse.rdf4j.model.Model;
import org.eclipse.rdf4j.model.util.Values;
import org.eclipse.rdf4j.query.BindingSet;
import org.eclipse.rdf4j.query.QueryResults;
import org.eclipse.rdf4j.repository.sparql.SPARQLRepository;
import org.eclipse.rdf4j.repository.util.Repositories;
import org.eclipse.rdf4j.rio.RDFFormat;
import org.eclipse.rdf4j.rio.RDFParseException;
import org.eclipse.rdf4j.rio.Rio;
import org.eclipse.rdf4j.rio.UnsupportedRDFormatException;
import org.linkedopenactors.rdfpub.domain.DomainObjectFactory;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.reactive.function.BodyInserters;
import org.springframework.web.reactive.function.client.ClientResponse;
import org.springframework.web.reactive.function.client.WebClient;

import lombok.extern.slf4j.Slf4j;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.util.retry.Retry;

/**
 * Default implementation of {@link RdfPubClient}.  
 * @author <a href="http://hauschel.de">SofwareEngineering Hauschel</a>
 */
@Slf4j
public class RdfPubClientWebFluxDefault implements RdfPubClientWebFlux {
		
	private static final MediaType MEDIA_TYPE_JSON_LD = new MediaType("application", "ld+json");
	private WebClient webClient;
	private Map<IRI,SPARQLRepository> sparqlRepositories = new HashMap<>();
	private String rdfPubServerUrl;
	private String userId;

	/**
	 * 
	 * @param rdfPubServerUrl rdfPubServerUrl
	 * @param webClient The webclient to use for http communication.
	 * @param userId userId
	 */
	public RdfPubClientWebFluxDefault(String rdfPubServerUrl, WebClient webClient, String userId) {
		this.rdfPubServerUrl = rdfPubServerUrl;
		this.webClient = webClient;
		this.userId = userId;
	}
	
	@Override
	public Mono<org.linkedopenactors.rdfpub.domain.ActorProfile> getCurrentActorProfile() {		
			return getActorId()
					.flatMap(this::toActorProfile);
	}

	private Mono<org.linkedopenactors.rdfpub.domain.ActorProfile> toActorProfile(IRI actorId) {
		DomainObjectFactory domainObjectFactory = new DomainObjectFactory();
		return httpGetApString(actorId)				
				.map(model -> domainObjectFactory.createRdfObject(model)) 
				.map(rdfObject -> domainObjectFactory.createActorProfile(rdfObject));
	}
	
	
	@Override
	public Mono<IRI> getActorId() {		
		IRI webFingerUrl = Values.iri(rdfPubServerUrl + ".well-known/webfinger?resource=" + userId);
		return httpGetJson(webFingerUrl).map(json-> {
			WebfingerParser p = new WebfingerParser();
			JsonResourceDescriptor jrd = p.parseJRD(json);
			Link l = jrd.getLinkByRel("self");
			return Values.iri(l.getHref().toString());
		});
	}
	
	@Override
	public Mono<RdfPubServerProfile> getServerProfile() {
		return getServerProfileModel()
			.map(model->new RdfPubServerProfileDefault(model));
	}
	
	private Mono<org.eclipse.rdf4j.model.Model> getServerProfileModel() { 
		WebClient client = WebClient.create(rdfPubServerUrl);
		return client.get()				
			    .retrieve()
			    .bodyToMono(String.class)
			    .map(httpResponseBody->{
					try {
						return Rio.parse(new StringReader(httpResponseBody), RDFFormat.TURTLE);
					} catch (RDFParseException | UnsupportedRDFormatException | IOException e) {
						String message = "Error parsing httpResponseBody. " + e.getMessage();								
						log.trace(message + ": " + httpResponseBody);								
						throw new IllegalStateException(message, e);
					}
				});
	}	
	
	@Override
	public Mono<Model> read(IRI idOfTheResourceToRead) {
		log.trace("read(" + idOfTheResourceToRead + ")");
		return httpGetApString(idOfTheResourceToRead);
	}

	private Mono<Model> httpGetApString(IRI idOfTheResourceToRead) {
		log.trace("httpGetApString(" + idOfTheResourceToRead+")");
		var request = webClient
				.get()
				.uri(idOfTheResourceToRead.stringValue())
				.accept(MEDIA_TYPE_JSON_LD)
				.header("profile", "https://www.w3.org/ns/activitystreams");
		
		Mono<Model> d = request 
				.retrieve()
				.onStatus(HttpStatus::is5xxServerError, 
				          response -> Mono.error(new ServiceException("Server error", response.rawStatusCode())))
				.onStatus(HttpStatus::is4xxClientError, ClientResponse::createException)
				.bodyToMono(String.class)
				.retryWhen(Retry.backoff(3, Duration.ofSeconds(2))
					.filter(throwable -> throwable instanceof ServiceException))
				.timeout(Duration.ofMinutes(3))
				.map(Optional::ofNullable)
				.map(httpResponseBodyOptional->
					{
						return httpResponseBodyOptional.map(httpResponseBody->{
							try {
								return Rio.parse(new StringReader(httpResponseBody), RDFFormat.JSONLD);
							} catch (RDFParseException | UnsupportedRDFormatException | IOException e1) {
								String message = "Error parsing httpResponseBody. " + e1.getMessage();								
								log.trace(message + ": " + httpResponseBody);								
								throw new RuntimeException(message, e1);
								}
							});
					})
				.flatMap(optional -> optional.map(Mono::just).orElseGet(Mono::empty));
		return d;
	}
	
	private Mono<String> httpGetJson(IRI idOfTheResourceToRead) {
		log.trace("httpGetApString(" + idOfTheResourceToRead+")");
		var request = webClient
				.get()
				.uri(idOfTheResourceToRead.stringValue())
				.accept(MediaType.APPLICATION_JSON);
		
		Mono<String> d = request 
				.retrieve()
				.onStatus(HttpStatus::is5xxServerError, 
				          response -> Mono.error(new ServiceException("Server error", response.rawStatusCode())))
				.onStatus(HttpStatus::is4xxClientError, ClientResponse::createException)
				.bodyToMono(String.class)
				.retryWhen(Retry.backoff(3, Duration.ofSeconds(2))
					.filter(throwable -> throwable instanceof ServiceException))
				.timeout(Duration.ofMinutes(3));
		return d;
	}
	@Override
	public Flux<BindingSet> tupleQueryOutbox(String query, String authToken) {
		Mono<SPARQLRepository> repoMono = getCurrentActorProfile()
				.map(org.linkedopenactors.rdfpub.domain.ActorProfile::getOutboxSparqlEndpoint)
				.map(endpoint->getSparqlRepository(endpoint, authToken));
		return tupleQuery(query, authToken, repoMono);
	}
	
	@Override
	public Flux<BindingSet> tupleQueryInbox(String query, String authToken) {
		Mono<SPARQLRepository> repoMono = getCurrentActorProfile()
			.map(org.linkedopenactors.rdfpub.domain.ActorProfile::getInboxSparqlEndpoint)
			.map(endpoint->getSparqlRepository(endpoint, authToken));
		return tupleQuery(query, authToken, repoMono);
	}

	@Override
	public Flux<BindingSet> tupleQueryAsPublic(String query) {
	return getServerProfile()
		.map(profile->profile.getPublicSparqlEndpoint())
		.map(this::getSparqlRepository2)
		.map(repo->tupleQuery2(query, repo))
		.flatMapIterable(list -> list);
	}

	private Flux<BindingSet> tupleQuery(String query, String token, Mono<SPARQLRepository> repoMono) {
		Mono<List<BindingSet>> monoListOfBindingSets = repoMono.map(repo->{
			log.trace("now querying using 'Repositories.tupleQuery(...)' ["+repo+"]");
			log.trace("query("+query+")");

			final List<BindingSet> bindingSets = Repositories.tupleQuery(repo, query, r -> QueryResults.asList(r));
			return bindingSets;
		});
		
		Flux<BindingSet> fluxOfBindingSets =  monoListOfBindingSets
	      .flatMapMany(Flux::fromIterable)
	      .log();
		
		return fluxOfBindingSets;
	}

	private List<BindingSet> tupleQuery2(String query, SPARQLRepository repo) {
			log.trace("now querying using 'Repositories.tupleQuery(...)' ["+repo+"]");
			log.trace("query("+query+")");

			final List<BindingSet> bindingSets = Repositories.tupleQuery(repo, query, r -> QueryResults.asList(r));
		return bindingSets;
	}
	
	public Mono<SPARQLRepository> getSparqlRepository(Mono<IRI> sparqlEndpointMono, String token) {
		return sparqlEndpointMono.map(sparqlEndpoint->{
			log.trace("getSparqlRepository("+sparqlEndpoint+")");
			SPARQLRepository repo = sparqlRepositories.get(sparqlEndpoint);
			if(sparqlRepositories.containsKey(sparqlEndpoint)) {
				repo = sparqlRepositories.get(sparqlEndpoint);
			} else {
				repo = new SPARQLRepository(sparqlEndpoint.stringValue());
				sparqlRepositories.put(sparqlEndpoint, repo);
			}
			if(token!=null) {
				Map<String, String> additionalHttpHeaders = getAdditionalHttpHeaders(repo);
				additionalHttpHeaders.put("Authorization", "Bearer " + token);
				repo.setAdditionalHttpHeaders(additionalHttpHeaders);
				}
			return repo;
		});
	}

	public Mono<SPARQLRepository> getSparqlRepository(Mono<IRI> sparqlEndpointMono) {
		return getSparqlRepository(sparqlEndpointMono, null);
	}

	public SPARQLRepository getSparqlRepository2(IRI sparqlEndpointMono) {
		return getSparqlRepository2(sparqlEndpointMono, null);
	}

	public SPARQLRepository getSparqlRepository2(IRI sparqlEndpoint, String token) {
		log.trace("getSparqlRepository("+sparqlEndpoint+")");
		SPARQLRepository repo = sparqlRepositories.get(sparqlEndpoint);
		if(sparqlRepositories.containsKey(sparqlEndpoint)) {
			repo = sparqlRepositories.get(sparqlEndpoint);
		} else {
			repo = new SPARQLRepository(sparqlEndpoint.stringValue());
			sparqlRepositories.put(sparqlEndpoint, repo);
		}
		if(token!=null) {
			Map<String, String> additionalHttpHeaders = getAdditionalHttpHeaders(repo);
			additionalHttpHeaders.put("Authorization", "Bearer " + token);
			repo.setAdditionalHttpHeaders(additionalHttpHeaders);
			}
		return repo;
	}

	public SPARQLRepository getSparqlRepository(IRI sparqlEndpoint, String token) {
		log.trace("getSparqlRepository("+sparqlEndpoint+")");
		SPARQLRepository repo = sparqlRepositories.get(sparqlEndpoint);
		if(sparqlRepositories.containsKey(sparqlEndpoint)) {
			repo = sparqlRepositories.get(sparqlEndpoint);
		} else {
			repo = new SPARQLRepository(sparqlEndpoint.stringValue());
			sparqlRepositories.put(sparqlEndpoint, repo);
		}
		Map<String, String> additionalHttpHeaders = getAdditionalHttpHeaders(repo);
		additionalHttpHeaders.put("Authorization", "Bearer " + token);
		repo.setAdditionalHttpHeaders(additionalHttpHeaders);
		return repo;
	}

	private Map<String, String> getAdditionalHttpHeaders(SPARQLRepository repo) {
		Map<String, String> newAdditionalHttpHeaders = new HashMap<>();
		Map<String, String> additionalHttpHeaders = repo.getAdditionalHttpHeaders();
		if(additionalHttpHeaders!=null) {
			newAdditionalHttpHeaders.putAll(additionalHttpHeaders);
		}		
		return newAdditionalHttpHeaders;		
	}

	@Override
	public Mono<IRI> postActivity(Model activity, String token) {
		log.trace("postActivity("+activity+")");
		Mono<String> activityIriAsString = getCurrentActorProfile()
				.map(org.linkedopenactors.rdfpub.domain.ActorProfile::getOutboxEndpoint)
				.flatMap(outboxId->doPost(activity, outboxId.stringValue()));
		return activityIriAsString.map(Values::iri);
	}

	private Mono<String> doPost(Model activity, String outbox) {
		log.trace("outbox: " + outbox);		 
		StringWriter bodyAsStringWriter = new StringWriter();
		Rio.write(activity, bodyAsStringWriter, RDFFormat.JSONLD);
		
		String body = bodyAsStringWriter.toString();
		log.trace("body: " + body );
		
		Mono<String> activityIriAsString = webClient
		.post()			
		.uri(outbox)		
		.contentType(MEDIA_TYPE_JSON_LD)
//		.accept(MEDIA_TYPE_JSON_LD)
		.body(BodyInserters.fromValue(body))		
		.header("profile", "https://www.w3.org/ns/activitystreams")
		.retrieve()
		.onStatus(HttpStatus::isError, ClientResponse::createException)
		.toEntity(String.class)
		.doOnNext(it->log.trace("post response: " + it))		
		.flatMap(res->{
			return getLocationHeader(res.getHeaders().get("Location"));
		})
		.doOnNext(location->log.trace("post location: " + location))
		.timeout(Duration.ofMinutes(3));
		return activityIriAsString;
	}
	
	
	private Mono<String> getLocationHeader(List<String> locs) {
		if(locs == null) {
			return Mono.just("");
		}
		return Mono.just(locs)
		.map(List::stream)
		.map(Stream::findFirst)
		.map(optional->optional.orElse(""));
	}

//	@Override
//	public IRI getActorId() {
//		return this.clientApplicationActor;
//	}
}
