package org.linkedopenactors.rdfpub.client;

import java.io.IOException;
import java.io.StringReader;
import java.time.Duration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.eclipse.rdf4j.model.IRI;
import org.eclipse.rdf4j.model.Model;
import org.eclipse.rdf4j.query.BindingSet;
import org.eclipse.rdf4j.query.QueryResults;
import org.eclipse.rdf4j.repository.sparql.SPARQLRepository;
import org.eclipse.rdf4j.repository.util.Repositories;
import org.eclipse.rdf4j.rio.RDFFormat;
import org.eclipse.rdf4j.rio.RDFParseException;
import org.eclipse.rdf4j.rio.Rio;
import org.eclipse.rdf4j.rio.UnsupportedRDFormatException;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.reactive.function.client.ClientResponse;
import org.springframework.web.reactive.function.client.WebClient;

import lombok.extern.slf4j.Slf4j;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.util.retry.Retry;

/**
 * Default implementation of {@link RdfPubClient}.  
 * @author <a href="http://hauschel.de">SofwareEngineering Hauschel</a>
 */
@Slf4j
public class RdfPubClientWebFluxAnonymousDefault implements RdfPubClientAnonymousWebFlux {
		
	private static final MediaType MEDIA_TYPE_JSON_LD = new MediaType("application", "ld+json");
	private Map<IRI,SPARQLRepository> sparqlRepositories = new HashMap<>();
	private WebClient webClient;
	private String rdfPubServerUrl;

	/**
	 * @param webClient The webclient to use for http communication.
	 * @param rdfPubServerUrl rdfPubServerUrl
	 */
	public RdfPubClientWebFluxAnonymousDefault(String rdfPubServerUrl, WebClient webClient) {
		this.rdfPubServerUrl = rdfPubServerUrl;
		this.webClient = webClient;
	}
	
	@Override
	public Mono<RdfPubServerProfile> getServerProfile() {
		return getServerProfileModel()
			.map(model->new RdfPubServerProfileDefault(model));
	}
	
	private Mono<org.eclipse.rdf4j.model.Model> getServerProfileModel() { 
		WebClient client = WebClient.create(rdfPubServerUrl);
		return client.get()				
			    .retrieve()
			    .bodyToMono(String.class)
			    .map(httpResponseBody->{
					try {
						return Rio.parse(new StringReader(httpResponseBody), RDFFormat.TURTLE);
					} catch (RDFParseException | UnsupportedRDFormatException | IOException e) {
						String message = "Error parsing httpResponseBody. " + e.getMessage();								
						log.trace(message + ": " + httpResponseBody);								
						throw new IllegalStateException(message, e);
					}
				});
	}	
	
	@Override
	public Mono<Model> read(IRI idOfTheResourceToRead) {
		log.trace("read(" + idOfTheResourceToRead + ")");
		return httpGetApString(idOfTheResourceToRead);
	}

	private Mono<Model> httpGetApString(IRI idOfTheResourceToRead) {
		log.trace("httpGetApString(" + idOfTheResourceToRead+")");
		var request = webClient
				.get()
				.uri(idOfTheResourceToRead.stringValue())
				.accept(MEDIA_TYPE_JSON_LD)
				.header("profile", "https://www.w3.org/ns/activitystreams");
		
		Mono<Model> d = request 
				.retrieve()
				.onStatus(HttpStatus::is5xxServerError, 
				          response -> Mono.error(new ServiceException("Server error", response.rawStatusCode())))
				.onStatus(HttpStatus::is4xxClientError, ClientResponse::createException)
				.bodyToMono(String.class)
				.retryWhen(Retry.backoff(3, Duration.ofSeconds(2))
					.filter(throwable -> throwable instanceof ServiceException))
				.timeout(Duration.ofMinutes(3))
				.map(Optional::ofNullable)
				.map(httpResponseBodyOptional->
					{
						return httpResponseBodyOptional.map(httpResponseBody->{
							try {
								return Rio.parse(new StringReader(httpResponseBody), RDFFormat.JSONLD);
							} catch (RDFParseException | UnsupportedRDFormatException | IOException e1) {
								String message = "Error parsing httpResponseBody. " + e1.getMessage();								
								log.trace(message + ": " + httpResponseBody);								
								throw new RuntimeException(message, e1);
								}
							});
					})
				.flatMap(optional -> optional.map(Mono::just).orElseGet(Mono::empty));
		return d;
	}
	

	@Override
	public Flux<BindingSet> tupleQueryAsPublic(String query) {
	return getServerProfile()
		.map(profile->profile.getPublicSparqlEndpoint())
		.map(this::getSparqlRepository2)
		.map(repo->tupleQuery2(query, repo))
		.flatMapIterable(list -> list);
	}

	public SPARQLRepository getSparqlRepository2(IRI sparqlEndpointMono) {
		return getSparqlRepository2(sparqlEndpointMono, null);
	}

	public SPARQLRepository getSparqlRepository2(IRI sparqlEndpoint, String token) {
		log.trace("getSparqlRepository("+sparqlEndpoint+")");
		SPARQLRepository repo = sparqlRepositories.get(sparqlEndpoint);
		if(sparqlRepositories.containsKey(sparqlEndpoint)) {
			repo = sparqlRepositories.get(sparqlEndpoint);
		} else {
			repo = new SPARQLRepository(sparqlEndpoint.stringValue());
			sparqlRepositories.put(sparqlEndpoint, repo);
		}
		if(token!=null) {
			Map<String, String> additionalHttpHeaders = getAdditionalHttpHeaders(repo);
			additionalHttpHeaders.put("Authorization", "Bearer " + token);
			repo.setAdditionalHttpHeaders(additionalHttpHeaders);
			}
		return repo;
	}

	private Map<String, String> getAdditionalHttpHeaders(SPARQLRepository repo) {
		Map<String, String> newAdditionalHttpHeaders = new HashMap<>();
		Map<String, String> additionalHttpHeaders = repo.getAdditionalHttpHeaders();
		if(additionalHttpHeaders!=null) {
			newAdditionalHttpHeaders.putAll(additionalHttpHeaders);
		}		
		return newAdditionalHttpHeaders;		
	}

	private List<BindingSet> tupleQuery2(String query, SPARQLRepository repo) {
		log.trace("now querying using 'Repositories.tupleQuery(...)' ["+repo+"]");
		log.trace("query("+query+")");

		final List<BindingSet> bindingSets = Repositories.tupleQuery(repo, query, r -> QueryResults.asList(r));
	return bindingSets;
}
}

