package org.linkedopenactors.rdfpub.client;

import static org.eclipse.rdf4j.model.util.Values.iri;

import java.io.StringWriter;
import java.util.List;
import java.util.Optional;
import java.util.stream.Stream;

import org.eclipse.rdf4j.model.IRI;
import org.eclipse.rdf4j.model.Model;
import org.eclipse.rdf4j.model.Resource;
import org.eclipse.rdf4j.model.Value;
import org.eclipse.rdf4j.model.util.Models;
import org.eclipse.rdf4j.query.BindingSet;
import org.eclipse.rdf4j.query.QueryResults;
import org.eclipse.rdf4j.repository.Repository;
import org.eclipse.rdf4j.repository.RepositoryConnection;
import org.eclipse.rdf4j.repository.util.Repositories;
import org.eclipse.rdf4j.rio.RDFFormat;
import org.eclipse.rdf4j.rio.Rio;
import org.linkedopenactors.ns.rdfpub.RDFPUB;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.web.reactive.function.BodyInserters;
import org.springframework.web.reactive.function.client.ClientResponse;
import org.springframework.web.reactive.function.client.WebClient;

import de.naturzukunft.rdf4j.utils.ModelLogger;
import de.naturzukunft.rdf4j.vocabulary.AS;
import lombok.extern.slf4j.Slf4j;
import reactor.core.publisher.Mono;

/**
 * Default implementation of {@link RdfPubClient}.  
 * @author <a href="http://hauschel.de">SofwareEngineering Hauschel</a>
 */
@Slf4j
public class RdfPubClientDefault extends RdfPubClientAbstract implements RdfPubClient {
	
	private IRI clientApplicationActor = null;
	private IRI normalizedClientApplicationActor = null;
	private Optional<Model> profileOptional;
	
	/**
	 * @param rdfPubServerActorIri The iri/url of the server actor. Used to get the public sparql endpoint
	 * @param webClient The webclient to use for http communication.
	 * @param clientApplicationActor The iri/url of the user. TODO determinate this by webfinger ?!
	 */
	public RdfPubClientDefault(IRI rdfPubServerActorIri, WebClient webClient, IRI clientApplicationActor) {
		super(rdfPubServerActorIri, webClient);
		this.clientApplicationActor = clientApplicationActor;
	}
	
	/**
	 * Determinates the actorIRI in it's original form with it's userId. E.g. http://localhost:8080/camel/0815
	 * We allow actor iri's with preferredUserNames, but the system (other camel components) expect the 'real' actor with it's userId.
	 * So we replace the preferredUserName by it's userId.
	 * @param actorId With preferredUserName or UserId. E.g. http://localhost:8080/camel/max or http://localhost:8080/camel/0815 
	 * @return The actor in it's original form with it's userId. E.g. http://localhost:8080/camel/0815
	 */
    private IRI normalize(IRI actorId) {
		Model profile = getProfile(actorId).orElseThrow(()->new RuntimeException("no profile for '" + actorId + "'"));
		return (IRI)profile.filter(null,  null, null).stream().findFirst().orElseThrow(()->new RuntimeException("no statement in profile for '" + actorId + "'")).getSubject();
	}

    @Override
	public IRI postActivity(Model activity, String authToken) {
		log.trace("postActivity("+activity+")");
		IRI outbox = getOutboxId(getActorId());
		log.trace("outbox: " + outbox);		 
		StringWriter bodyAsStringWriter = new StringWriter();
		Rio.write(activity, bodyAsStringWriter, RDFFormat.JSONLD);
		
		String body = bodyAsStringWriter.toString();
		log.trace("body: " + body );
		
		String activityIriAsString = webClient
		.post()			
		.uri(outbox.stringValue())
		.contentType(MEDIA_TYPE_JSON_LD)
		.body(BodyInserters.fromValue(body))
		.header(HttpHeaders.AUTHORIZATION, authToken)
		.header("profile", "https://www.w3.org/ns/activitystreams")
		.retrieve()
		.onStatus(HttpStatus::isError, ClientResponse::createException)
		.toEntity(String.class)
		.doOnNext(it->log.trace("post response: " + it))
		.flatMap(res->{
			return getLocationHeader(res.getHeaders().get("Location"));
		})
		.doOnNext(location->log.trace("post location: " + location))
		.block();
		return iri(activityIriAsString);
	}

	private IRI getOutboxId(IRI actiorIri) {
		Model profile = getProfile().orElseThrow();		
		Model outboxStatement = profile.filter(null, AS.outbox, null, AS.Public);		
		if(outboxStatement.size()!=1) {
			String message = "no property "+AS.outbox+" in pofile.";
			ModelLogger.error(log, profile, message + ": ");
			throw new RuntimeException(message);	
		}
		IRI outbox = iri(outboxStatement.stream().findFirst().get().getObject().stringValue());
		log.debug("determined outbox id for '"+actiorIri+"' is " + outbox);
		return outbox;
	}

	private Mono<String> getLocationHeader(List<String> locs) {
		if(locs == null) {
			return Mono.just("");
		}
		return Mono.just(locs)
		.map(List::stream)
		.map(Stream::findFirst)
		.map(optional->optional.orElse(""));
	}

	@Override
	public Optional<Model> getProfile() {
		return getProfile(getActorId());
	}

	@Override
	public Optional<Model> getProfile(String authToken) {
		return getProfile(getActorId(), authToken);
	}

	private Optional<Model> getProfile(IRI actorId) {
		return getProfile(actorId, null);
	}
	
	private Optional<Model> getProfile(IRI actorId, String authToken) {
		log.trace("getProfile("+actorId+")");
		if(profileOptional==null) {
			profileOptional = httpGetApString(actorId, authToken);
			if(profileOptional.isPresent()) {
				ModelLogger.trace(log, profileOptional.get(), "profile: ");	
			} else {
				log.warn("no profile for: " + actorId);
			}
		} else {
			log.trace("returning cached profile!");
		}
		return profileOptional;
	}

	@Override
	public Optional<Model> read(IRI idOfTheResourceToRead, String authToken) {
		log.trace("read(" + idOfTheResourceToRead + ")");
		Optional<Model> httpGetApString = null;
//		try {
			httpGetApString = httpGetApString(idOfTheResourceToRead, authToken);
			if(httpGetApString == null) {
				log.error("httpGetApString delivers NULL for " + idOfTheResourceToRead);
				httpGetApString = Optional.empty(); // TODO whats going wrong here ??
			} else {
				httpGetApString.ifPresent(x->ModelLogger.info(log, x, "readed:"));
			}
//		} catch (Exception e) {
//			log.error("error reading object:",e);
//		}
		
		return httpGetApString;
	}
	
	@Override
	public IRI getActorId() {
//		if(normalizedClientApplicationActor==null) {
//			normalizedClientApplicationActor = normalize(clientApplicationActor);
//		}
		return clientApplicationActor;
	}

	@Override
	public Model graphQueryOutbox(String query, String authToken) {
		Repository repo = getSparqlRepository(getSparqlEndpoint(RDFPUB.OUTBOX_SPARQL), authToken);
		return Repositories.graphQuery(repo, query, r -> QueryResults.asModel(r));
	}

	@Override
	public List<BindingSet> tupleQueryOutbox(String query, String authToken) {
		Repository repo = getSparqlRepository(getSparqlEndpoint(RDFPUB.OUTBOX_SPARQL), authToken);
		return Repositories.tupleQuery(repo, query, r -> QueryResults.asList(r));
	}

	@Override
	public Model getStatementsOutbox(String authToken, Resource subj, IRI pred, Value obj, Resource... contexts) {
		Repository repo = getSparqlRepository(getSparqlEndpoint(RDFPUB.OUTBOX_SPARQL), authToken);
		return getStatements(subj, pred, obj, repo, contexts);
	}
	
	private Model getStatements(Resource subj, IRI pred, Value obj, Repository repo, Resource... contexts) {
		try(RepositoryConnection con =  repo.getConnection()) {
			return QueryResults.asModel(con.getStatements(subj, pred, obj, contexts));
		}
	}

	private IRI getSparqlEndpoint(IRI profilePredicate) {
		Model profileModel = getProfile().orElseThrow();
		ModelLogger.debug(log, profileModel, "profileModel:");
		IRI sparqlEndpoint = Models.getPropertyIRI(profileModel, getActorId(), profilePredicate).orElseThrow(()->{
			ModelLogger.error(log, profileModel, "profile: ");
			return new RuntimeException("no '" + profilePredicate + "' for '" + getActorId() + "'");
			});
		return sparqlEndpoint;
	}
}
