package org.linkedopenactors.rdfpub.client;

import java.io.IOException;
import java.io.StringReader;
import java.time.Duration;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

import org.eclipse.rdf4j.model.IRI;
import org.eclipse.rdf4j.model.Model;
import org.eclipse.rdf4j.model.Resource;
import org.eclipse.rdf4j.model.Value;
import org.eclipse.rdf4j.model.util.Models;
import org.eclipse.rdf4j.query.QueryResults;
import org.eclipse.rdf4j.repository.Repository;
import org.eclipse.rdf4j.repository.RepositoryConnection;
import org.eclipse.rdf4j.repository.sparql.SPARQLRepository;
import org.eclipse.rdf4j.rio.RDFFormat;
import org.eclipse.rdf4j.rio.RDFParseException;
import org.eclipse.rdf4j.rio.Rio;
import org.eclipse.rdf4j.rio.UnsupportedRDFormatException;
import org.linkedopenactors.ns.rdfpub.RDFPUB;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.reactive.function.client.ClientResponse;
import org.springframework.web.reactive.function.client.ExchangeFilterFunction;
import org.springframework.web.reactive.function.client.WebClient;

import de.naturzukunft.rdf4j.utils.ModelLogger;
import de.naturzukunft.rdf4j.vocabulary.AS;
import lombok.extern.slf4j.Slf4j;
import reactor.core.publisher.Mono;
import reactor.util.retry.Retry;

/**
 * Default implementation of {@link RdfPubClient}.  
 * @author <a href="http://hauschel.de">SofwareEngineering Hauschel</a>
 */
@Slf4j
public abstract class RdfPubClientAbstract implements RdfPubClientAnonymous {
	
	protected static final MediaType MEDIA_TYPE_JSON_LD = new MediaType("application", "ld+json");
	protected WebClient webClient;
	private Map<IRI,SPARQLRepository> sparqlRepositories = new HashMap<>();
	private IRI rdfPubServerActorIri;
	
	/**
	 * @param webClient The webclient to use for http communication.
	 * @param rdfPubServerActorIri The iri/url of the server actor. Used to get the public sparql endpoint
	 */
	public RdfPubClientAbstract(IRI rdfPubServerActorIri, WebClient webClient) {
		this.rdfPubServerActorIri = rdfPubServerActorIri;
		this.webClient = webClient;
	}
	
	@Override
	public Optional<Model> read(IRI idOfTheResourceToRead) {
		log.trace("read(" + idOfTheResourceToRead + ")");
		return httpGetApString(idOfTheResourceToRead, null);
	}
	
	protected Optional<Model> httpGetApString(IRI idOfTheResourceToRead) {
		return httpGetApString(idOfTheResourceToRead, null);	
	}

	protected Optional<Model> httpGetApString(IRI idOfTheResourceToRead, String authToken) {
		var request = 
				webClient
				.get()
				.uri(idOfTheResourceToRead.stringValue())
				.accept(MEDIA_TYPE_JSON_LD)
				.header("profile", "https://www.w3.org/ns/activitystreams");
		Optional.ofNullable(authToken).ifPresent(token->request.header(HttpHeaders.AUTHORIZATION, token));
		log.debug("GET with access token !!");
		return request 
				.retrieve()
				.onStatus(HttpStatus::is5xxServerError, 
				          response -> Mono.error(new ServiceException("Server error", response.rawStatusCode())))
				.onStatus(HttpStatus::is4xxClientError, ClientResponse::createException)				
				.bodyToMono(String.class)
				.retryWhen(Retry.backoff(3, Duration.ofSeconds(2))
					.filter(throwable -> throwable instanceof ServiceException))
				.map(Optional::ofNullable)
				.map(httpResponseBodyOptional->
					{
						return httpResponseBodyOptional.map(httpResponseBody->{
							try {
								return Rio.parse(new StringReader(httpResponseBody), RDFFormat.JSONLD);
							} catch (RDFParseException | UnsupportedRDFormatException | IOException e1) {
								String message = "Error parsing httpResponseBody. " + e1.getMessage();								
								log.trace(message + ": " + httpResponseBody);								
								throw new RuntimeException(message, e1);
								}
							});
					})
				.block();
	}
	
	// This method returns filter function which will log request data
    private static ExchangeFilterFunction logRequest() {
        return ExchangeFilterFunction.ofRequestProcessor(clientRequest -> {
            log.info("Request: {} {}", clientRequest.method(), clientRequest.url());
            clientRequest.headers().forEach((name, values) -> values.forEach(value -> log.info("{}={}", name, value)));
            return Mono.just(clientRequest);
        });
    }
    
	protected SPARQLRepository getSparqlRepository(IRI sparqlEndpoint) {
		return getSparqlRepository(sparqlEndpoint, null);
	}
	
	protected SPARQLRepository getSparqlRepository(IRI sparqlEndpoint, String authToken) {
		log.trace("getSparqlRepository("+sparqlEndpoint+")");
		SPARQLRepository repo = sparqlRepositories.get(sparqlEndpoint);
		if(sparqlRepositories.containsKey(sparqlEndpoint)) {
			repo = sparqlRepositories.get(sparqlEndpoint);
		} else {
			repo = new SPARQLRepository(sparqlEndpoint.stringValue());
			sparqlRepositories.put(sparqlEndpoint, repo);
		}
		if(authToken!=null) {
			Map<String, String> additionalHttpHeaders = getAdditionalHttpHeaders(repo);
			if(!authToken.startsWith("Bearer ")) {
				authToken = "Bearer " + authToken;
			}
			additionalHttpHeaders.put("Authorization", authToken);
			repo.setAdditionalHttpHeaders(additionalHttpHeaders);
		}
		return repo;
	}
	
	private Map<String, String> getAdditionalHttpHeaders(SPARQLRepository repo) {
		Map<String, String> newAdditionalHttpHeaders = new HashMap<>();
		Map<String, String> additionalHttpHeaders = repo.getAdditionalHttpHeaders();
		if(additionalHttpHeaders!=null) {
			newAdditionalHttpHeaders.putAll(additionalHttpHeaders);
		}		
		return newAdditionalHttpHeaders;		
	}
	
	@Override
	public Model getStatementsPublic(Resource subj, IRI pred, Value obj, Resource... contexts) {
		Repository repo = getSparqlRepository(getPublicSparqlEndpoint());
		return getStatements(subj, pred, obj, repo, contexts);
	}

	private Model getStatements(Resource subj, IRI pred, Value obj, Repository repo, Resource... contexts) {
		try(RepositoryConnection con =  repo.getConnection()) {
			return QueryResults.asModel(con.getStatements(subj, pred, obj, contexts));
		}
	}
	
	private IRI getPublicSparqlEndpoint() {
		Model profile = read(rdfPubServerActorIri)
				.orElseThrow(()->new RuntimeException("no profile found for rdfPubServerActorIri: " + rdfPubServerActorIri));
		IRI endpoints = Models.getPropertyIRI(profile, rdfPubServerActorIri, AS.endpoints)
				.orElseThrow(()->{
					String message = "no "+AS.endpoints+" property found in profile of rdfPubServerActorIri: " + rdfPubServerActorIri;
					ModelLogger.error(log, profile, message);					
					return new RuntimeException(message);
					});
		Model endpointsModel = read(endpoints)
				.orElseThrow(()->new RuntimeException("no model found at endpoints: " + endpoints));
		return Models.getPropertyIRI(endpointsModel, endpoints, RDFPUB.PUBLIC_SPARQL)
				.orElseThrow(()->{
					String message = "no "+RDFPUB.PUBLIC_SPARQL+" found in model.";
					ModelLogger.error(log, profile, message);					
					return new RuntimeException(message); 
				});
	}
}
