package org.linkedopenactors.rdfpub.client;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.springframework.boot.json.JacksonJsonParser;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.web.reactive.function.BodyInserters;
import org.springframework.web.reactive.function.client.ClientResponse;
import org.springframework.web.reactive.function.client.WebClient;

import lombok.extern.slf4j.Slf4j;
import reactor.core.publisher.Mono;

/**
 * AccessTokenManager.
 * @author naturzukunft@mastodon.social
 *
 */
@Slf4j
public class AccessTokenManager {

	private WebClient webClient;
	private String keycloakServerUrl;
	private String clientApplicationRealm;
	private String clientApplicationClientId;
	private String clientApplicationClientSecret;
	private Map<String,Instant> authTokenCreationTimes = new HashMap<>(); 
	private Map<String,String> authTokens = new HashMap<>();
	private int authTokenLifetime;

	/**
	 * 
	 * @param webClient The webclient to use for http communication.
	 * @param keycloakServerUrl The base url of the auth server. E.g. http://localhost:8080/auth
	 * @param clientApplicationRealm The realm name of the keycloak server that contains the cliet application. E.g. LOA
	 * @param clientApplicationClientId The oauth2 client_id (The client_id is a public identifier for apps). 
	 * @param clientApplicationClientSecret The oauth2 client_secret (The client_secret is a secret known only to the application and the authorization server).
	 * @param authTokenLifetime The lifetime of the token
	 */
	public AccessTokenManager(
				WebClient webClient, 
				String keycloakServerUrl, 
				String clientApplicationRealm,
				String clientApplicationClientId, 
				String clientApplicationClientSecret, 
				int authTokenLifetime
			) {
		this.authTokenLifetime = authTokenLifetime;
		this.webClient = webClient;
		this.keycloakServerUrl = keycloakServerUrl;
		this.clientApplicationRealm = clientApplicationRealm;
		this.clientApplicationClientId = clientApplicationClientId;
		this.clientApplicationClientSecret = clientApplicationClientSecret;		
	}

	public synchronized String getAuthToken(String clientApplicationUserName, String clientApplicationPassword) {
		Instant authTokenCreationTime = authTokenCreationTimes.get(clientApplicationUserName);
		if( authTokenCreationTime == null || authTokenCreationTime.isBefore(Instant.now().minusSeconds(authTokenLifetime))) {
			log.info("REFRESHING authToken authTokenCreationTime: " + authTokenCreationTime + ", '"+clientApplicationRealm+"', clientApplicationClientId '"+clientApplicationClientId+"', clientApplicationUserName '"+clientApplicationUserName+"'");
			authTokenCreationTimes.put(clientApplicationUserName, Instant.now());			
			try {
				String authToken = "Bearer " + getTokenNonBlocking(clientApplicationRealm, clientApplicationClientId, clientApplicationUserName, clientApplicationPassword, Collections.emptyList());
				authTokens.put(clientApplicationUserName, authToken); 
			} catch (Exception e) {
				log.error("error getting token for clientApplicationRealm '"+clientApplicationRealm+"', clientApplicationClientId '"+clientApplicationClientId+"', clientApplicationUserName '"+clientApplicationUserName+"'");
				throw e;
			}
		}
    	return authTokens.get(clientApplicationUserName);
    }

	private String getTokenNonBlocking(String realm, String clientId, String username, String password, List<String> scopes) {
		HttpHeaders headers = new HttpHeaders();
		headers.setAccept(Arrays.asList(MediaType.APPLICATION_JSON));
		headers.add("Content-Type", "application/x-www-form-urlencoded");

		MultiValueMap<String, String> map = new LinkedMultiValueMap<String, String>();
		map.add("password",password);
		map.add("grant_type","password");
		map.add("username",username);
		map.add("client_id",clientId);
		map.add("client_secret",clientApplicationClientSecret);
		if(!scopes.isEmpty()) {
			map.add("scope",scopes.stream().collect(Collectors.joining(" ")));
		}
		String accessTokenUrl = keycloakServerUrl + "/realms/"+realm+"/protocol/openid-connect/token";
		log.trace("getting token for '"+username+"' ("+clientId+") from " + accessTokenUrl);
		JacksonJsonParser jsonParser = new JacksonJsonParser();
		Mono<String> token = webClient
				.post().uri(accessTokenUrl)
				.header("Content-Type", "application/x-www-form-urlencoded")
				.body(BodyInserters.fromFormData(map))
				.retrieve()
				.onStatus(org.springframework.http.HttpStatus::isError, ClientResponse::createException)
				.bodyToMono(String.class)
				.map(jsonParser::parseMap)
				.map(json->json.get("access_token").toString());
		String tokenStr = token.block(); 
		log.trace("got token for " + username); 
		return tokenStr;
	}
}
