/*
 * Decompiled with CFR 0.152.
 */
package org.komamitsu.fluency.buffer;

import java.io.Closeable;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.komamitsu.fluency.buffer.Buffer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileBackup {
    private static final Logger LOG = LoggerFactory.getLogger(FileBackup.class);
    private static final String PARAM_DELIM_IN_FILENAME = "#";
    private static final String EXT_FILENAME = ".buf";
    private final File backupDir;
    private final Buffer userBuffer;
    private final Pattern pattern;
    private final String prefix;

    public FileBackup(File backupDir, Buffer userBuffer, String prefix) {
        if (backupDir.mkdir()) {
            LOG.info("Created backupDir: dir={}", (Object)backupDir);
        }
        if (!(backupDir.isDirectory() && backupDir.canRead() && backupDir.canWrite())) {
            throw new IllegalArgumentException("backupDir[" + backupDir + "] needs to be a readable & writable directory");
        }
        this.backupDir = backupDir;
        this.userBuffer = userBuffer;
        this.prefix = prefix;
        this.pattern = Pattern.compile(userBuffer.bufferFormatType() + this.prefix() + PARAM_DELIM_IN_FILENAME + "([\\w\\." + PARAM_DELIM_IN_FILENAME + "]+)" + EXT_FILENAME);
        LOG.debug(this.toString());
    }

    private String prefix() {
        return this.prefix == null ? "" : "_" + this.prefix;
    }

    public String toString() {
        return "FileBackup{backupDir=" + this.backupDir + ", userBuffer=" + this.userBuffer + ", pattern=" + this.pattern + ", prefix='" + this.prefix + '\'' + '}';
    }

    public List<SavedBuffer> getSavedFiles() {
        File[] files = this.backupDir.listFiles();
        if (files == null) {
            LOG.warn("Failed to list the backup directory. {}", (Object)this.backupDir);
            return new ArrayList<SavedBuffer>();
        }
        LOG.debug("Checking backup files. files.length={}", (Object)files.length);
        ArrayList<SavedBuffer> savedBuffers = new ArrayList<SavedBuffer>();
        for (File f : files) {
            Matcher matcher = this.pattern.matcher(f.getName());
            if (matcher.find()) {
                if (matcher.groupCount() != 1) {
                    LOG.warn("Invalid backup filename: file={}", (Object)f.getName());
                    continue;
                }
                String concatParams = matcher.group(1);
                String[] params = concatParams.split(PARAM_DELIM_IN_FILENAME);
                LinkedList<String> paramList = new LinkedList<String>(Arrays.asList(params));
                LOG.debug("Saved buffer params={}", paramList);
                paramList.removeLast();
                savedBuffers.add(new SavedBuffer(f, paramList));
                continue;
            }
            LOG.trace("Found a file in backup dir, but the file path doesn't match the pattern. file={}", (Object)f.getAbsolutePath());
        }
        return savedBuffers;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void saveBuffer(List<String> params, ByteBuffer buffer) {
        ArrayList<String> copiedParams = new ArrayList<String>(params);
        copiedParams.add(String.valueOf(System.nanoTime()));
        boolean isFirst = true;
        StringBuilder sb = new StringBuilder();
        for (String param : copiedParams) {
            if (isFirst) {
                isFirst = false;
            } else {
                sb.append(PARAM_DELIM_IN_FILENAME);
            }
            sb.append(param);
        }
        String filename = this.userBuffer.bufferFormatType() + this.prefix() + PARAM_DELIM_IN_FILENAME + sb.toString() + EXT_FILENAME;
        File file = new File(this.backupDir, filename);
        LOG.debug("Backing up buffer: path={}, size={}", (Object)file.getAbsolutePath(), (Object)buffer.remaining());
        FileChannel channel = null;
        try {
            channel = new FileOutputStream(file).getChannel();
            channel.write(buffer);
        }
        catch (Exception e) {
            LOG.error("Failed to save buffer to file: params=" + copiedParams + ", path=" + file.getAbsolutePath() + ", buffer=" + buffer, (Throwable)e);
        }
        finally {
            if (channel != null) {
                try {
                    channel.close();
                }
                catch (IOException e) {
                    LOG.warn("Failed to close Channel: channel=" + channel);
                }
            }
        }
    }

    public static class SavedBuffer
    implements Closeable {
        private final List<String> params;
        private final File savedFile;
        private FileChannel channel;

        public SavedBuffer(File savedFile, List<String> params) {
            this.savedFile = savedFile;
            this.params = params;
        }

        public void open(Callback callback) {
            try {
                this.channel = new RandomAccessFile(this.savedFile, "rw").getChannel();
                callback.process(this.params, this.channel);
                this.success();
            }
            catch (Exception e) {
                LOG.error("Failed to process file. Skipping the file: file=" + this.savedFile, (Throwable)e);
            }
            finally {
                try {
                    this.close();
                }
                catch (IOException e) {
                    LOG.warn("Failed to close file: file=" + this.savedFile, (Throwable)e);
                }
            }
        }

        public void remove() {
            if (!this.savedFile.delete()) {
                LOG.warn("Failed to delete file: file=" + this.savedFile);
            }
        }

        private void success() {
            try {
                this.close();
            }
            catch (IOException e) {
                LOG.warn("Failed to close file: file=" + this.savedFile, (Throwable)e);
            }
            finally {
                this.remove();
            }
        }

        @Override
        public void close() throws IOException {
            if (this.channel != null && this.channel.isOpen()) {
                this.channel.close();
                this.channel = null;
            }
        }

        public static interface Callback {
            public void process(List<String> var1, FileChannel var2);
        }
    }
}

