/*
 * Decompiled with CFR 0.152.
 */
package org.komamitsu.fluency.buffer;

import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.atomic.AtomicLong;
import org.komamitsu.fluency.BufferFullException;
import org.komamitsu.fluency.EventTime;
import org.komamitsu.fluency.buffer.BufferPool;
import org.komamitsu.fluency.buffer.FileBackup;
import org.komamitsu.fluency.ingester.Ingester;
import org.komamitsu.fluency.recordformat.RecordFormatter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Buffer
implements Closeable {
    private static final Logger LOG = LoggerFactory.getLogger(Buffer.class);
    private final FileBackup fileBackup;
    private final RecordFormatter recordFormatter;
    private final Config config;
    private final Map<String, RetentionBuffer> retentionBuffers = new HashMap<String, RetentionBuffer>();
    private final LinkedBlockingQueue<TaggableBuffer> flushableBuffers = new LinkedBlockingQueue();
    private final Queue<TaggableBuffer> backupBuffers = new ConcurrentLinkedQueue<TaggableBuffer>();
    private final BufferPool bufferPool;

    public Buffer(RecordFormatter recordFormatter) {
        this(new Config(), recordFormatter);
    }

    public Buffer(Config config, RecordFormatter recordFormatter) {
        this.config = config;
        this.fileBackup = config.getFileBackupDir() != null ? new FileBackup(new File(config.getFileBackupDir()), this, config.getFileBackupPrefix()) : null;
        this.recordFormatter = recordFormatter;
        if (config.getChunkInitialSize() > config.getChunkRetentionSize()) {
            LOG.warn("Initial Buffer Chunk Size ({}) shouldn't be more than Buffer Chunk Retention Size ({}) for better performance.", (Object)config.getChunkInitialSize(), (Object)config.getChunkRetentionSize());
        }
        this.bufferPool = new BufferPool(config.getChunkInitialSize(), config.getMaxBufferSize(), config.jvmHeapBufferMode);
        this.init();
    }

    private void init() {
        if (this.fileBackup != null) {
            for (FileBackup.SavedBuffer savedBuffer : this.fileBackup.getSavedFiles()) {
                savedBuffer.open((params, channel) -> {
                    try {
                        LOG.info("Loading buffer: params={}, buffer.size={}", (Object)params, (Object)channel.size());
                    }
                    catch (IOException e) {
                        LOG.error("Failed to access the backup file: params={}", (Object)params, (Object)e);
                    }
                    this.loadBufferFromFile(params, channel);
                });
            }
        }
    }

    protected void saveBuffer(List<String> params, ByteBuffer buffer) {
        if (this.fileBackup == null) {
            return;
        }
        LOG.info("Saving buffer: params={}, buffer={}", (Object)params, (Object)buffer);
        this.fileBackup.saveBuffer(params, buffer);
    }

    public void flush(Ingester ingester, boolean force) throws IOException {
        LOG.trace("flush(): force={}, bufferUsage={}", (Object)force, (Object)Float.valueOf(this.getBufferUsage()));
        this.flushInternal(ingester, force);
    }

    @Override
    public void close() {
        try {
            LOG.debug("Saving all buffers");
            this.saveAllBuffersToFile();
        }
        catch (Exception e) {
            LOG.warn("Failed to save all buffers", e);
        }
        LOG.debug("Closing buffers");
        this.closeInternal();
    }

    private long getMaxSize() {
        return this.config.getMaxBufferSize();
    }

    public float getBufferUsage() {
        return (float)this.getAllocatedSize() / (float)this.getMaxSize();
    }

    public void clearBackupFiles() {
        if (this.fileBackup != null) {
            for (FileBackup.SavedBuffer buffer : this.fileBackup.getSavedFiles()) {
                buffer.remove();
            }
        }
    }

    public long getMaxBufferSize() {
        return this.config.getMaxBufferSize();
    }

    public String getFileBackupPrefix() {
        return this.config.getFileBackupPrefix();
    }

    public String getFileBackupDir() {
        return this.config.getFileBackupDir();
    }

    private RetentionBuffer prepareBuffer(String tag, int writeSize) throws BufferFullException {
        int newBufferChunkRetentionSize;
        RetentionBuffer retentionBuffer = this.retentionBuffers.get(tag);
        if (retentionBuffer != null && retentionBuffer.getByteBuffer().remaining() > writeSize) {
            return retentionBuffer;
        }
        int existingDataSize = 0;
        if (retentionBuffer == null) {
            newBufferChunkRetentionSize = this.config.getChunkInitialSize();
        } else {
            existingDataSize = retentionBuffer.getByteBuffer().position();
            newBufferChunkRetentionSize = (int)((float)retentionBuffer.getByteBuffer().capacity() * this.config.getChunkExpandRatio());
        }
        while (newBufferChunkRetentionSize < writeSize + existingDataSize) {
            newBufferChunkRetentionSize = (int)((float)newBufferChunkRetentionSize * this.config.getChunkExpandRatio());
        }
        ByteBuffer acquiredBuffer = this.bufferPool.acquireBuffer(newBufferChunkRetentionSize);
        if (acquiredBuffer == null) {
            throw new BufferFullException("Buffer is full. config=" + this.config + ", bufferPool=" + this.bufferPool);
        }
        RetentionBuffer newBuffer = new RetentionBuffer(acquiredBuffer);
        if (retentionBuffer != null) {
            retentionBuffer.getByteBuffer().flip();
            newBuffer.getByteBuffer().put(retentionBuffer.getByteBuffer());
            newBuffer.getCreatedTimeMillis().set(System.currentTimeMillis());
            this.bufferPool.returnBuffer(retentionBuffer.getByteBuffer());
        }
        LOG.trace("prepareBuffer(): allocate a new buffer. tag={}, buffer={}", (Object)tag, (Object)newBuffer);
        this.retentionBuffers.put(tag, newBuffer);
        return newBuffer;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void loadDataToRetentionBuffers(String tag, ByteBuffer src) throws IOException {
        Map<String, RetentionBuffer> map = this.retentionBuffers;
        synchronized (map) {
            RetentionBuffer buffer = this.prepareBuffer(tag, src.remaining());
            buffer.getByteBuffer().put(src);
            this.moveRetentionBufferIfNeeded(tag, buffer);
        }
    }

    protected void loadBufferFromFile(List<String> params, FileChannel channel) {
        if (params.size() != 1) {
            throw new IllegalArgumentException("The number of params should be 1: params=" + params);
        }
        String tag = params.get(0);
        try {
            MappedByteBuffer src = channel.map(FileChannel.MapMode.PRIVATE, 0L, channel.size());
            this.loadDataToRetentionBuffers(tag, src);
        }
        catch (Exception e) {
            LOG.error("Failed to load data to flushableBuffers: params={}, channel={}", (Object)params, (Object)channel);
        }
    }

    private void saveBuffer(TaggableBuffer buffer) {
        this.saveBuffer(Collections.singletonList(buffer.getTag()), buffer.getByteBuffer());
    }

    protected void saveAllBuffersToFile() throws IOException {
        TaggableBuffer flushableBuffer;
        this.moveRetentionBuffersToFlushable(true);
        while ((flushableBuffer = this.flushableBuffers.poll()) != null) {
            this.saveBuffer(flushableBuffer);
        }
        while ((flushableBuffer = this.backupBuffers.poll()) != null) {
            this.saveBuffer(flushableBuffer);
        }
    }

    private void appendMapInternal(String tag, Object timestamp, Map<String, Object> data) throws IOException {
        this.loadDataToRetentionBuffers(tag, ByteBuffer.wrap(this.recordFormatter.format(tag, timestamp, data)));
    }

    private void appendMessagePackMapValueInternal(String tag, Object timestamp, byte[] mapValue, int offset, int len) throws IOException {
        this.loadDataToRetentionBuffers(tag, ByteBuffer.wrap(this.recordFormatter.formatFromMessagePack(tag, timestamp, mapValue, offset, len)));
    }

    private void appendMessagePackMapValueInternal(String tag, Object timestamp, ByteBuffer mapValue) throws IOException {
        this.loadDataToRetentionBuffers(tag, ByteBuffer.wrap(this.recordFormatter.formatFromMessagePack(tag, timestamp, mapValue)));
    }

    public void append(String tag, long timestamp, Map<String, Object> data) throws IOException {
        this.appendMapInternal(tag, timestamp, data);
    }

    public void append(String tag, EventTime timestamp, Map<String, Object> data) throws IOException {
        this.appendMapInternal(tag, timestamp, data);
    }

    public void appendMessagePackMapValue(String tag, long timestamp, byte[] mapValue, int offset, int len) throws IOException {
        this.appendMessagePackMapValueInternal(tag, timestamp, mapValue, offset, len);
    }

    public void appendMessagePackMapValue(String tag, EventTime timestamp, byte[] mapValue, int offset, int len) throws IOException {
        this.appendMessagePackMapValueInternal(tag, timestamp, mapValue, offset, len);
    }

    public void appendMessagePackMapValue(String tag, long timestamp, ByteBuffer mapValue) throws IOException {
        this.appendMessagePackMapValueInternal(tag, timestamp, mapValue);
    }

    public void appendMessagePackMapValue(String tag, EventTime timestamp, ByteBuffer mapValue) throws IOException {
        this.appendMessagePackMapValueInternal(tag, timestamp, mapValue);
    }

    private void moveRetentionBufferIfNeeded(String tag, RetentionBuffer buffer) throws IOException {
        if (buffer.getByteBuffer().position() > this.config.getChunkRetentionSize()) {
            this.moveRetentionBufferToFlushable(tag, buffer);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void moveRetentionBuffersToFlushable(boolean force) throws IOException {
        long expiredThreshold = System.currentTimeMillis() - (long)this.config.getChunkRetentionTimeMillis();
        Map<String, RetentionBuffer> map = this.retentionBuffers;
        synchronized (map) {
            for (Map.Entry<String, RetentionBuffer> entry : this.retentionBuffers.entrySet()) {
                if (entry.getValue() == null || !force && entry.getValue().getCreatedTimeMillis().get() >= expiredThreshold) continue;
                this.moveRetentionBufferToFlushable(entry.getKey(), entry.getValue());
            }
        }
    }

    private void moveRetentionBufferToFlushable(String tag, RetentionBuffer buffer) throws IOException {
        try {
            LOG.trace("moveRetentionBufferToFlushable(): tag={}, buffer={}", (Object)tag, (Object)buffer);
            buffer.getByteBuffer().flip();
            this.flushableBuffers.put(new TaggableBuffer(tag, buffer.getByteBuffer()));
            this.retentionBuffers.put(tag, null);
        }
        catch (InterruptedException e) {
            throw new IOException("Failed to move retention buffer due to interruption", e);
        }
    }

    protected void flushInternal(Ingester ingester, boolean force) throws IOException {
        TaggableBuffer flushableBuffer;
        this.moveRetentionBuffersToFlushable(force);
        while (!Thread.currentThread().isInterrupted() && (flushableBuffer = this.flushableBuffers.poll()) != null) {
            boolean keepBuffer = false;
            try {
                LOG.trace("flushInternal(): bufferUsage={}, flushableBuffer={}", (Object)Float.valueOf(this.getBufferUsage()), (Object)flushableBuffer);
                String tag = flushableBuffer.getTag();
                ByteBuffer dataBuffer = flushableBuffer.getByteBuffer();
                ingester.ingest(tag, dataBuffer);
            }
            catch (IOException e) {
                LOG.warn("Failed to send data. The data is going to be saved into the buffer again: data={}", (Object)flushableBuffer);
                keepBuffer = true;
                throw e;
            }
            finally {
                if (keepBuffer) {
                    try {
                        this.flushableBuffers.put(flushableBuffer);
                    }
                    catch (InterruptedException e1) {
                        LOG.warn("Failed to save the data into the buffer. Trying to save it in extra buffer: chunk={}", (Object)flushableBuffer);
                        this.backupBuffers.add(flushableBuffer);
                    }
                    continue;
                }
                this.bufferPool.returnBuffer(flushableBuffer.getByteBuffer());
            }
        }
    }

    protected synchronized void closeInternal() {
        this.retentionBuffers.clear();
        this.bufferPool.releaseBuffers();
    }

    public long getAllocatedSize() {
        return this.bufferPool.getAllocatedSize();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public long getBufferedDataSize() {
        long size = 0L;
        Map<String, RetentionBuffer> map = this.retentionBuffers;
        synchronized (map) {
            for (Map.Entry<String, RetentionBuffer> buffer : this.retentionBuffers.entrySet()) {
                if (buffer.getValue() == null || buffer.getValue().getByteBuffer() == null) continue;
                size += (long)buffer.getValue().getByteBuffer().position();
            }
        }
        for (TaggableBuffer buffer : this.flushableBuffers) {
            if (buffer.getByteBuffer() == null) continue;
            size += (long)buffer.getByteBuffer().remaining();
        }
        return size;
    }

    public boolean getJvmHeapBufferMode() {
        return this.bufferPool.getJvmHeapBufferMode();
    }

    public String bufferFormatType() {
        return "packed_forward";
    }

    public int getChunkInitialSize() {
        return this.config.getChunkInitialSize();
    }

    public float getChunkExpandRatio() {
        return this.config.getChunkExpandRatio();
    }

    public int getChunkRetentionSize() {
        return this.config.getChunkRetentionSize();
    }

    public int getChunkRetentionTimeMillis() {
        return this.config.getChunkRetentionTimeMillis();
    }

    public String toString() {
        return "PackedForwardBuffer{retentionBuffers=" + this.retentionBuffers + ", flushableBuffers=" + this.flushableBuffers + ", backupBuffers=" + this.backupBuffers + ", bufferPool=" + this.bufferPool + ", config=" + this.config + "} " + super.toString();
    }

    public static class Config {
        private long maxBufferSize = 0x20000000L;
        private String fileBackupDir;
        private String fileBackupPrefix;
        private int chunkInitialSize = 0x100000;
        private float chunkExpandRatio = 2.0f;
        private int chunkRetentionSize = 0x400000;
        private int chunkRetentionTimeMillis = 1000;
        private boolean jvmHeapBufferMode = false;

        public long getMaxBufferSize() {
            return this.maxBufferSize;
        }

        public void setMaxBufferSize(long maxBufferSize) {
            this.maxBufferSize = maxBufferSize;
        }

        public String getFileBackupDir() {
            return this.fileBackupDir;
        }

        public void setFileBackupDir(String fileBackupDir) {
            this.fileBackupDir = fileBackupDir;
        }

        public String getFileBackupPrefix() {
            return this.fileBackupPrefix;
        }

        public void setFileBackupPrefix(String fileBackupPrefix) {
            this.fileBackupPrefix = fileBackupPrefix;
        }

        public int getChunkInitialSize() {
            return this.chunkInitialSize;
        }

        public void setChunkInitialSize(int chunkInitialSize) {
            this.chunkInitialSize = chunkInitialSize;
        }

        public float getChunkExpandRatio() {
            return this.chunkExpandRatio;
        }

        public void setChunkExpandRatio(float chunkExpandRatio) {
            this.chunkExpandRatio = chunkExpandRatio;
        }

        public int getChunkRetentionSize() {
            return this.chunkRetentionSize;
        }

        public void setChunkRetentionSize(int chunkRetentionSize) {
            this.chunkRetentionSize = chunkRetentionSize;
        }

        public int getChunkRetentionTimeMillis() {
            return this.chunkRetentionTimeMillis;
        }

        public void setChunkRetentionTimeMillis(int chunkRetentionTimeMillis) {
            this.chunkRetentionTimeMillis = chunkRetentionTimeMillis;
        }

        public boolean getJvmHeapBufferMode() {
            return this.jvmHeapBufferMode;
        }

        public void setJvmHeapBufferMode(boolean jvmHeapBufferMode) {
            this.jvmHeapBufferMode = jvmHeapBufferMode;
        }

        public String toString() {
            return "Config{maxBufferSize=" + this.maxBufferSize + ", fileBackupDir='" + this.fileBackupDir + '\'' + ", fileBackupPrefix='" + this.fileBackupPrefix + '\'' + ", chunkInitialSize=" + this.chunkInitialSize + ", chunkExpandRatio=" + this.chunkExpandRatio + ", chunkRetentionSize=" + this.chunkRetentionSize + ", chunkRetentionTimeMillis=" + this.chunkRetentionTimeMillis + ", jvmHeapBufferMode=" + this.jvmHeapBufferMode + '}';
        }
    }

    private static class TaggableBuffer {
        private final String tag;
        private final ByteBuffer byteBuffer;

        public TaggableBuffer(String tag, ByteBuffer byteBuffer) {
            this.tag = tag;
            this.byteBuffer = byteBuffer;
        }

        public String getTag() {
            return this.tag;
        }

        public ByteBuffer getByteBuffer() {
            return this.byteBuffer;
        }

        public String toString() {
            return "TaggableBuffer{tag='" + this.tag + '\'' + ", byteBuffer=" + this.byteBuffer + '}';
        }
    }

    private static class RetentionBuffer {
        private final AtomicLong createdTimeMillis = new AtomicLong();
        private final ByteBuffer byteBuffer;

        RetentionBuffer(ByteBuffer byteBuffer) {
            this.byteBuffer = byteBuffer;
        }

        AtomicLong getCreatedTimeMillis() {
            return this.createdTimeMillis;
        }

        ByteBuffer getByteBuffer() {
            return this.byteBuffer;
        }

        public String toString() {
            return "RetentionBuffer{createdTimeMillis=" + this.createdTimeMillis + ", byteBuffer=" + this.byteBuffer + '}';
        }
    }
}

