/*
 * Copyright 2010-2013 Ning, Inc.
 * Copyright 2014 Groupon, Inc
 * Copyright 2014 The Billing Project, LLC
 *
 * The Billing Project licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

package org.killbill.billing.payment.api;

import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

import org.killbill.billing.account.api.Account;
import org.killbill.billing.security.RequiresPermissions;
import org.killbill.billing.util.callcontext.CallContext;
import org.killbill.billing.util.callcontext.TenantContext;
import org.killbill.billing.util.entity.Pagination;

import static org.killbill.billing.security.Permission.INVOICE_CAN_ADJUST;
import static org.killbill.billing.security.Permission.INVOICE_CAN_ITEM_ADJUST;
import static org.killbill.billing.security.Permission.PAYMENT_CAN_CREATE_EXTERNAL_PAYMENT;
import static org.killbill.billing.security.Permission.PAYMENT_CAN_REFUND;
import static org.killbill.billing.security.Permission.PAYMENT_CAN_TRIGGER_PAYMENT;

public interface PaymentApi {

    /**
     * @param account    the account
     * @param invoiceId  the invoice id
     * @param amount     the amount to pay
     * @param properties plugin specific properties
     * @param context    the call context
     * @return the payment
     * @throws PaymentApiException
     */
    @RequiresPermissions(PAYMENT_CAN_TRIGGER_PAYMENT)
    public Payment createPayment(Account account, UUID invoiceId, BigDecimal amount, Iterable<PluginProperty> properties, CallContext context)
            throws PaymentApiException;

    /**
     * @param account   the account
     * @param invoiceId the invoice id
     * @param amount    the amount to pay
     * @param context   the call context
     * @return the payment
     * @throws PaymentApiException
     */
    @RequiresPermissions(PAYMENT_CAN_CREATE_EXTERNAL_PAYMENT)
    public Payment createExternalPayment(Account account, UUID invoiceId, BigDecimal amount, CallContext context)
            throws PaymentApiException;

    /**
     * @param account   the account
     * @param paymentId the payment id
     * @param isSuccess whether pending payment turned out to be successful
     * @param context   the call context
     */
    public void notifyPendingPaymentOfStateChanged(Account account, UUID paymentId, boolean isSuccess, CallContext context)
            throws PaymentApiException;

    /**
     * @param account    the account
     * @param paymentId  the payment id
     * @param properties plugin specific properties
     * @param context
     * @return
     * @throws PaymentApiException
     */
    public Payment retryPayment(Account account, UUID paymentId, Iterable<PluginProperty> properties, CallContext context)
            throws PaymentApiException;

    /**
     * Create a refund for a given payment. The associated invoice is not adjusted.
     *
     * @param account      account to refund
     * @param paymentId    payment associated with that refund
     * @param refundAmount amount to refund
     * @param properties   plugin specific properties
     * @param context      the call context
     * @return the created Refund
     * @throws PaymentApiException
     */
    @RequiresPermissions(PAYMENT_CAN_REFUND)
    public Refund createRefund(Account account, UUID paymentId, BigDecimal refundAmount, Iterable<PluginProperty> properties, CallContext context)
            throws PaymentApiException;

    /**
     * @param account   the account
     * @param paymentId the payment id
     * @param isSuccess whether pending refund turned out to be successful
     * @param context   the call context
     */
    public void notifyPendingRefundOfStateChanged(Account account, UUID paymentId, boolean isSuccess, CallContext context)
            throws PaymentApiException;

    /**
     * @param refundId       the refund id
     * @param withPluginInfo whether to fetch plugin info
     * @param properties     plugin specific properties
     * @param context        the call context  @return the refund
     * @throws PaymentApiException
     */
    public Refund getRefund(UUID refundId, boolean withPluginInfo, Iterable<PluginProperty> properties, TenantContext context)
            throws PaymentApiException;

    /**
     * Create a refund for a given payment. The associated invoice is adjusted.
     *
     * @param account      account to refund
     * @param paymentId    payment associated with that refund
     * @param refundAmount amount to refund
     * @param properties   plugin specific properties
     * @param context      the call context
     * @return the created Refund
     * @throws PaymentApiException
     */
    @RequiresPermissions({PAYMENT_CAN_REFUND, INVOICE_CAN_ADJUST})
    public Refund createRefundWithAdjustment(Account account, UUID paymentId, BigDecimal refundAmount, Iterable<PluginProperty> properties, CallContext context)
            throws PaymentApiException;

    /**
     * Create a refund for a given payment. The specified invoice items are fully adjusted.
     * The refund amount will be the sum of all invoice items amounts.
     *
     * @param account        account to refund
     * @param paymentId      payment associated with that refund
     * @param invoiceItemIds invoice item ids to adjust
     * @param properties     plugin specific properties
     * @param context        the call context
     * @return the created Refund
     * @throws PaymentApiException
     */
    @RequiresPermissions({PAYMENT_CAN_REFUND, INVOICE_CAN_ITEM_ADJUST})
    public Refund createRefundWithItemsAdjustments(Account account, UUID paymentId, Set<UUID> invoiceItemIds, Iterable<PluginProperty> properties, CallContext context)
            throws PaymentApiException;

    /**
     * Create a refund for a given payment. The specified invoice items are partially adjusted.
     * The refund amount will be the sum of all adjustments.
     *
     * @param account                   account to refund
     * @param paymentId                 payment associated with that refund
     * @param invoiceItemIdsWithAmounts invoice item ids and associated amounts to adjust
     * @param properties                plugin specific properties
     * @param context                   the call context
     * @return the created Refund
     * @throws PaymentApiException
     */
    @RequiresPermissions({PAYMENT_CAN_REFUND, INVOICE_CAN_ITEM_ADJUST})
    public Refund createRefundWithItemsAdjustments(Account account, UUID paymentId, Map<UUID, BigDecimal> invoiceItemIdsWithAmounts, Iterable<PluginProperty> properties, CallContext context)
            throws PaymentApiException;

    /**
     * @param account the account
     * @param context the call context
     * @return the list of refund on this account
     * @throws PaymentApiException
     */
    public List<Refund> getAccountRefunds(Account account, TenantContext context)
            throws PaymentApiException;

    /**
     * @param paymentId the payment id
     * @param context   the call context
     * @return the list of refund on this account
     * @throws PaymentApiException
     */
    public List<Refund> getPaymentRefunds(UUID paymentId, TenantContext context)
            throws PaymentApiException;

    /**
     * @param invoiceId the invoice id
     * @param context   the call context
     * @return the list of payment on this invoice
     * @throws PaymentApiException
     */
    public List<Payment> getInvoicePayments(UUID invoiceId, TenantContext context)
            throws PaymentApiException;

    /**
     * @param accountId the account id
     * @param context   the call context
     * @return the list of payment on this account
     * @throws PaymentApiException
     */
    public List<Payment> getAccountPayments(UUID accountId, TenantContext context)
            throws PaymentApiException;

    /**
     * @param paymentId      the payment id
     * @param withPluginInfo whether to fetch plugin info
     * @param properties     plugin specific properties
     * @param context        the call context
     * @return the payment
     * @throws PaymentApiException
     */
    public Payment getPayment(UUID paymentId, final boolean withPluginInfo, Iterable<PluginProperty> properties, TenantContext context)
            throws PaymentApiException;

    /**
     * Find all payments across all plugins
     *
     * @param offset     the offset of the first result
     * @param limit      the maximum number of results to retrieve
     * @param properties plugin specific properties
     * @param context    the user context
     * @return the list of payments for that tenant
     */
    public Pagination<Payment> getPayments(Long offset, Long limit, Iterable<PluginProperty> properties, TenantContext context);

    /**
     * Find all payments in a given plugin
     *
     * @param offset     the offset of the first result
     * @param limit      the maximum number of results to retrieve
     * @param pluginName the payment plugin name
     * @param properties plugin specific properties
     * @param context    the user context
     * @return the list of payments for that tenant
     * @throws PaymentApiException
     */
    public Pagination<Payment> getPayments(Long offset, Long limit, String pluginName, Iterable<PluginProperty> properties, TenantContext context) throws PaymentApiException;

    /**
     * Find all payments matching the search key across all plugins
     * <p/>
     * The match will be plugin specific: for instance some plugins will try to match the key
     * against the transaction ids, etc.
     *
     * @param searchKey  the search key
     * @param offset     the offset of the first result
     * @param limit      the maximum number of results to retrieve
     * @param properties plugin specific properties
     * @param context    the user context
     * @return the list of payments matching this search key for that tenant
     */
    public Pagination<Payment> searchPayments(String searchKey, Long offset, Long limit, Iterable<PluginProperty> properties, TenantContext context);

    /**
     * Find all payments matching the search key in a given plugin
     * <p/>
     * The match will be plugin specific: for instance some plugins will try to match the key
     * against the transaction ids, etc.
     *
     * @param searchKey  the search key
     * @param offset     the offset of the first result
     * @param limit      the maximum number of results to retrieve
     * @param pluginName the payment plugin name
     * @param properties plugin specific properties
     * @param context    the user context
     * @return the list of payments matching this search key for that tenant
     * @throws PaymentApiException
     */
    public Pagination<Payment> searchPayments(String searchKey, Long offset, Long limit, String pluginName, Iterable<PluginProperty> properties, TenantContext context) throws PaymentApiException;

    /**
     * Find all refunds across all plugins
     *
     * @param offset     the offset of the first result
     * @param limit      the maximum number of results to retrieve
     * @param properties plugin specific properties
     * @param context    the user context
     * @return the list of refunds for that tenant
     */
    public Pagination<Refund> getRefunds(Long offset, Long limit, Iterable<PluginProperty> properties, TenantContext context);

    /**
     * Find all refunds in a given plugin
     *
     * @param offset     the offset of the first result
     * @param limit      the maximum number of results to retrieve
     * @param pluginName the refund plugin name
     * @param properties plugin specific properties
     * @param context    the user context
     * @return the list of refunds for that tenant
     * @throws PaymentApiException
     */
    public Pagination<Refund> getRefunds(Long offset, Long limit, String pluginName, Iterable<PluginProperty> properties, TenantContext context) throws PaymentApiException;

    /**
     * Find all refunds matching the search key across all plugins
     * <p/>
     * The match will be plugin specific: for instance some plugins will try to match the key
     * against the transaction ids, etc.
     *
     * @param searchKey  the search key
     * @param offset     the offset of the first result
     * @param limit      the maximum number of results to retrieve
     * @param properties plugin specific properties
     * @param context    the user context
     * @return the list of refunds matching this search key for that tenant
     */
    public Pagination<Refund> searchRefunds(String searchKey, Long offset, Long limit, Iterable<PluginProperty> properties, TenantContext context);

    /**
     * Find all refunds matching the search key in a given plugin
     * <p/>
     * The match will be plugin specific: for instance some plugins will try to match the key
     * against the transaction ids, etc.
     *
     * @param searchKey  the search key
     * @param offset     the offset of the first result
     * @param limit      the maximum number of results to retrieve
     * @param properties plugin specific properties
     * @param pluginName the refund plugin name
     * @param context    the user context
     * @return the list of refunds matching this search key for that tenant
     * @throws PaymentApiException
     */
    public Pagination<Refund> searchRefunds(String searchKey, Long offset, Long limit, String pluginName, Iterable<PluginProperty> properties, TenantContext context) throws PaymentApiException;

    /**
     * @return a list of all the payment plugins registered
     */
    public Set<String> getAvailablePlugins();

    /**
     * @param pluginName        the plugin name
     * @param account           the account
     * @param setDefault        whether this should be set as a default payment method
     * @param paymentMethodInfo the details for the payment method
     * @param properties        plugin specific properties
     * @param context           the call context
     * @return the uuid of the payment method
     * @throws PaymentApiException
     */
    public UUID addPaymentMethod(String pluginName, Account account, boolean setDefault, PaymentMethodPlugin paymentMethodInfo, Iterable<PluginProperty> properties, CallContext context)
            throws PaymentApiException;

    /**
     * @param account        the account id
     * @param withPluginInfo whether we want to retrieve the plugin info for that payment method
     * @param properties     plugin specific properties
     * @param context        the call context
     * @return the list of payment methods
     * @throws PaymentApiException
     */
    public List<PaymentMethod> getPaymentMethods(Account account, final boolean withPluginInfo, Iterable<PluginProperty> properties, TenantContext context)
            throws PaymentApiException;

    /**
     * @param paymentMethodId  the payment methid id
     * @param includedInactive returns the payment method even if this is not active
     * @param withPluginInfo   whether we want to retrieve the plugin info for that payment method
     * @param properties       plugin specific properties
     * @param context          the call context   @return the payment method
     * @throws PaymentApiException
     */
    public PaymentMethod getPaymentMethodById(UUID paymentMethodId, final boolean includedInactive, final boolean withPluginInfo, Iterable<PluginProperty> properties, TenantContext context)
            throws PaymentApiException;

    /**
     * Find all payment methods across all plugins
     *
     * @param offset     the offset of the first result
     * @param limit      the maximum number of results to retrieve
     * @param properties plugin specific properties
     * @param context    the user context
     * @return the list of payment methods for that tenant
     */
    public Pagination<PaymentMethod> getPaymentMethods(Long offset, Long limit, Iterable<PluginProperty> properties, TenantContext context);

    /**
     * Find all payment methods in a given plugin
     *
     * @param offset     the offset of the first result
     * @param limit      the maximum number of results to retrieve
     * @param pluginName the payment plugin name
     * @param properties plugin specific properties
     * @param context    the user context
     * @return the list of payment methods for that tenant
     * @throws PaymentApiException
     */
    public Pagination<PaymentMethod> getPaymentMethods(Long offset, Long limit, String pluginName, Iterable<PluginProperty> properties, TenantContext context) throws PaymentApiException;

    /**
     * Find all payment methods matching the search key across all plugins
     * <p/>
     * The match will be plugin specific: for instance some plugins will try to match the key
     * against the last 4 credit cards digits, agreement ids, etc.
     *
     * @param searchKey  the search key
     * @param offset     the offset of the first result
     * @param limit      the maximum number of results to retrieve
     * @param properties plugin specific properties
     * @param context    the user context
     * @return the list of payment methods matching this search key for that tenant
     */
    public Pagination<PaymentMethod> searchPaymentMethods(String searchKey, Long offset, Long limit, Iterable<PluginProperty> properties, TenantContext context);

    /**
     * Find all payment methods matching the search key in a given plugin
     * <p/>
     * The match will be plugin specific: for instance some plugins will try to match the key
     * against the last 4 credit cards digits, agreement ids, etc.
     *
     * @param searchKey  the search key
     * @param offset     the offset of the first result
     * @param limit      the maximum number of results to retrieve
     * @param pluginName the payment plugin name
     * @param properties plugin specific properties
     * @param context    the user context
     * @return the list of payment methods matching this search key for that tenant
     * @throws PaymentApiException
     */
    public Pagination<PaymentMethod> searchPaymentMethods(String searchKey, Long offset, Long limit, String pluginName, Iterable<PluginProperty> properties, TenantContext context) throws PaymentApiException;

    /**
     * @param account                                  the account
     * @param paymentMethodId                          the id of the payment  method
     * @param deleteDefaultPaymentMethodWithAutoPayOff whether to allow deletion of default payment method and set account into AUTO_PAY_OFF
     * @param properties                               plugin specific properties
     * @param context                                  the call context
     * @throws PaymentApiException
     */
    public void deletedPaymentMethod(Account account, UUID paymentMethodId, boolean deleteDefaultPaymentMethodWithAutoPayOff, Iterable<PluginProperty> properties, CallContext context)
            throws PaymentApiException;

    /**
     * @param account         the account
     * @param paymentMethodId the payment method id
     * @param properties      plugin specific properties
     * @param context         the call context
     * @throws PaymentApiException
     */
    public void setDefaultPaymentMethod(Account account, UUID paymentMethodId, Iterable<PluginProperty> properties, CallContext context)
            throws PaymentApiException;

    /**
     * @param pluginName the name of the plugin
     * @param account    the account
     * @param properties plugin specific properties
     * @param context    the call context
     * @return the list of payment methods for that account
     * @throws PaymentApiException
     */
    public List<PaymentMethod> refreshPaymentMethods(String pluginName, Account account, Iterable<PluginProperty> properties, CallContext context)
            throws PaymentApiException;

    /**
     * Refresh all payment methods across all plugins
     * <p/>
     * This call is not atomic.
     *
     * @param account    the account
     * @param properties plugin specific properties
     * @param context    the call context
     * @return the list of payment methods for that account
     * @throws PaymentApiException
     */
    public List<PaymentMethod> refreshPaymentMethods(Account account, Iterable<PluginProperty> properties, CallContext context)
            throws PaymentApiException;

}
