/*
 * Copyright 2021 Red Hat, Inc. and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.acme.deals;

import java.net.URI;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import com.fasterxml.jackson.databind.jsonschema.JsonSchema;
import org.jbpm.util.JsonSchemaUtil;
import org.kie.kogito.Application;
import org.kie.kogito.process.Process;
import org.kie.kogito.process.ProcessInstance;
import org.kie.kogito.process.ProcessInstanceReadMode;
import org.kie.kogito.process.impl.Sig;
import org.kie.kogito.process.ProcessInstanceExecutionException;
import org.kie.kogito.process.WorkItem;
import org.kie.kogito.process.workitem.Attachment;
import org.kie.kogito.process.workitem.AttachmentInfo;
import org.kie.kogito.process.workitem.Comment;
import org.kie.kogito.process.workitem.Policies;
import org.kie.kogito.process.workitem.TaskModel;
import org.kie.kogito.services.uow.UnitOfWorkExecutor;
import org.jbpm.process.instance.impl.humantask.HumanTaskHelper;
import org.jbpm.process.instance.impl.humantask.HumanTaskTransition;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.http.ResponseEntity;
import org.springframework.web.util.UriComponents;
import org.springframework.web.util.UriComponentsBuilder;
import org.acme.deals.DealreviewsModelOutput;

@RestController
@RequestMapping("/dealreviews")
@org.springframework.stereotype.Component()
public class DealreviewsResource {

    @org.springframework.beans.factory.annotation.Autowired()
    @org.springframework.beans.factory.annotation.Qualifier("dealreviews")
    Process<DealreviewsModel> process;

    @org.springframework.beans.factory.annotation.Autowired()
    Application application;

    @PostMapping(produces = MediaType.APPLICATION_JSON_VALUE, consumes = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<DealreviewsModelOutput> createResource_dealreviews(@RequestHeader HttpHeaders httpHeaders, @RequestParam(value = "businessKey", required = false) String businessKey, @RequestBody(required = false) @javax.validation.Valid() @javax.validation.constraints.NotNull() DealreviewsModelInput resource, UriComponentsBuilder uriComponentsBuilder) {
        return UnitOfWorkExecutor.executeInUnitOfWork(application.unitOfWorkManager(), () -> {
            DealreviewsModelInput inputModel = resource != null ? resource : new DealreviewsModelInput();
            ProcessInstance<DealreviewsModel> pi = process.createInstance(businessKey, inputModel.toModel());
            List<String> startFromNode = httpHeaders.get("X-KOGITO-StartFromNode");
            if (startFromNode != null && !startFromNode.isEmpty()) {
                pi.startFrom(startFromNode.get(0));
            } else {
                pi.start();
            }
            UriComponents uriComponents = uriComponentsBuilder.path("/dealreviews/{id}").buildAndExpand(pi.id());
            URI location = uriComponents.toUri();
            return ResponseEntity.created(location).body(pi.checkError().variables().toOutput());
        });
    }

    @GetMapping(produces = MediaType.APPLICATION_JSON_VALUE)
    public List<DealreviewsModelOutput> getResources_dealreviews() {
        return process.instances().values().stream().map(pi -> pi.variables().toOutput()).collect(Collectors.toList());
    }

    @GetMapping(value = "/{id}", produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<DealreviewsModelOutput> getResource_dealreviews(@PathVariable("id") String id) {
        return process.instances().findById(id, ProcessInstanceReadMode.READ_ONLY).map(m -> ResponseEntity.ok(m.variables().toOutput())).orElseGet(() -> ResponseEntity.notFound().build());
    }

    @DeleteMapping(value = "/{id}", produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<DealreviewsModelOutput> deleteResource_dealreviews(@PathVariable("id") final String id) {
        return UnitOfWorkExecutor.executeInUnitOfWork(application.unitOfWorkManager(), () -> process.instances().findById(id).map(pi -> {
            pi.abort();
            return pi.checkError().variables().toOutput();
        }).map(m -> ResponseEntity.ok(m))).orElseGet(() -> ResponseEntity.notFound().build());
    }

    @PutMapping(value = "/{id}", produces = MediaType.APPLICATION_JSON_VALUE, consumes = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<DealreviewsModelOutput> updateModel_dealreviews(@PathVariable("id") String id, @RequestBody(required = false) DealreviewsModel resource) {
        return UnitOfWorkExecutor.executeInUnitOfWork(application.unitOfWorkManager(), () -> process.instances().findById(id).map(pi -> pi.updateVariables(resource).toOutput()).map(m -> ResponseEntity.ok(m))).orElseGet(() -> ResponseEntity.notFound().build());
    }

    @GetMapping(value = "/{id}/tasks", produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<List<TaskModel>> getTasks_dealreviews(@PathVariable("id") String id, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups) {
        return process.instances().findById(id, ProcessInstanceReadMode.READ_ONLY).map(pi -> pi.workItems(Policies.of(user, groups)).stream().map(org.acme.deals.Dealreviews_TaskModelFactory::from).collect(Collectors.toList())).map(m -> ResponseEntity.ok(m)).orElseGet(() -> ResponseEntity.notFound().build());
    }

    @PostMapping(value = "/{id}/review/{taskId}", produces = MediaType.APPLICATION_JSON_VALUE, consumes = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<DealreviewsModelOutput> taskTransition_review_0(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @RequestParam(value = "phase", required = false, defaultValue = "complete") final String phase, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups, @RequestBody(required = false) final org.acme.deals.Dealreviews_2_TaskOutput model) {
        return UnitOfWorkExecutor.executeInUnitOfWork(application.unitOfWorkManager(), () -> process.instances().findById(id).map(pi -> {
            pi.transitionWorkItem(taskId, HumanTaskTransition.withModel(phase, model, Policies.of(user, groups)));
            return ResponseEntity.ok(pi.checkError().variables().toOutput());
        }).orElseGet(() -> ResponseEntity.notFound().build()));
    }

    @PutMapping(value = "/{id}/review/{taskId}", consumes = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<org.acme.deals.Dealreviews_2_TaskOutput> saveTask_review_0(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups, @RequestBody(required = false) final org.acme.deals.Dealreviews_2_TaskOutput model) {
        return UnitOfWorkExecutor.executeInUnitOfWork(application.unitOfWorkManager(), () -> process.instances().findById(id).map(pi -> ResponseEntity.ok(org.acme.deals.Dealreviews_2_TaskOutput.fromMap(pi.updateWorkItem(taskId, wi -> HumanTaskHelper.updateContent(wi, model), Policies.of(user, groups))))).orElseGet(() -> ResponseEntity.notFound().build()));
    }

    @PostMapping(value = "/{id}/review/{taskId}/phases/{phase}", produces = MediaType.APPLICATION_JSON_VALUE, consumes = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<DealreviewsModelOutput> completeTask_review_0(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @PathVariable("phase") final String phase, @RequestParam("user") final String user, @RequestParam("group") final List<String> groups, @RequestBody(required = false) final org.acme.deals.Dealreviews_2_TaskOutput model) {
        return UnitOfWorkExecutor.executeInUnitOfWork(application.unitOfWorkManager(), () -> process.instances().findById(id).map(pi -> {
            pi.transitionWorkItem(taskId, HumanTaskTransition.withModel(phase, model, Policies.of(user, groups)));
            return ResponseEntity.ok(pi.checkError().variables().toOutput());
        }).orElseGet(() -> ResponseEntity.notFound().build()));
    }

    @PostMapping(value = "/{id}/review/{taskId}/comments", produces = MediaType.APPLICATION_JSON_VALUE, consumes = MediaType.TEXT_PLAIN_VALUE)
    public ResponseEntity<Comment> addComment_review_0(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups, @RequestBody String commentInfo, UriComponentsBuilder uriComponentsBuilder) {
        return UnitOfWorkExecutor.executeInUnitOfWork(application.unitOfWorkManager(), () -> process.instances().findById(id).map(pi -> {
            Comment comment = pi.updateWorkItem(taskId, wi -> HumanTaskHelper.addComment(wi, commentInfo, user), Policies.of(user, groups));
            return ResponseEntity.created(uriComponentsBuilder.path("/dealreviews/{id}/review/{taskId}/comments/{commentId}").buildAndExpand(id, taskId, comment.getId()).toUri()).body(comment);
        }).orElseGet(() -> ResponseEntity.notFound().build()));
    }

    @PutMapping(value = "/{id}/review/{taskId}/comments/{commentId}", produces = MediaType.APPLICATION_JSON_VALUE, consumes = MediaType.TEXT_PLAIN_VALUE)
    public ResponseEntity<Comment> updateComment_review_0(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @PathVariable("commentId") final String commentId, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups, @RequestBody String comment) {
        return UnitOfWorkExecutor.executeInUnitOfWork(application.unitOfWorkManager(), () -> process.instances().findById(id).map(pi -> ResponseEntity.ok(pi.updateWorkItem(taskId, wi -> HumanTaskHelper.updateComment(wi, commentId, comment, user), Policies.of(user, groups)))).orElseGet(() -> ResponseEntity.notFound().build()));
    }

    @DeleteMapping(value = "/{id}/review/{taskId}/comments/{commentId}")
    public ResponseEntity deleteComment_review_0(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @PathVariable("commentId") final String commentId, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups) {
        return UnitOfWorkExecutor.executeInUnitOfWork(application.unitOfWorkManager(), () -> process.instances().findById(id).map(pi -> {
            boolean removed = pi.updateWorkItem(taskId, wi -> HumanTaskHelper.deleteComment(wi, commentId, user), Policies.of(user, groups));
            return removed ? ResponseEntity.ok().build() : ResponseEntity.notFound().build();
        }).orElseGet(() -> ResponseEntity.notFound().build()));
    }

    @PostMapping(value = "/{id}/review/{taskId}/attachments", produces = MediaType.APPLICATION_JSON_VALUE, consumes = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<Attachment> addAttachment_review_0(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups, @RequestBody AttachmentInfo attachmentInfo, UriComponentsBuilder uriComponentsBuilder) {
        return UnitOfWorkExecutor.executeInUnitOfWork(application.unitOfWorkManager(), () -> process.instances().findById(id).map(pi -> {
            Attachment attachment = pi.updateWorkItem(taskId, wi -> HumanTaskHelper.addAttachment(wi, attachmentInfo, user), Policies.of(user, groups));
            return ResponseEntity.created(uriComponentsBuilder.path("/dealreviews/{id}/review/{taskId}/attachments/{attachmentId}").buildAndExpand(id, taskId, attachment.getId()).toUri()).body(attachment);
        }).orElseGet(() -> ResponseEntity.notFound().build()));
    }

    @PutMapping(value = "/{id}/review/{taskId}/attachments/{attachmentId}", produces = MediaType.APPLICATION_JSON_VALUE, consumes = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<Attachment> updateAttachment_review_0(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @PathVariable("attachmentId") final String attachmentId, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups, @RequestBody AttachmentInfo attachment) {
        return UnitOfWorkExecutor.executeInUnitOfWork(application.unitOfWorkManager(), () -> process.instances().findById(id).map(pi -> ResponseEntity.ok(pi.updateWorkItem(taskId, wi -> HumanTaskHelper.updateAttachment(wi, attachmentId, attachment, user), Policies.of(user, groups)))).orElseGet(() -> ResponseEntity.notFound().build()));
    }

    @DeleteMapping(value = "/{id}/review/{taskId}/attachments/{attachmentId}")
    public ResponseEntity deleteAttachment_review_0(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @PathVariable("attachmentId") final String attachmentId, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups) {
        return UnitOfWorkExecutor.executeInUnitOfWork(application.unitOfWorkManager(), () -> process.instances().findById(id).map(pi -> {
            boolean removed = pi.updateWorkItem(taskId, wi -> HumanTaskHelper.deleteAttachment(wi, attachmentId, user), Policies.of(user, groups));
            return (removed ? ResponseEntity.ok() : ResponseEntity.notFound()).build();
        }).orElseGet(() -> ResponseEntity.notFound().build()));
    }

    @GetMapping(value = "/{id}/review/{taskId}/attachments/{attachmentId}", produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<Attachment> getAttachment_review_0(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @PathVariable("attachmentId") final String attachmentId, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups) {
        Optional<ProcessInstance<DealreviewsModel>> pi = process.instances().findById(id);
        if (pi.isPresent()) {
            Attachment attachment = HumanTaskHelper.findTask(pi.get(), taskId, Policies.of(user, groups)).getAttachments().get(attachmentId);
            if (attachment != null) {
                return ResponseEntity.ok(attachment);
            }
        }
        return ResponseEntity.notFound().build();
    }

    @GetMapping(value = "/{id}/review/{taskId}/attachments", produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<Collection<Attachment>> getAttachments_review_0(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @RequestParam(value = "user") final String user, @RequestParam(value = "group") final List<String> groups) {
        Optional<ProcessInstance<DealreviewsModel>> pi = process.instances().findById(id);
        return pi.isPresent() ? ResponseEntity.ok(HumanTaskHelper.findTask(pi.get(), taskId, Policies.of(user, groups)).getAttachments().values()) : ResponseEntity.notFound().build();
    }

    @GetMapping(value = "/{id}/review/{taskId}/comments/{commentId}", produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<Comment> getComment_review_0(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @PathVariable("commentId") final String commentId, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups) {
        Optional<ProcessInstance<DealreviewsModel>> pi = process.instances().findById(id);
        if (pi.isPresent()) {
            Comment comment = HumanTaskHelper.findTask(pi.get(), taskId, Policies.of(user, groups)).getComments().get(commentId);
            if (comment != null) {
                return ResponseEntity.ok(comment);
            }
        }
        return ResponseEntity.notFound().build();
    }

    @GetMapping(value = "/{id}/review/{taskId}/comments", produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<Collection<Comment>> getComments_review_0(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups) {
        Optional<ProcessInstance<DealreviewsModel>> pi = process.instances().findById(id);
        return pi.isPresent() ? ResponseEntity.ok(HumanTaskHelper.findTask(pi.get(), taskId, Policies.of(user, groups)).getComments().values()) : ResponseEntity.notFound().build();
    }

    @GetMapping(value = "/{id}/review/{taskId}", produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<org.acme.deals.Dealreviews_2_TaskModel> getTask_review_0(@PathVariable("id") String id, @PathVariable("taskId") String taskId, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups) {
        return process.instances().findById(id).map(pi -> org.acme.deals.Dealreviews_2_TaskModel.from(pi.workItem(taskId, Policies.of(user, groups)))).map(m -> ResponseEntity.ok(m)).orElseGet(() -> ResponseEntity.notFound().build());
    }

    @GetMapping(value = "review/schema", produces = MediaType.APPLICATION_JSON_VALUE)
    public Map<String, Object> getSchema_review_0() {
        return JsonSchemaUtil.load(this.getClass().getClassLoader(), process.id(), "review");
    }

    @GetMapping(value = "/{id}/review/{taskId}/schema", produces = MediaType.APPLICATION_JSON_VALUE)
    public Map<String, Object> getSchemaAndPhases_review_0(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups) {
        return JsonSchemaUtil.addPhases(process, application, id, taskId, Policies.of(user, groups), JsonSchemaUtil.load(this.getClass().getClassLoader(), process.id(), "review"));
    }

    @DeleteMapping(value = "/{id}/review/{taskId}", produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<DealreviewsModelOutput> abortTask_review_0(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @RequestParam(value = "phase", required = false, defaultValue = "abort") final String phase, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups) {
        return UnitOfWorkExecutor.executeInUnitOfWork(application.unitOfWorkManager(), () -> process.instances().findById(id).map(pi -> {
            pi.transitionWorkItem(taskId, HumanTaskTransition.withoutModel(phase, Policies.of(user, groups)));
            return ResponseEntity.ok(pi.checkError().variables().toOutput());
        }).orElseGet(() -> ResponseEntity.notFound().build()));
    }
}
