/*
 * Decompiled with CFR 0.152.
 */
package org.jreleaser.sdk.git;

import java.io.IOException;
import java.io.Reader;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;
import org.eclipse.jgit.api.Git;
import org.eclipse.jgit.api.errors.GitAPIException;
import org.eclipse.jgit.lib.AnyObjectId;
import org.eclipse.jgit.lib.ObjectId;
import org.eclipse.jgit.lib.Ref;
import org.eclipse.jgit.revwalk.RevCommit;
import org.jreleaser.model.Changelog;
import org.jreleaser.model.GitService;
import org.jreleaser.model.JReleaserContext;
import org.jreleaser.sdk.git.GitSdk;
import org.jreleaser.util.CollectionUtils;
import org.jreleaser.util.MustacheUtils;
import org.jreleaser.util.StringUtils;
import org.jreleaser.util.Version;

public class ChangelogGenerator {
    private static final String UNCATEGORIZED = "<<UNCATEGORIZED>>";

    public static String generate(JReleaserContext context) throws IOException {
        if (!context.getModel().getRelease().getGitService().getChangelog().isEnabled()) {
            return "";
        }
        return ChangelogGenerator.createChangelog(context);
    }

    private static String createChangelog(JReleaserContext context) throws IOException {
        GitService gitService = context.getModel().getRelease().getGitService();
        Changelog changelog = gitService.getChangelog();
        String separator = System.lineSeparator();
        if ("gitlab".equals(gitService.getServiceName())) {
            separator = separator + System.lineSeparator();
        }
        String commitSeparator = separator;
        try {
            Git git = GitSdk.of(context).open();
            context.getLogger().debug("resolving commits");
            Iterable<RevCommit> commits = ChangelogGenerator.resolveCommits(git, context);
            Comparator<RevCommit> revCommitComparator = Comparator.comparing(RevCommit::getCommitTime).reversed();
            if (changelog.getSort() == Changelog.Sort.ASC) {
                revCommitComparator = Comparator.comparing(RevCommit::getCommitTime);
            }
            context.getLogger().debug("sorting commits {}", new Object[]{changelog.getSort()});
            if (changelog.resolveFormatted(context.getModel().getProject())) {
                return ChangelogGenerator.formatChangelog(context, changelog, commits, revCommitComparator, commitSeparator);
            }
            String commitsUrl = gitService.getResolvedCommitUrl(context.getModel());
            return "## Changelog" + System.lineSeparator() + System.lineSeparator() + StreamSupport.stream(commits.spliterator(), false).sorted(revCommitComparator).map(commit -> ChangelogGenerator.formatCommit(commit, commitsUrl, changelog, commitSeparator)).collect(Collectors.joining(commitSeparator));
        }
        catch (GitAPIException e) {
            throw new IOException(e);
        }
    }

    private static String formatCommit(RevCommit commit, String commitsUrl, Changelog changelog, String commitSeparator) {
        String commitHash = commit.getId().name();
        String abbreviation = commit.getId().abbreviate(7).name();
        String[] input = commit.getFullMessage().trim().split(System.lineSeparator());
        ArrayList<String> lines = new ArrayList<String>();
        if (changelog.isLinks()) {
            lines.add("[" + abbreviation + "](" + commitsUrl + "/" + commitHash + ") " + input[0].trim());
        } else {
            lines.add(abbreviation + " " + input[0].trim());
        }
        return String.join((CharSequence)commitSeparator, lines);
    }

    private static Version versionOf(Ref tag, Pattern versionPattern) {
        Matcher matcher = versionPattern.matcher(GitSdk.extractTagName(tag));
        if (matcher.matches()) {
            return Version.of((String)matcher.group(1));
        }
        return Version.of((String)"0.0.0");
    }

    private static Iterable<RevCommit> resolveCommits(Git git, JReleaserContext context) throws GitAPIException, IOException {
        List tags = git.tagList().call();
        GitService gitService = context.getModel().getRelease().getGitService();
        String effectiveTagName = gitService.getEffectiveTagName(context.getModel());
        String tagName = gitService.getConfiguredTagName();
        String tagPattern = tagName.replaceAll("\\{\\{.*}}", "\\.\\*");
        Pattern versionPattern = Pattern.compile(tagName.replaceAll("\\{\\{.*}}", "\\(\\.\\*\\)"));
        tags.sort((tag1, tag2) -> {
            Version v1 = ChangelogGenerator.versionOf(tag1, versionPattern);
            Version v2 = ChangelogGenerator.versionOf(tag2, versionPattern);
            return v2.compareTo(v1);
        });
        ObjectId head = git.getRepository().resolve("HEAD");
        context.getLogger().debug("looking for tag that matches '{}'", new Object[]{effectiveTagName});
        Optional<Ref> tag = tags.stream().filter(ref -> GitSdk.extractTagName(ref).equals(effectiveTagName)).findFirst();
        if ("early-access".equals(effectiveTagName)) {
            if (!tag.isPresent()) {
                context.getLogger().debug("looking for tags that match '{}', excluding '{}'", new Object[]{tagPattern, effectiveTagName});
                tag = tags.stream().filter(ref -> !GitSdk.extractTagName(ref).equals(effectiveTagName)).filter(ref -> GitSdk.extractTagName(ref).matches(tagPattern)).findFirst();
            }
            if (tag.isPresent()) {
                context.getLogger().debug("found tag {}", new Object[]{GitSdk.extractTagName(tag.get())});
                ObjectId fromRef = ChangelogGenerator.getObjectId(git, tag.get());
                return git.log().addRange((AnyObjectId)fromRef, (AnyObjectId)head).call();
            }
            return git.log().add((AnyObjectId)head).call();
        }
        if (!tag.isPresent()) {
            context.getLogger().debug("looking for tags that match '{}', excluding '{}'", new Object[]{tagPattern, effectiveTagName});
            tag = tags.stream().filter(ref -> !GitSdk.extractTagName(ref).equals(effectiveTagName)).filter(ref -> GitSdk.extractTagName(ref).matches(tagPattern)).findFirst();
            if (tag.isPresent()) {
                context.getLogger().debug("found tag {}", new Object[]{GitSdk.extractTagName(tag.get())});
                ObjectId fromRef = ChangelogGenerator.getObjectId(git, tag.get());
                return git.log().addRange((AnyObjectId)fromRef, (AnyObjectId)head).call();
            }
            return git.log().add((AnyObjectId)head).call();
        }
        context.getLogger().debug("looking for a tag before '{}' that matches '{}'", new Object[]{effectiveTagName, tagPattern});
        Version currentVersion = ChangelogGenerator.versionOf(tag.get(), versionPattern);
        Optional<Ref> previousTag = tags.stream().filter(ref -> GitSdk.extractTagName(ref).matches(tagPattern)).filter(ref -> ChangelogGenerator.versionOf(ref, versionPattern).lessThan(currentVersion)).findFirst();
        if (previousTag.isPresent()) {
            context.getLogger().debug("found tag {}", new Object[]{GitSdk.extractTagName(previousTag.get())});
            ObjectId fromRef = ChangelogGenerator.getObjectId(git, previousTag.get());
            ObjectId toRef = ChangelogGenerator.getObjectId(git, tag.get());
            return git.log().addRange((AnyObjectId)fromRef, (AnyObjectId)toRef).call();
        }
        ObjectId toRef = ChangelogGenerator.getObjectId(git, tag.get());
        return git.log().add((AnyObjectId)toRef).call();
    }

    private static ObjectId getObjectId(Git git, Ref ref) throws IOException {
        Ref peeled = git.getRepository().getRefDatabase().peel(ref);
        return peeled.getPeeledObjectId() != null ? peeled.getPeeledObjectId() : peeled.getObjectId();
    }

    private static String formatChangelog(JReleaserContext context, Changelog changelog, Iterable<RevCommit> commits, Comparator<RevCommit> revCommitComparator, String lineSeparator) {
        LinkedHashSet contributorNames = new LinkedHashSet();
        LinkedHashMap categories = new LinkedHashMap();
        StreamSupport.stream(commits.spliterator(), false).sorted(revCommitComparator).map(Commit::of).peek(c -> {
            contributorNames.add(((Commit)c).author);
            if (StringUtils.isNotBlank((String)((Commit)c).committer)) {
                contributorNames.add(((Commit)c).committer);
            }
        }).peek(c -> ChangelogGenerator.applyLabels(c, changelog.getLabelers())).filter(c -> ChangelogGenerator.checkLabels(c, changelog)).forEach(commit -> categories.computeIfAbsent(ChangelogGenerator.categorize(commit, changelog), k -> new ArrayList()).add(commit));
        GitService gitService = context.getModel().getRelease().getGitService();
        String commitsUrl = gitService.getResolvedCommitUrl(context.getModel());
        StringBuilder changes = new StringBuilder();
        for (Changelog.Category category : changelog.getCategories()) {
            String categoryTitle = category.getTitle();
            if (!categories.containsKey(categoryTitle)) continue;
            changes.append("## ").append(categoryTitle).append(lineSeparator);
            changes.append(((List)categories.get(categoryTitle)).stream().map(c -> MustacheUtils.applyTemplate((String)changelog.getChange(), c.asContext(changelog.isLinks(), commitsUrl))).collect(Collectors.joining(lineSeparator))).append(lineSeparator).append(System.lineSeparator());
        }
        if (!changelog.isHideUncategorized() && categories.containsKey(UNCATEGORIZED)) {
            if (changes.length() > 0) {
                changes.append("---").append(lineSeparator);
            }
            changes.append(((List)categories.get(UNCATEGORIZED)).stream().map(c -> MustacheUtils.applyTemplate((String)changelog.getChange(), c.asContext(changelog.isLinks(), commitsUrl))).collect(Collectors.joining(lineSeparator))).append(lineSeparator).append(System.lineSeparator());
        }
        StringBuilder contributors = new StringBuilder("## Contributors").append(lineSeparator).append(String.join((CharSequence)", ", contributorNames)).append(lineSeparator);
        Map props = context.props();
        props.put("changelogChanges", MustacheUtils.passThrough((String)changes.toString()));
        props.put("changelogContributors", MustacheUtils.passThrough((String)contributors.toString()));
        return ChangelogGenerator.applyReplacers(changelog, StringUtils.stripMargin((String)MustacheUtils.applyTemplate((Reader)changelog.getResolvedContentTemplate(context), (Map)props)));
    }

    private static String applyReplacers(Changelog changelog, String text) {
        for (Changelog.Replacer replacer : changelog.getReplacers()) {
            text = text.replaceAll(replacer.getSearch(), replacer.getReplace());
        }
        return text;
    }

    private static String categorize(Commit commit, Changelog changelog) {
        if (!commit.labels.isEmpty()) {
            for (Changelog.Category category : changelog.getCategories()) {
                if (!CollectionUtils.intersects((Set)category.getLabels(), (Set)commit.labels)) continue;
                return category.getTitle();
            }
        }
        return UNCATEGORIZED;
    }

    private static void applyLabels(Commit commit, Set<Changelog.Labeler> labelers) {
        for (Changelog.Labeler labeler : labelers) {
            if (StringUtils.isNotBlank((String)labeler.getTitle()) && (commit.title.contains(labeler.getTitle()) || commit.title.matches(StringUtils.toSafeRegexPattern((String)labeler.getTitle())))) {
                commit.labels.add(labeler.getLabel());
            }
            if (!StringUtils.isNotBlank((String)labeler.getBody()) || !commit.body.contains(labeler.getBody()) && !commit.body.matches(StringUtils.toSafeRegexPattern((String)labeler.getBody()))) continue;
            commit.labels.add(labeler.getLabel());
        }
    }

    private static boolean checkLabels(Commit commit, Changelog changelog) {
        if (!changelog.getIncludeLabels().isEmpty()) {
            return CollectionUtils.intersects((Set)changelog.getIncludeLabels(), (Set)commit.labels);
        }
        if (!changelog.getExcludeLabels().isEmpty()) {
            return !CollectionUtils.intersects((Set)changelog.getExcludeLabels(), (Set)commit.labels);
        }
        return true;
    }

    private static class Commit {
        private final Set<String> labels = new LinkedHashSet<String>();
        private String fullHash;
        private String shortHash;
        private String title;
        private String body;
        private String author;
        private String committer;
        private int time;

        private Commit() {
        }

        Map<String, Object> asContext(boolean links, String commitsUrl) {
            LinkedHashMap<String, Object> context = new LinkedHashMap<String, Object>();
            if (links) {
                context.put("commitShortHash", MustacheUtils.passThrough((String)("[" + this.shortHash + "](" + commitsUrl + "/" + this.shortHash + ")")));
            } else {
                context.put("commitShortHash", this.shortHash);
            }
            context.put("commitFullHash", this.fullHash);
            context.put("commitTitle", MustacheUtils.passThrough((String)this.title));
            context.put("commitAuthor", MustacheUtils.passThrough((String)this.author));
            return context;
        }

        static Commit of(RevCommit rc) {
            Commit c = new Commit();
            c.fullHash = rc.getId().name();
            c.shortHash = rc.getId().abbreviate(7).name();
            c.body = rc.getFullMessage();
            c.title = c.body.split(System.lineSeparator())[0];
            c.author = rc.getAuthorIdent().getName();
            c.committer = rc.getCommitterIdent().getName();
            c.time = rc.getCommitTime();
            return c;
        }
    }
}

