/**
 * Licensed to jclouds, Inc. (jclouds) under one or more
 * contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  jclouds licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.jclouds.elasticstack.compute.functions;

import java.util.Map;

import javax.inject.Inject;
import javax.inject.Singleton;

import org.jclouds.compute.domain.Image;
import org.jclouds.compute.domain.ImageBuilder;
import org.jclouds.compute.domain.OperatingSystem;
import org.jclouds.domain.Credentials;
import org.jclouds.domain.Location;
import org.jclouds.elasticstack.domain.DriveInfo;
import org.jclouds.elasticstack.domain.WellKnownImage;

import com.google.common.base.Function;
import com.google.common.base.Supplier;
import com.google.common.collect.ImmutableMap;

/**
 * @author Adrian Cole
 */
@Singleton
public class WellKnownImageToImage implements Function<DriveInfo, Image> {
   private final Supplier<Location> locationSupplier;
   private final Map<String, WellKnownImage> preinstalledImages;

   @Inject
   public WellKnownImageToImage(Supplier<Location> locationSupplier, Map<String, WellKnownImage> preinstalledImages) {
      this.locationSupplier = locationSupplier;
      this.preinstalledImages = preinstalledImages;
   }

   @Override
   public Image apply(DriveInfo drive) {
      WellKnownImage input = preinstalledImages.get(drive.getUuid());
      return new ImageBuilder().ids(drive.getUuid()).userMetadata(
               ImmutableMap.<String, String> builder().putAll(drive.getUserMetadata())
                        .put("size", input.getSize() + "").build()).defaultCredentials(
               new Credentials(input.getLoginUser(), null)).location(locationSupplier.get()).name(
               input.getDescription()).description(drive.getName()).operatingSystem(
               new OperatingSystem.Builder().family(input.getOsFamily()).version(input.getOsVersion()).name(
                        input.getDescription()).description(drive.getName()).is64Bit(input.is64bit()).build()).version(
               "").build();
   }
}