/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.managed.api;

import java.io.Serializable;
import java.lang.annotation.Annotation;
import java.util.Collection;
import java.util.Map;
import java.util.Set;

import org.jboss.managed.api.annotation.ActivationPolicy;
import org.jboss.managed.api.annotation.ManagementObjectRef;
import org.jboss.managed.api.annotation.ViewUse;
import org.jboss.metatype.api.types.MetaType;
import org.jboss.metatype.api.values.MetaValue;

/**
 * ManagedProperty.
 * 
 * @author <a href="adrian@jboss.com">Adrian Brock</a>
 * @author Scott.Stark@jboss.org
 * @version $Revision: 1.1 $
 */
public interface ManagedProperty extends Serializable, TransientAttachments
{
   /**
    * Get the managed object the property is associated with.
    * 
    * @return the managed object
    */
   ManagedObject getManagedObject();
   void setManagedObject(ManagedObject managedObject);

   /**
    * Get the ManagedObject 
    * @see {@linkplain ManagementObjectRef}
    * @return the ManagedObject the property references, null
    *    if there is no reference or its unresolved.
    */
   ManagedObject getTargetManagedObject();
   void setTargetManagedObject(ManagedObject target);

   /**
    * Get the fields
    * 
    * @return the fields
    */
   Fields getFields();

   /**
    * Create a copy of the managed property. This is a copy that whose fields
    * are copies of the original property. Changes to the copies fields have
    * not affect on the original.
    * @return a copy of the managed property.
    */
   ManagedProperty copy();

   /**
    * Get a field
    *
    * @param <T> the expected type
    * @param fieldName the field name
    * @param expected the expected type
    * @return the value
    */
   <T> T getField(String fieldName, Class<T> expected);
   
   /**
    * Set a field
    *
    * @param fieldName the field name
    * @param value the value
    */
   void setField(String fieldName, Serializable value);
   
   /**
    * Get the property's name
    * 
    * @return the property's name
    */
   String getName();

   /**
    * Get the property's mapped name. This is an optional name
    * that allows for an external name to be mapped to an
    * internal one.
    * 
    * @return the mapped name if it exists, null if there is no
    * mapped name.
    */
   String getMappedName();

   /**
    * Get the description
    * 
    * @return the description
    */
   String getDescription();

   /**
    * Get the type
    * 
    * @return the type
    */
   MetaType getMetaType();

   /**
    * Get the annotations associated with the property
    * @return the annotations associated with the property
    */
   Map<String, Annotation> getAnnotations();

   /**
    * Does the property have the annotation referenced by key.
    * @param key the key into {@link #getAnnotations()}
    * @return true if the annotation exists, false otherwise.
    */
   boolean hasAnnotation(String key);

   /**
    * See if the property has the indicated ViewUse among its
    * @ManagementProperty annotation uses.
    * 
    * @param use - the ViewUse to check for
    * @return true if the ViewUse exists in the property uses, false otherwise
    */
   boolean hasViewUse(ViewUse use);

   /**
    * Return the admin view names associated with the property
    * @return the potentially empty admin view names associated with the property
    */
   Collection<String> getAdminViewUses();

   /**
    * Get the activation policy for property value updates.
    * @return the property activation policy
    */
   ActivationPolicy getActivationPolicy();

   /**
    * Get the value
    * 
    * @return the value
    */
   MetaValue getValue();

   /**
    * Set the value
    * 
    * @param value the value
    */
   void setValue(MetaValue value);

   /**
    * Get the legal values
    * 
    * @return the legal values
    */
   Set<MetaValue> getLegalValues();

   /**
    * Get the property default value if one exists.
    * @return the default value if one exists, null otherwise.
    */
   MetaValue getDefaultValue();

   /**
    * Get the minimum value
    * 
    * @return the minimum value, a MetaValue implementing Comparable
    */
   Comparable<? extends MetaValue> getMinimumValue();

   /**
    * Get the maximum value
    * 
    * @return the maximum value, a MetaValue implementing Comparable
    */
   Comparable<? extends MetaValue> getMaximumValue();

   /**
    * Check whether this is a valid value
    * 
    * @param value the value
    * @return null for a valid value, an error message otherwise
    */
   String checkValidValue(MetaValue value);
   
   /**
    * Whether the property is mandatory
    * 
    * @return true when mandatory
    */
   boolean isMandatory();

   /**
    * Whether the property is read only
    * @return true when read only
    */
   boolean isReadOnly();

   /**
    * Whether the property has been edited/modified.
    * 
    * @return true when modified
    */
   boolean isModified();

   /**
    * Set the modified state of the property value.
    * @param flag - whether the property has been modified
    */
   public void setModified(boolean flag);

   /**
    * Whether the property has been marked as removed from its ManagedObject.
    * @return true is the property has been removed.
    */
   boolean isRemoved();
   /**
    * Set whether a property has been marked as removed from its ManagedObject.
    * @param flag - true is the property has been removed, false if not.
    */
   void setRemoved(boolean flag);

}
