/*
* JBoss, Home of Professional Open Source
* Copyright 2007, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.managed.api.factory;

import org.jboss.managed.api.ManagedObject;
import org.jboss.managed.api.annotation.ManagementObjectID;
import org.jboss.managed.plugins.factory.ManagedObjectFactoryBuilder;
import org.jboss.managed.spi.factory.InstanceClassFactory;
import org.jboss.managed.spi.factory.ManagedObjectBuilder;
import org.jboss.metadata.spi.MetaData;

/**
 * ManagedObjectFactory.
 * 
 * @author <a href="adrian@jboss.com">Adrian Brock</a>
 * @version $Revision: 1.1 $
 */
public abstract class ManagedObjectFactory
{
   /** The managed object factory instance */
   private static ManagedObjectFactoryBuilder builder = new ManagedObjectFactoryBuilder();
   
   /**
    * Get the managed object factory instance
    * 
    * @return the instance
    */
   public static ManagedObjectFactory getInstance()
   {
      return builder.create();
   }
   
   /**
    * Create a managed object from the given object
    * 
    * @param instance - the object to build the ManagedObject from
    * @param name - the name of the managed object. If null, the name will
    *    be derived from the object annotations or attachment name.
    * @param nameType - the name of the managed object. If null, the name will
    *    be derived from the object annotations or default to "".
    * 
    * @see ManagementObjectID
    * 
    * @return the managed object
    * @throws IllegalArgumentException for a null object
    */
   public ManagedObject initManagedObject(Object instance, String name, String nameType)
   {
      return initManagedObject(instance, null, name, nameType);
   }
   public ManagedObject initManagedObject(Object instance, Class<?> instanceType,
         String name, String nameType)
   {
      return initManagedObject(instance, null, null, name, nameType);
   }
   public ManagedObject initManagedObject(Object instance, MetaData metaData)
   {
      return initManagedObject(instance, null, metaData, null, null);
   }

   /**
    * Create a managed object from the given object
    * 
    * @param instance - the object to build the ManagedObject from
    * @param instanceType - the optional type to determine InstanceClassFactory registration from.
    * If not specified instance.getClass() is used.
    * @param metaData - the optional metadata repository accessor used to query
    * for management annotation overrides/additions to the clazz
    * @param name - the name of the managed object. If null, the name will
    *    be derived from the object annotations or attachment name.
    * @param nameType - the name of the managed object. If null, the name will
    *    be derived from the object annotations or default to "".
    * 
    * @see ManagementObjectID
    * 
    * @return the managed object
    */
   public abstract ManagedObject initManagedObject(Object instance, Class<?> instanceType,
         MetaData metaData, String name, String nameType);

   /**
    * Create a shell managed object from the given class
    *
    * @param <T> the class
    * @param clazz the class
    * @return the managed object
    * @throws IllegalArgumentException for a null class
    */
   public <T> ManagedObject createManagedObject(Class<T> clazz)
   {
      return createManagedObject(clazz, null);
   }
   public abstract <T> ManagedObject createManagedObject(Class<T> clazz, MetaData metaData);

   /**
    * Set a managed object builder
    * 
    * @param clazz the class
    * @param builder the builder (null to remove the builder)
    */
   public abstract void setBuilder(Class<?> clazz, ManagedObjectBuilder builder);

   /**
    * Set the InstanceClassFactory for an instance type.
    *
    * @param <T> the class type
    * @param clazz the class
    * @param factory - the factory used to obtain the class to scan for
    * management annotations.
    */
   public abstract <X> void setInstanceClassFactory(Class<X> clazz, InstanceClassFactory<X> factory);
   /**
    * Add an InstanceClassFactory. Calls setInstanceClassFactory(factory.getType(), factory);.
    * @param factory - the factory used to obtain the class to scan for
    */
   public <X> void addInstanceClassFactory(InstanceClassFactory<X> factory)
   {
      setInstanceClassFactory(factory.getType(), factory);
   }
   /**
    * Remove an InstanceClassFactory. Calls setInstanceClassFactory(factory.getType(), null);.
    * @param factory
    */
   public <X> void removeInstanceClassFactory(InstanceClassFactory<X> factory)
   {
      setInstanceClassFactory(factory.getType(), null);
   }

   /**
    * Get the InstanceClassFactory for an instance type.
    *
    * @param <T> the class type
    * @param clazz the class
    * @return the factory used to obtain the class to scan for
    * management annotations. 
    */
   public <T> InstanceClassFactory<T> getInstanceClassFactory(Class<T> clazz)
   {
      return getInstanceClassFactory(clazz, null);
   }
   public abstract <T> InstanceClassFactory<T> getInstanceClassFactory(Class<T> clazz, MetaData metaData);
}
