/*
 * JBoss, Home of Professional Open Source
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.metatype.plugins.values.mappers;

import java.lang.reflect.Type;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Map;
import java.util.Map.Entry;

import javax.management.ObjectName;

import org.jboss.metatype.api.types.CompositeMetaType;
import org.jboss.metatype.api.types.MetaType;
import org.jboss.metatype.api.types.SimpleMetaType;
import org.jboss.metatype.api.values.CompositeValue;
import org.jboss.metatype.api.values.MapCompositeValueSupport;
import org.jboss.metatype.api.values.MetaValue;
import org.jboss.metatype.api.values.SimpleValue;
import org.jboss.metatype.api.values.SimpleValueSupport;
import org.jboss.metatype.plugins.types.MutableCompositeMetaType;
import org.jboss.metatype.spi.values.MetaMapper;

/**
 * @author Scott.Stark@jboss.org
 * @version $Revision:$
 */
public class ObjectNameMetaMapper extends MetaMapper<ObjectName>
{
   CompositeMetaType type;

   public ObjectNameMetaMapper()
   {
      type = new MutableCompositeMetaType(ObjectName.class.getName(), "JMX ObjectName");
   }
   public ObjectNameMetaMapper(String... args)
   {
      MutableCompositeMetaType mcmt = new MutableCompositeMetaType(ObjectName.class.getName(), "JMX ObjectName");
      if(((args.length %2) == 0) == false)
         throw new IllegalStateException("args counts must be 2*n for n key,key-description pairs");
      mcmt.addItem("domain", "the object name domain", SimpleMetaType.STRING);
      for(int n = 0; n < args.length; n += 2)
      {
         String key = args[n];
         String description = args[n+1];
         mcmt.addItem(key, description, SimpleMetaType.STRING);
      }
      mcmt.freeze();
      type = mcmt;
   }

   @Override
   public MetaType getMetaType()
   {
      return type;
   }

   @Override
   public Type mapToType()
   {
      return ObjectName.class;
   }

   @Override
   public MetaValue createMetaValue(MetaType metaType, ObjectName object)
   {
      String domain = object.getDomain();
      Map<String, String> props = object.getKeyPropertyList();
      Map<String, MetaValue> metaProps = new HashMap<String, MetaValue>();
      metaProps.put("domain", SimpleValueSupport.wrap(domain));
      for(Entry<String,String> entry : props.entrySet())
      {
         metaProps.put(entry.getKey(), SimpleValueSupport.wrap(entry.getValue()));
      }
      MapCompositeValueSupport map = new MapCompositeValueSupport(metaProps, SimpleMetaType.STRING);
      return map;
   }

   @Override
   public ObjectName unwrapMetaValue(MetaValue metaValue)
   {
      CompositeValue cv = (CompositeValue) metaValue;
      String domain = null;
      Hashtable<String, String> props = new Hashtable<String, String>();
      for(String key : cv.getMetaType().itemSet())
      {
         SimpleValue svalue = (SimpleValue) cv.get(key);
         String value = (String) svalue.getValue();
         if(key.equalsIgnoreCase("domain"))
            domain = value;
         else
            props.put(key, value);
      }

      try
      {
         ObjectName name = new ObjectName(domain, props);
         return name;
      }
      catch(Exception e)
      {
         throw new IllegalArgumentException("Failed to build ObjectName from MetaValue: "+metaValue, e);
      }
   }

}
