/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.master.replication;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.DoNotRetryIOException;
import org.apache.hadoop.hbase.HBaseConfiguration;
import org.apache.hadoop.hbase.ReplicationPeerNotFoundException;
import org.apache.hadoop.hbase.ServerName;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.replication.ReplicationPeerConfigUtil;
import org.apache.hadoop.hbase.replication.HBaseReplicationEndpoint;
import org.apache.hadoop.hbase.replication.ReplicationEndpoint;
import org.apache.hadoop.hbase.replication.ReplicationException;
import org.apache.hadoop.hbase.replication.ReplicationPeerConfig;
import org.apache.hadoop.hbase.replication.ReplicationPeerConfigBuilder;
import org.apache.hadoop.hbase.replication.ReplicationPeerDescription;
import org.apache.hadoop.hbase.replication.ReplicationPeerStorage;
import org.apache.hadoop.hbase.replication.ReplicationQueueInfo;
import org.apache.hadoop.hbase.replication.ReplicationQueueStorage;
import org.apache.hadoop.hbase.replication.ReplicationStorageFactory;
import org.apache.hadoop.hbase.replication.ReplicationUtils;
import org.apache.hadoop.hbase.shaded.org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.hbase.shaded.org.apache.zookeeper.KeeperException;
import org.apache.hadoop.hbase.zookeeper.ZKClusterId;
import org.apache.hadoop.hbase.zookeeper.ZKConfig;
import org.apache.hadoop.hbase.zookeeper.ZKWatcher;
import org.apache.yetus.audience.InterfaceAudience;

@InterfaceAudience.Private
public class ReplicationPeerManager {
    private final ReplicationPeerStorage peerStorage;
    private final ReplicationQueueStorage queueStorage;
    private final ConcurrentMap<String, ReplicationPeerDescription> peers;
    private final String clusterId;
    private final Configuration conf;

    ReplicationPeerManager(ReplicationPeerStorage peerStorage, ReplicationQueueStorage queueStorage, ConcurrentMap<String, ReplicationPeerDescription> peers, Configuration conf, String clusterId) {
        this.peerStorage = peerStorage;
        this.queueStorage = queueStorage;
        this.peers = peers;
        this.conf = conf;
        this.clusterId = clusterId;
    }

    private void checkQueuesDeleted(String peerId) throws ReplicationException, DoNotRetryIOException {
        for (ServerName replicator : this.queueStorage.getListOfReplicators()) {
            List<String> queueIds = this.queueStorage.getAllQueues(replicator);
            for (String queueId : queueIds) {
                ReplicationQueueInfo queueInfo = new ReplicationQueueInfo(queueId);
                if (!queueInfo.getPeerId().equals(peerId)) continue;
                throw new DoNotRetryIOException("undeleted queue for peerId: " + peerId + ", replicator: " + replicator + ", queueId: " + queueId);
            }
        }
        if (this.queueStorage.getAllPeersFromHFileRefsQueue().contains(peerId)) {
            throw new DoNotRetryIOException("Undeleted queue for peer " + peerId + " in hfile-refs");
        }
    }

    void preAddPeer(String peerId, ReplicationPeerConfig peerConfig) throws DoNotRetryIOException, ReplicationException {
        if (peerId.contains("-")) {
            throw new DoNotRetryIOException("Found invalid peer name: " + peerId);
        }
        this.checkPeerConfig(peerConfig);
        if (this.peers.containsKey(peerId)) {
            throw new DoNotRetryIOException("Replication peer " + peerId + " already exists");
        }
        this.checkQueuesDeleted(peerId);
    }

    private ReplicationPeerDescription checkPeerExists(String peerId) throws DoNotRetryIOException {
        ReplicationPeerDescription desc = (ReplicationPeerDescription)this.peers.get(peerId);
        if (desc == null) {
            throw new ReplicationPeerNotFoundException(peerId);
        }
        return desc;
    }

    ReplicationPeerConfig preRemovePeer(String peerId) throws DoNotRetryIOException {
        return this.checkPeerExists(peerId).getPeerConfig();
    }

    void preEnablePeer(String peerId) throws DoNotRetryIOException {
        ReplicationPeerDescription desc = this.checkPeerExists(peerId);
        if (desc.isEnabled()) {
            throw new DoNotRetryIOException("Replication peer " + peerId + " has already been enabled");
        }
    }

    void preDisablePeer(String peerId) throws DoNotRetryIOException {
        ReplicationPeerDescription desc = this.checkPeerExists(peerId);
        if (!desc.isEnabled()) {
            throw new DoNotRetryIOException("Replication peer " + peerId + " has already been disabled");
        }
    }

    ReplicationPeerDescription preUpdatePeerConfig(String peerId, ReplicationPeerConfig peerConfig) throws DoNotRetryIOException {
        this.checkPeerConfig(peerConfig);
        ReplicationPeerDescription desc = this.checkPeerExists(peerId);
        ReplicationPeerConfig oldPeerConfig = desc.getPeerConfig();
        if (!this.isStringEquals(peerConfig.getClusterKey(), oldPeerConfig.getClusterKey())) {
            throw new DoNotRetryIOException("Changing the cluster key on an existing peer is not allowed. Existing key '" + oldPeerConfig.getClusterKey() + "' for peer " + peerId + " does not match new key '" + peerConfig.getClusterKey() + "'");
        }
        if (!this.isStringEquals(peerConfig.getReplicationEndpointImpl(), oldPeerConfig.getReplicationEndpointImpl())) {
            throw new DoNotRetryIOException("Changing the replication endpoint implementation class on an existing peer is not allowed. Existing class '" + oldPeerConfig.getReplicationEndpointImpl() + "' for peer " + peerId + " does not match new class '" + peerConfig.getReplicationEndpointImpl() + "'");
        }
        return desc;
    }

    public void addPeer(String peerId, ReplicationPeerConfig peerConfig, boolean enabled) throws ReplicationException {
        if (this.peers.containsKey(peerId)) {
            return;
        }
        peerConfig = ReplicationPeerConfigUtil.updateReplicationBasePeerConfigs(this.conf, peerConfig);
        ReplicationPeerConfig copiedPeerConfig = ReplicationPeerConfig.newBuilder(peerConfig).build();
        this.peerStorage.addPeer(peerId, copiedPeerConfig, enabled);
        this.peers.put(peerId, new ReplicationPeerDescription(peerId, enabled, copiedPeerConfig));
    }

    public void removePeer(String peerId) throws ReplicationException {
        if (!this.peers.containsKey(peerId)) {
            return;
        }
        this.peerStorage.removePeer(peerId);
        this.peers.remove(peerId);
    }

    private void setPeerState(String peerId, boolean enabled) throws ReplicationException {
        ReplicationPeerDescription desc = (ReplicationPeerDescription)this.peers.get(peerId);
        if (desc.isEnabled() == enabled) {
            return;
        }
        this.peerStorage.setPeerState(peerId, enabled);
        this.peers.put(peerId, new ReplicationPeerDescription(peerId, enabled, desc.getPeerConfig()));
    }

    public void enablePeer(String peerId) throws ReplicationException {
        this.setPeerState(peerId, true);
    }

    public void disablePeer(String peerId) throws ReplicationException {
        this.setPeerState(peerId, false);
    }

    public void updatePeerConfig(String peerId, ReplicationPeerConfig peerConfig) throws ReplicationException {
        ReplicationPeerDescription desc = (ReplicationPeerDescription)this.peers.get(peerId);
        ReplicationPeerConfig oldPeerConfig = desc.getPeerConfig();
        ReplicationPeerConfigBuilder newPeerConfigBuilder = ReplicationPeerConfig.newBuilder(peerConfig);
        newPeerConfigBuilder.putAllConfiguration(oldPeerConfig.getConfiguration());
        newPeerConfigBuilder.putAllConfiguration(peerConfig.getConfiguration());
        ReplicationPeerConfig newPeerConfig = newPeerConfigBuilder.build();
        this.peerStorage.updatePeerConfig(peerId, newPeerConfig);
        this.peers.put(peerId, new ReplicationPeerDescription(peerId, desc.isEnabled(), newPeerConfig));
    }

    public List<ReplicationPeerDescription> listPeers(Pattern pattern) {
        if (pattern == null) {
            return new ArrayList<ReplicationPeerDescription>(this.peers.values());
        }
        return this.peers.values().stream().filter(r -> pattern.matcher(r.getPeerId()).matches()).collect(Collectors.toList());
    }

    public Optional<ReplicationPeerConfig> getPeerConfig(String peerId) {
        ReplicationPeerDescription desc = (ReplicationPeerDescription)this.peers.get(peerId);
        return desc != null ? Optional.of(desc.getPeerConfig()) : Optional.empty();
    }

    void removeAllLastPushedSeqIds(String peerId) throws ReplicationException {
        this.queueStorage.removeLastSequenceIds(peerId);
    }

    void removeAllQueuesAndHFileRefs(String peerId) throws ReplicationException {
        ReplicationUtils.removeAllQueues(this.queueStorage, peerId);
        ReplicationUtils.removeAllQueues(this.queueStorage, peerId);
        this.queueStorage.removePeerFromHFileRefs(peerId);
    }

    private void checkPeerConfig(ReplicationPeerConfig peerConfig) throws DoNotRetryIOException {
        String replicationEndpointImpl = peerConfig.getReplicationEndpointImpl();
        ReplicationEndpoint endpoint = null;
        if (!StringUtils.isBlank(replicationEndpointImpl)) {
            try {
                endpoint = Class.forName(replicationEndpointImpl).asSubclass(ReplicationEndpoint.class).getDeclaredConstructor(new Class[0]).newInstance(new Object[0]);
            }
            catch (Throwable e) {
                throw new DoNotRetryIOException("Can not instantiate configured replication endpoint class=" + replicationEndpointImpl, e);
            }
        }
        if (endpoint == null || endpoint instanceof HBaseReplicationEndpoint) {
            this.checkClusterKey(peerConfig.getClusterKey());
            if (endpoint == null || !endpoint.canReplicateToSameCluster()) {
                this.checkSameClusterKey(peerConfig.getClusterKey());
            }
        }
        if (peerConfig.replicateAllUserTables()) {
            if (peerConfig.getNamespaces() != null && !peerConfig.getNamespaces().isEmpty() || peerConfig.getTableCFsMap() != null && !peerConfig.getTableCFsMap().isEmpty()) {
                throw new DoNotRetryIOException("Need clean namespaces or table-cfs config firstly when you want replicate all cluster");
            }
            this.checkNamespacesAndTableCfsConfigConflict(peerConfig.getExcludeNamespaces(), peerConfig.getExcludeTableCFsMap());
        } else {
            if (peerConfig.getExcludeNamespaces() != null && !peerConfig.getExcludeNamespaces().isEmpty() || peerConfig.getExcludeTableCFsMap() != null && !peerConfig.getExcludeTableCFsMap().isEmpty()) {
                throw new DoNotRetryIOException("Need clean exclude-namespaces or exclude-table-cfs config firstly when replicate_all flag is false");
            }
            this.checkNamespacesAndTableCfsConfigConflict(peerConfig.getNamespaces(), peerConfig.getTableCFsMap());
        }
        this.checkConfiguredWALEntryFilters(peerConfig);
    }

    private void checkNamespacesAndTableCfsConfigConflict(Set<String> namespaces, Map<TableName, ? extends Collection<String>> tableCfs) throws DoNotRetryIOException {
        if (namespaces == null || namespaces.isEmpty()) {
            return;
        }
        if (tableCfs == null || tableCfs.isEmpty()) {
            return;
        }
        for (Map.Entry<TableName, ? extends Collection<String>> entry : tableCfs.entrySet()) {
            TableName table = entry.getKey();
            if (!namespaces.contains(table.getNamespaceAsString())) continue;
            throw new DoNotRetryIOException("Table-cfs " + table + " is conflict with namespaces " + table.getNamespaceAsString() + " in peer config");
        }
    }

    private void checkConfiguredWALEntryFilters(ReplicationPeerConfig peerConfig) throws DoNotRetryIOException {
        String filterCSV = peerConfig.getConfiguration().get("hbase.replication.source.custom.walentryfilters");
        if (filterCSV != null && !filterCSV.isEmpty()) {
            String[] filters;
            for (String filter : filters = filterCSV.split(",")) {
                try {
                    Class.forName(filter).getDeclaredConstructor(new Class[0]).newInstance(new Object[0]);
                }
                catch (Exception e) {
                    throw new DoNotRetryIOException("Configured WALEntryFilter " + filter + " could not be created. Failing add/update peer operation.", e);
                }
            }
        }
    }

    private void checkClusterKey(String clusterKey) throws DoNotRetryIOException {
        try {
            ZKConfig.validateClusterKey(clusterKey);
        }
        catch (IOException e) {
            throw new DoNotRetryIOException("Invalid cluster key: " + clusterKey, e);
        }
    }

    private void checkSameClusterKey(String clusterKey) throws DoNotRetryIOException {
        String peerClusterId = "";
        try {
            Configuration peerConf = HBaseConfiguration.createClusterConf(this.conf, clusterKey);
            try (ZKWatcher zkWatcher = new ZKWatcher(peerConf, this + "check-peer-cluster-id", null);){
                peerClusterId = ZKClusterId.readClusterIdZNode(zkWatcher);
            }
        }
        catch (IOException | KeeperException e) {
            throw new DoNotRetryIOException("Can't get peerClusterId for clusterKey=" + clusterKey, e);
        }
        if (this.clusterId.equals(peerClusterId)) {
            throw new DoNotRetryIOException("Invalid cluster key: " + clusterKey + ", should not replicate to itself for HBaseInterClusterReplicationEndpoint");
        }
    }

    public List<String> getSerialPeerIdsBelongsTo(TableName tableName) {
        return this.peers.values().stream().filter(p -> p.getPeerConfig().isSerial()).filter(p -> p.getPeerConfig().needToReplicate(tableName)).map(p -> p.getPeerId()).collect(Collectors.toList());
    }

    public ReplicationQueueStorage getQueueStorage() {
        return this.queueStorage;
    }

    public static ReplicationPeerManager create(ZKWatcher zk, Configuration conf, String clusterId) throws ReplicationException {
        ReplicationPeerStorage peerStorage = ReplicationStorageFactory.getReplicationPeerStorage(zk, conf);
        ConcurrentHashMap<String, ReplicationPeerDescription> peers = new ConcurrentHashMap<String, ReplicationPeerDescription>();
        for (String peerId : peerStorage.listPeerIds()) {
            ReplicationPeerConfig peerConfig = peerStorage.getPeerConfig(peerId);
            peerConfig = ReplicationPeerConfigUtil.updateReplicationBasePeerConfigs(conf, peerConfig);
            peerStorage.updatePeerConfig(peerId, peerConfig);
            boolean enabled = peerStorage.isPeerEnabled(peerId);
            peers.put(peerId, new ReplicationPeerDescription(peerId, enabled, peerConfig));
        }
        return new ReplicationPeerManager(peerStorage, ReplicationStorageFactory.getReplicationQueueStorage(zk, conf), peers, conf, clusterId);
    }

    private boolean isStringEquals(String s1, String s2) {
        if (StringUtils.isBlank(s1)) {
            return StringUtils.isBlank(s2);
        }
        return s1.equals(s2);
    }
}

