/*
 * Decompiled with CFR 0.152.
 */
package com.sleepycat.je.util;

import com.sleepycat.je.DatabaseException;
import com.sleepycat.je.DbInternal;
import com.sleepycat.je.Environment;
import com.sleepycat.je.EnvironmentConfig;
import com.sleepycat.je.JEVersion;
import com.sleepycat.je.cleaner.ExpirationProfile;
import com.sleepycat.je.cleaner.ExpirationTracker;
import com.sleepycat.je.cleaner.FileProcessor;
import com.sleepycat.je.cleaner.FileSummary;
import com.sleepycat.je.dbi.EnvironmentImpl;
import com.sleepycat.je.log.UtilizationFileReader;
import com.sleepycat.je.utilint.CmdUtil;
import com.sleepycat.je.utilint.DbLsn;
import com.sleepycat.je.utilint.Pair;
import com.sleepycat.utilint.FormatUtil;
import java.io.File;
import java.io.PrintStream;
import java.text.DateFormat;
import java.text.ParsePosition;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Date;
import java.util.Map;
import java.util.NavigableSet;
import java.util.SortedMap;

public class DbSpace {
    private static final String DATE_FORMAT = "yyyy-MM-dd'T'HHZ";
    private static final String DATE_EXAMPLE = "2016-03-17T22-0800";
    private static final String USAGE = "usage: " + CmdUtil.getJavaCommand(DbSpace.class) + "\n       -h <dir> # environment home directory\n       [-q]     # quiet, print grand totals only\n       [-u]     # sort by average utilization\n       [-d]     # dump file summary details\n       [-r]     # recalculate utilization (expensive)\n       [-R]     # recalculate expired data (expensive)\n       [-s]     # start file number or LSN, in hex\n       [-e]     # end file number or LSN, in hex\n       [-t]     # time for calculating expired data,\n                #   format: " + "yyyy-MM-dd'T'HHZ" + "\n                #  example: " + "2016-03-17T22-0800" + "\n       [-V]     # print JE version number";
    private File envHome = null;
    private EnvironmentImpl envImpl;
    private boolean quiet = false;
    private boolean sorted = false;
    private boolean details = false;
    private boolean doRecalcUtil = false;
    private boolean doRecalcExpired = false;
    private long startLsn = -1L;
    private long finishLsn = -1L;
    private long targetTime = System.currentTimeMillis();

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void main(String[] argv) throws Exception {
        DbSpace space = new DbSpace();
        space.parseArgs(argv);
        EnvironmentConfig envConfig = new EnvironmentConfig();
        envConfig.setReadOnly(true);
        Environment env = new Environment(space.envHome, envConfig);
        space.initEnv(DbInternal.getNonNullEnvImpl(env));
        try {
            space.print(System.out);
            System.exit(0);
        }
        catch (Throwable e) {
            e.printStackTrace(System.err);
            System.exit(1);
        }
        finally {
            try {
                env.close();
            }
            catch (Throwable e) {
                e.printStackTrace(System.err);
                System.exit(1);
            }
        }
    }

    private DbSpace() {
    }

    public DbSpace(Environment env, boolean quiet, boolean details, boolean sorted) {
        this(DbInternal.getNonNullEnvImpl(env), quiet, details, sorted);
    }

    public DbSpace(EnvironmentImpl envImpl, boolean quiet, boolean details, boolean sorted) {
        this.initEnv(envImpl);
        this.quiet = quiet;
        this.details = details;
        this.sorted = sorted;
    }

    private void initEnv(EnvironmentImpl envImpl) {
        this.envImpl = envImpl;
    }

    private void printUsage(String msg) {
        if (msg != null) {
            System.err.println(msg);
        }
        System.err.println(USAGE);
        System.exit(-1);
    }

    private void parseArgs(String[] argv) {
        int argc = 0;
        int nArgs = argv.length;
        if (nArgs == 0) {
            this.printUsage(null);
            System.exit(0);
        }
        while (argc < nArgs) {
            String thisArg;
            if ((thisArg = argv[argc++]).equals("-q")) {
                this.quiet = true;
                continue;
            }
            if (thisArg.equals("-u")) {
                this.sorted = true;
                continue;
            }
            if (thisArg.equals("-d")) {
                this.details = true;
                continue;
            }
            if (thisArg.equals("-r")) {
                this.doRecalcUtil = true;
                continue;
            }
            if (thisArg.equals("-R")) {
                this.doRecalcExpired = true;
                continue;
            }
            if (thisArg.equals("-V")) {
                System.out.println(JEVersion.CURRENT_VERSION);
                System.exit(0);
                continue;
            }
            if (thisArg.equals("-h")) {
                if (argc < nArgs) {
                    this.envHome = new File(argv[argc++]);
                    continue;
                }
                this.printUsage("-h requires an argument");
                continue;
            }
            if (thisArg.equals("-s")) {
                if (argc < nArgs) {
                    this.startLsn = CmdUtil.readLsn(argv[argc++]);
                    continue;
                }
                this.printUsage("-s requires an argument");
                continue;
            }
            if (thisArg.equals("-e")) {
                if (argc < nArgs) {
                    this.finishLsn = CmdUtil.readLsn(argv[argc++]);
                    continue;
                }
                this.printUsage("-e requires an argument");
                continue;
            }
            if (!thisArg.equals("-t")) continue;
            if (argc < nArgs) {
                String s;
                Date date;
                SimpleDateFormat format = new SimpleDateFormat(DATE_FORMAT);
                ParsePosition pp = new ParsePosition(0);
                if ((date = ((DateFormat)format).parse(s = argv[argc++], pp)) != null) {
                    this.targetTime = date.getTime();
                    continue;
                }
                this.printUsage("-t doesn't match format: yyyy-MM-dd'T'HHZ example: 2016-03-17T22-0800");
                continue;
            }
            this.printUsage("-t requires an argument");
        }
        if (this.envHome == null) {
            this.printUsage("-h is a required argument");
        }
        if (this.doRecalcUtil && this.doRecalcExpired) {
            this.printUsage("-r and -R cannot both be used");
        }
    }

    public void setRecalculate(boolean recalc) {
        this.doRecalcUtil = recalc;
    }

    public void setStartFile(long startFile) {
        this.startLsn = startFile;
    }

    public void setEndFile(long endFile) {
        this.finishLsn = endFile;
    }

    public void setTime(long time) {
        this.targetTime = time;
    }

    public void print(PrintStream out) throws DatabaseException {
        long startFile = this.startLsn != -1L ? DbLsn.getFileNumber(this.startLsn) : 0L;
        long finishFile = this.finishLsn != -1L ? DbLsn.getFileNumber(this.finishLsn) : Long.MAX_VALUE;
        SortedMap<Long, FileSummary> map = this.envImpl.getUtilizationProfile().getFileSummaryMap(true).subMap(startFile, finishFile);
        map.keySet().removeAll(this.envImpl.getCleaner().getFileSelector().getInProgressFiles());
        Map<Long, FileSummary> recalcMap = this.doRecalcUtil ? UtilizationFileReader.calcFileSummaryMap(this.envImpl, this.startLsn, this.finishLsn) : null;
        ExpirationProfile expProfile = new ExpirationProfile(this.envImpl.getExpirationProfile());
        expProfile.refresh(this.targetTime);
        int fileIndex = 0;
        Summary totals = new Summary();
        Summary[] summaries = null;
        if (!this.quiet) {
            summaries = new Summary[map.size()];
        }
        for (Map.Entry<Long, FileSummary> entry : map.entrySet()) {
            Long fileNum = entry.getKey();
            FileSummary fs = entry.getValue();
            FileSummary recalcFs = null;
            if (recalcMap != null) {
                recalcFs = recalcMap.get(fileNum);
            }
            int expiredSize = expProfile.getExpiredBytes(fileNum);
            int recalcExpiredSize = -1;
            ExpirationTracker expTracker = null;
            if (this.doRecalcExpired) {
                FileProcessor fileProcessor = this.envImpl.getCleaner().createProcessor();
                expTracker = fileProcessor.countExpiration(fileNum);
                recalcExpiredSize = expTracker.getExpiredBytes(this.targetTime);
            }
            Summary summary = new Summary(fileNum, fs, recalcFs, expiredSize, recalcExpiredSize);
            if (summaries != null) {
                summaries[fileIndex] = summary;
            }
            if (this.details) {
                out.println("File 0x" + Long.toHexString(fileNum) + " expired: " + expiredSize + " histogram: " + expProfile.toString(fileNum) + " " + fs);
                if (recalcMap != null) {
                    out.println("Recalc util 0x" + Long.toHexString(fileNum) + " " + recalcFs);
                }
                if (expTracker != null) {
                    out.println("Recalc expiration 0x" + Long.toHexString(fileNum) + " recalcExpired: " + recalcExpiredSize + " recalcHistogram: " + expTracker.toString());
                }
            }
            totals.add(summary);
            ++fileIndex;
        }
        if (this.details) {
            out.println();
        }
        out.println(this.doRecalcExpired ? "                      % Utilized    w/Expiration\n  File    Size (kB)  Avg  Min  Max  Recalculated\n--------  ---------  ---  ---  ---  ------------" : (this.doRecalcUtil ? "                      % Utilized    Recalculated\n  File    Size (kB)  Avg  Min  Max  Avg  Min  Max\n--------  ---------  ---  ---  ---  ---  ---  ---" : "                      % Utilized\n  File    Size (kB)  Avg  Min  Max  \n--------  ---------  ---- ---  ---"));
        if (summaries != null) {
            if (this.sorted) {
                Arrays.sort(summaries, new Comparator<Summary>(){

                    @Override
                    public int compare(Summary s1, Summary s2) {
                        return s1.avgUtilization() - s2.avgUtilization();
                    }
                });
            }
            for (Summary summary : summaries) {
                summary.print(out);
            }
        }
        totals.print(out);
        if (totals.expiredSize > 0L) {
            SimpleDateFormat format = new SimpleDateFormat(DATE_FORMAT);
            out.format("%nAs of %s, %,d kB are expired, resulting in the%ndifferences between minimum and maximum utilization.%n", format.format(this.targetTime), totals.expiredSize / 1024L);
        }
        Pair<Long, NavigableSet<Long>> reservedFileInfo = this.envImpl.getFileProtector().getReservedFileInfo();
        long reservedSize = reservedFileInfo.first();
        NavigableSet<Long> reservedFiles = reservedFileInfo.second();
        boolean printReservedFiles = !this.quiet && !reservedFiles.isEmpty();
        out.format("%n%,d kB are used by additional reserved files%s%n", reservedSize / 1024L, printReservedFiles ? ":" : ".");
        if (printReservedFiles) {
            out.println(FormatUtil.asHexString(reservedFiles));
        }
    }

    private class Summary {
        static final String HEADER = "                      % Utilized\n  File    Size (kB)  Avg  Min  Max  \n--------  ---------  ---- ---  ---";
        static final String RECALC_HEADER = "                      % Utilized    Recalculated\n  File    Size (kB)  Avg  Min  Max  Avg  Min  Max\n--------  ---------  ---  ---  ---  ---  ---  ---";
        static final String RECALC_EXPIRED_HEADER = "                      % Utilized    w/Expiration\n  File    Size (kB)  Avg  Min  Max  Recalculated\n--------  ---------  ---  ---  ---  ------------";
        Long fileNum;
        long totalSize;
        long obsoleteSize;
        long recalcObsoleteSize;
        long expiredSize;
        long recalcExpiredSize;

        Summary() {
        }

        Summary(Long fileNum, FileSummary summary, FileSummary recalcSummary, int expiredSize, int recalcExpiredSize) {
            this.fileNum = fileNum;
            this.totalSize = summary.totalSize;
            this.obsoleteSize = summary.getObsoleteSize();
            if (recalcSummary != null) {
                this.recalcObsoleteSize = recalcSummary.getObsoleteSize();
            }
            this.expiredSize = Math.min((long)expiredSize, this.totalSize);
            this.recalcExpiredSize = Math.min((long)recalcExpiredSize, this.totalSize);
        }

        void add(Summary o) {
            this.totalSize += o.totalSize;
            this.obsoleteSize += o.obsoleteSize;
            this.recalcObsoleteSize += o.recalcObsoleteSize;
            this.expiredSize += o.expiredSize;
            this.recalcExpiredSize += o.recalcExpiredSize;
        }

        void print(PrintStream out) {
            if (this.fileNum != null) {
                this.pad(out, Long.toHexString(this.fileNum), 8, '0');
            } else {
                out.print(" TOTALS ");
            }
            int kb = (int)(this.totalSize / 1024L);
            out.print("  ");
            this.pad(out, Integer.toString(kb), 9, ' ');
            out.print("  ");
            this.pad(out, Integer.toString(this.avgUtilization()), 3, ' ');
            out.print("  ");
            this.pad(out, Integer.toString(this.minUtilization()), 3, ' ');
            out.print("  ");
            this.pad(out, Integer.toString(this.maxUtilization()), 3, ' ');
            if (DbSpace.this.doRecalcExpired) {
                out.print("      ");
                this.pad(out, Integer.toString(this.expRecalcUtilization()), 3, ' ');
            } else if (DbSpace.this.doRecalcUtil) {
                out.print("  ");
                this.pad(out, Integer.toString(this.avgRecalcUtilization()), 3, ' ');
                out.print("  ");
                this.pad(out, Integer.toString(this.minRecalcUtilization()), 3, ' ');
                out.print("  ");
                this.pad(out, Integer.toString(this.maxRecalcUtilization()), 3, ' ');
            }
            out.println();
        }

        int avgUtilization() {
            return (this.minUtilization() + this.maxUtilization()) / 2;
        }

        int minUtilization() {
            return this.minUtilization(this.obsoleteSize, this.expiredSize);
        }

        int maxUtilization() {
            return this.maxUtilization(this.obsoleteSize, this.expiredSize);
        }

        int expRecalcUtilization() {
            return this.minUtilization(this.obsoleteSize, this.recalcExpiredSize);
        }

        int avgRecalcUtilization() {
            return (this.minRecalcUtilization() + this.maxRecalcUtilization()) / 2;
        }

        int minRecalcUtilization() {
            return this.minUtilization(this.recalcObsoleteSize, this.expiredSize);
        }

        int maxRecalcUtilization() {
            return this.maxUtilization(this.recalcObsoleteSize, this.expiredSize);
        }

        private int minUtilization(long obsolete, long expired) {
            return FileSummary.utilization(Math.min(obsolete + expired, this.totalSize), this.totalSize);
        }

        private int maxUtilization(long obsolete, long expired) {
            return FileSummary.utilization(Math.max(obsolete, expired), this.totalSize);
        }

        private void pad(PrintStream out, String val, int digits, char padChar) {
            int padSize = digits - val.length();
            for (int i = 0; i < padSize; ++i) {
                out.print(padChar);
            }
            out.print(val);
        }
    }
}

